package ru.yandex.travel.orders.workflows.orderitem.travelline.handlers;

import java.time.Instant;

import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.common.partners.base.CallContext;
import ru.yandex.travel.hotels.common.partners.travelline.TravellineClient;
import ru.yandex.travel.hotels.common.partners.travelline.exceptions.ReturnedErrorException;
import ru.yandex.travel.orders.entities.Payment;
import ru.yandex.travel.orders.entities.TravellineOrderItem;
import ru.yandex.travel.orders.proto.EPaymentType;
import ru.yandex.travel.orders.services.finances.FinancialEventService;
import ru.yandex.travel.orders.services.hotels.Meters;
import ru.yandex.travel.orders.services.partners.BillingPartnerService;
import ru.yandex.travel.orders.workflow.hotels.proto.TRefundStart;
import ru.yandex.travel.orders.workflow.hotels.travelline.proto.ETravellineItemState;
import ru.yandex.travel.orders.workflow.hotels.travelline.proto.TChangeAgreement;
import ru.yandex.travel.orders.workflow.hotels.travelline.proto.TRefundCommit;
import ru.yandex.travel.orders.workflow.order.proto.TMoneyAcquired;
import ru.yandex.travel.orders.workflow.payments.proto.EPaymentState;
import ru.yandex.travel.orders.workflows.orderitem.travelline.TravellineConfigurationProperties;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.exceptions.RetryableException;

@Slf4j
public class ConfirmedStateHandler extends BaseTravellineHandler {

    private final FinancialEventService financialEventService;
    private final BillingPartnerService billingPartnerService;
    private final TravellineConfigurationProperties properties;
    private final TravellineClient client;

    public ConfirmedStateHandler(Meters meters,
                                 FinancialEventService financialEventService,
                                 BillingPartnerService billingPartnerService,
                                 TravellineConfigurationProperties properties,
                                 TravellineClient client) {
        super(meters);
        this.financialEventService = financialEventService;
        this.billingPartnerService = billingPartnerService;
        this.properties = properties;
        this.client = client;
    }

    @HandleEvent
    public void scheduleRefund(TRefundStart message, StateContext<ETravellineItemState, TravellineOrderItem> context) {
        context.setState(ETravellineItemState.IS_REFUNDING);
        context.scheduleEvent(TRefundCommit.newBuilder()
                .setToken(message.getToken())
                .setForce(message.getForce())
                .setReason(message.getReason())
                .setSkipFinEvents(message.getSkipFinEvents())
                .setMoneyRefundMode(message.getMoneyRefundMode())
                .setRefundDescription(message.getRefundDescription())
                .build());
        log.info("scheduled refund");
    }

    @HandleEvent
    public void handleMoneyAcquired(TMoneyAcquired event, StateContext<ETravellineItemState, TravellineOrderItem> ctx) {
        Payment payment = ctx.getWorkflowEntity().getOrder().getPayments().stream()
                .filter(pi -> pi.getId().toString().equals(event.getPaymentId()))
                .findFirst().orElseThrow(() -> new IllegalStateException("No invoice found"));
        if (ctx.getWorkflowEntity().getOrder().getCreatedAt().isBefore(properties.getEnableMakeExtraPaymentSince())) {
            log.info("Extra/final payment acquired, but order is too old, skipping");
            return;
        }
        Money extraAmount = null;
        if (payment.getPaymentType() == EPaymentType.PT_PAYMENT_SCHEDULE && payment.getState() == EPaymentState.PS_FULLY_PAID
                && ctx.getWorkflowEntity().getOrder().getPaymentSchedule() != null) {
            Money initiallyPaid =
                    ctx.getWorkflowEntity().getOrder().getPaymentSchedule().getInitialPendingInvoice().getTotalAmount();
            Money totalAmount = ctx.getWorkflowEntity().getHotelItinerary().getFiscalPrice();
            extraAmount = totalAmount.subtract(initiallyPaid);
            log.info("Notifying partner on payment schedule final payment, amount is {}", extraAmount);
        } else if (payment.getPaymentType() == EPaymentType.PT_PENDING_INVOICE && payment.getState() == EPaymentState.PS_FULLY_PAID) {
            extraAmount = payment.getPaidAmount();
            log.info("Notifying partner on extra payment, amount is {}", extraAmount);
        }
        if (extraAmount != null) {
            Money finalExtraAmount = extraAmount;
            try {
                wrap(() -> client
                        .withCallContext(getCallContext(ctx.getWorkflowEntity(), CallContext.CallPhase.ORDER_CONFIRMATION))
                        .makeExtraPaymentSync(ctx.getWorkflowEntity().getItinerary().getYandexNumber(), event.getPaymentId(),
                                finalExtraAmount), false);
            } catch (ReturnedErrorException ex) {
                if (ex.getErrors().stream().anyMatch(e -> e.getMessage().startsWith("Concurrent booking modification is forbidden"))) {
                    log.warn("Concurrent modification error", ex);
                    throw new RetryableException("Concurrent modification error on extra payment", ex, properties.getConcurrentModificationRetryIn());
                }
            }
        } else {
            log.warn("No extra amount present");
        }
    }

    @HandleEvent
    public void handleTChangeAgreement(TChangeAgreement message, StateContext<ETravellineItemState,
            TravellineOrderItem> context) {
        TravellineOrderItem orderItem = context.getWorkflowEntity();
        log.info("Moving order item from client {} to client {}",
                orderItem.getAgreement().getFinancialClientId(),
                message.getClientId());
        Instant payoutAt = null;
        if (message.hasPayoutDate()) {
            payoutAt = ProtoUtils.toInstant(message.getPayoutDate());
        }
        // TODO(mbobrov): think of better naming and a better place for this call
        financialEventService.registerServiceFullCorrection(orderItem, payoutAt);

        var oldClientId = orderItem.getAgreement().getFinancialClientId();
        var oldContractId = orderItem.getAgreement().getFinancialContractId();

        // Need to update agreement for correct events in registerConfirmedService
        billingPartnerService.checkNewClientForAgreement(orderItem, message.getClientId(), message.getContractId());
        orderItem.getAgreement().setFinancialClientId(message.getClientId());
        orderItem.getAgreement().setFinancialContractId(message.getContractId());

        financialEventService.registerConfirmedService(orderItem, payoutAt);

        // Setting clientId and contractId back as we are afraid to change these values in order
        orderItem.getAgreement().setFinancialClientId(oldClientId);
        orderItem.getAgreement().setFinancialContractId(oldContractId);
    }
}
