package ru.yandex.travel.orders.workflows.orderitem.travelline.handlers;

import java.time.Instant;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.travel.hotels.common.orders.CancellationDetails;
import ru.yandex.travel.hotels.common.orders.ConfirmationInfo;
import ru.yandex.travel.hotels.common.orders.TravellineHotelItinerary;
import ru.yandex.travel.hotels.common.partners.base.CallContext;
import ru.yandex.travel.hotels.common.partners.travelline.TravellineClient;
import ru.yandex.travel.hotels.common.partners.travelline.exceptions.ReturnedErrorException;
import ru.yandex.travel.hotels.common.partners.travelline.model.BookingStatus;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.orders.entities.TravellineOrderItem;
import ru.yandex.travel.orders.services.hotels.Meters;
import ru.yandex.travel.orders.workflow.hotels.travelline.proto.ETravellineItemState;
import ru.yandex.travel.orders.workflow.hotels.travelline.proto.TConfirmationCommit;
import ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled;
import ru.yandex.travel.orders.workflow.order.proto.TServiceConfirmed;
import ru.yandex.travel.orders.workflows.orderitem.travelline.TravellineConfigurationProperties;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.exceptions.RetryableException;

@Slf4j
public class ConfirmingStateHandler extends BaseTravellineHandler {
    private final TravellineClient client;
    private final TravellineConfigurationProperties properties;

    public ConfirmingStateHandler(TravellineClient client, Meters meters, TravellineConfigurationProperties properties) {
        super(meters);
        this.client = client;
        this.properties = properties;
    }

    private Money getPaymentAmount(TravellineOrderItem item) {
        if (item.getOrder().getPaymentSchedule() != null) {
            return item.getOrder().getPaymentSchedule().getInitialPendingInvoice().getTotalAmount();
        } else {
            return item.getItinerary().getRealHotelPrice();
        }
    }

    @HandleEvent
    public void handleConfirmationCommit(TConfirmationCommit message,
                                         StateContext<ETravellineItemState, TravellineOrderItem> context) {
        TravellineHotelItinerary itinerary = context.getWorkflowEntity().getItinerary();
        Preconditions.checkArgument(itinerary.getServiceId().equals(context.getWorkflowEntity().getId().toString()));
        Preconditions.checkNotNull(itinerary.getYandexNumber());
        Preconditions.checkNotNull(itinerary.getTravellineNumber());
        var readReservationResponse = wrap(() -> client
                .withCallContext(getCallContext(context.getWorkflowEntity(), CallContext.CallPhase.ORDER_CONFIRMATION))
                .readReservationSync(itinerary.getYandexNumber()), true);
        Preconditions.checkNotNull(readReservationResponse, "Reservation not found");
        Preconditions.checkArgument(readReservationResponse.getHotelReservations().size() == 1,
                "Unexpected number of hotel reservations");
        var reservation = readReservationResponse.getHotelReservations().get(0);
        log.info("existing reservation found with reservation code #{} and status {}",
                reservation.getReservationNumber(), reservation.getStatus());
        switch (reservation.getStatus()) {
            case PENDING:
                // TODO(tivelkov): this one may be unrelated - think a better way of passing it if it is really needed
                var paymentId = context.getWorkflowEntity().getOrder().getId().toString();
                log.info("calling Travelline ConfirmReservation API");
                try {
                    var res = wrap(() -> client
                                    .withCallContext(getCallContext(context.getWorkflowEntity(),
                                            CallContext.CallPhase.ORDER_CONFIRMATION))
                                    .confirmReservationSync(itinerary.getYandexNumber(), paymentId, getPaymentAmount(context.getWorkflowEntity())),
                            true);
                    Preconditions.checkArgument(res.getHotelReservations().size() == 1,
                            "Unexpected number of hotel reservations");
                    reservation = res.getHotelReservations().get(0);
                } catch (ReturnedErrorException ex) {
                    if (ex.getErrors().stream().anyMatch(e -> e.getMessage().startsWith("Concurrent booking modification is forbidden"))) {
                        log.warn("Concurrent modification error", ex);
                        throw new RetryableException("Concurrent modification error on confirmation", ex, properties.getConcurrentModificationRetryIn());
                    }
                }
                break;
            case CONFIRMED:
                log.info("reservation is already confirmed. Suspecting duplicate confirmation call");
                break;
            case CANCELLED:
                log.info("reservation is CANCELLED, probably expired");
                itinerary.setOrderCancellationDetails(CancellationDetails.create(CancellationDetails.Reason.HOLD_EXPIRED));
                context.setState(ETravellineItemState.IS_CANCELLED);
                meters.incrementCancellationCounter(EPartnerId.PI_TRAVELLINE, CancellationDetails.Reason.HOLD_EXPIRED);
                context.scheduleExternalEvent(context.getWorkflowEntity().getOrderWorkflowId(),
                        TServiceCancelled.newBuilder().setServiceId(context.getWorkflowEntity().getId().toString()).build());
                return;
            default:
                throw new RuntimeException(String.format("Unexpected status %s", reservation.getStatus()));
        }
        Preconditions.checkArgument(reservation.getStatus() == BookingStatus.CONFIRMED, "Unexpected booking status");
        itinerary.setConfirmation(new ConfirmationInfo());
        itinerary.getConfirmation().setPartnerConfirmationId(itinerary.getTravellineNumber());
        context.setState(ETravellineItemState.IS_CONFIRMED);
        context.getWorkflowEntity().setConfirmedAt(Instant.now());
        context.scheduleExternalEvent(context.getWorkflowEntity().getOrderWorkflowId(),
                TServiceConfirmed.newBuilder().setServiceId(context.getWorkflowEntity().getId().toString()).build());
        meters.incrementConfirmationCounter(EPartnerId.PI_TRAVELLINE);
        log.info("confirmation info received, itinerary is CONFIRMED");
    }
}
