package ru.yandex.travel.orders.workflows.orderitem.travelline.handlers;

import java.math.BigDecimal;
import java.util.Set;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.common.orders.BaseRate;
import ru.yandex.travel.hotels.common.orders.TravellineHotelItinerary;
import ru.yandex.travel.hotels.common.partners.base.CallContext;
import ru.yandex.travel.hotels.common.partners.travelline.TravellineClient;
import ru.yandex.travel.hotels.common.partners.travelline.exceptions.ReturnedErrorException;
import ru.yandex.travel.hotels.common.partners.travelline.model.BookingStatus;
import ru.yandex.travel.hotels.common.partners.travelline.model.CancelReservationResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.CancellationInfo;
import ru.yandex.travel.hotels.common.partners.travelline.model.ErrorType;
import ru.yandex.travel.hotels.common.partners.travelline.model.ReadReservationResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.RespHotelReservation;
import ru.yandex.travel.orders.entities.TravellineOrderItem;
import ru.yandex.travel.orders.proto.THotelRefundToken;
import ru.yandex.travel.orders.services.hotels.Meters;
import ru.yandex.travel.orders.workflow.hotels.travelline.proto.ETravellineItemState;
import ru.yandex.travel.orders.workflow.hotels.travelline.proto.TRefundCommit;
import ru.yandex.travel.orders.workflows.orderitem.RefundingUtils;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.HandleEvent;

@Slf4j
public class RefundingStateHandler extends BaseTravellineHandler {
    private final TravellineClient client;

    public RefundingStateHandler(TravellineClient client, Meters meters) {
        super(meters);
        this.client = client;
    }

    @HandleEvent
    public void handleRefundCommit(TRefundCommit message,
                                   StateContext<ETravellineItemState, TravellineOrderItem> context) {
        TravellineHotelItinerary itinerary = context.getWorkflowEntity().getItinerary();
        Preconditions.checkArgument(itinerary.getServiceId().equals(context.getWorkflowEntity().getId().toString()));
        Preconditions.checkNotNull(itinerary.getYandexNumber());
        log.debug("Calling Travelline cancel reservation API");
        ReadReservationResponse reservationResponse =
                wrap(() -> client
                        .withCallContext(getCallContext(context.getWorkflowEntity(),
                                CallContext.CallPhase.ORDER_REFUND))
                        .readReservationSync(itinerary.getYandexNumber()), true);
        Preconditions.checkState(reservationResponse.getHotelReservations().size() == 1,
                "Unexpected number of hotel reservations");
        RespHotelReservation hotelReservation = reservationResponse.getHotelReservations().get(0);
        Preconditions.checkState(Set.of(BookingStatus.CONFIRMED, BookingStatus.CANCELLED).contains(hotelReservation.getStatus()),
                "Unexpected booking status: " + hotelReservation.getStatus());
        THotelRefundToken refundToken = RefundingUtils.getRefundToken(message.getToken());
        if (hotelReservation.getStatus() == BookingStatus.CONFIRMED) {
            try {
                CancelReservationResponse cancelReservationResponse =
                        wrap(() -> client
                                .withCallContext(getCallContext(context.getWorkflowEntity(),
                                        CallContext.CallPhase.ORDER_REFUND))
                                .cancelReservationSync(itinerary.getYandexNumber()), true);
                Preconditions.checkState(cancelReservationResponse.getHotelReservations().size() == 1,
                        "Unexpected number of hotel reservations");
                hotelReservation = cancelReservationResponse.getHotelReservations().get(0);
            } catch (ReturnedErrorException ex) {
                if (ex.getErrors().size() == 1 && ex.getErrors().get(0).getErrorCode() == ErrorType.BOOKING_CANNOT_BE_CANCELLED && message.getForce()) {
                    log.warn("Travelline reports that booking cannot be cancelled, but the FORCE flag is present so " +
                            "the error is ignored and the service is considered refunded");
                    itinerary.setRefundInfo(RefundingUtils.createRefundInfo(refundToken, message.getReason()));
                    context.setState(ETravellineItemState.IS_REFUNDED);
                    RefundingUtils.scheduleServiceRefundedEvent(context, refundToken,
                            message.getSkipFinEvents(), message.getMoneyRefundMode(), message.getRefundDescription());
                    return;
                }
            }
        }
        Preconditions.checkState(hotelReservation.getStatus() == BookingStatus.CANCELLED,
                "Unexpected booking status: " + hotelReservation.getStatus());

        CancellationInfo actualCancellation = hotelReservation.getCancellationInfo();
        BigDecimal actualPenaltyAmount;
        if (actualCancellation != null && actualCancellation.getPenalty() != null
                && actualCancellation.getPenalty().getPriceBeforeTax() != null) {
            actualPenaltyAmount = BigDecimal.valueOf(actualCancellation.getPenalty().getPriceBeforeTax());
        } else {
            actualPenaltyAmount = BigDecimal.ZERO;
        }
        Money actualMoney = Money.of(
                actualPenaltyAmount,
                context.getWorkflowEntity().getOrder().getCurrency());
        BaseRate actualPrice = BaseRate.fromTPrice(ProtoUtils.toTPrice(actualMoney));
        BaseRate expectedPrice = BaseRate.fromTPrice(refundToken.getPenaltyAmount());
        if (!expectedPrice.equals(actualPrice)) {
            String errorMsg = String.format(
                    "Actual and expected penalty amounts diverged. Actual: %s; expected: %s",
                    actualPrice.getAmount(),
                    expectedPrice.getAmount());
            if (message.getForce()) {
                log.warn(errorMsg);
            } else {
                log.warn(errorMsg);
                // TODO(tivelkov): TRAVELBACK-2960 uncomment when TL fixes their bug
                // throw new RuntimeException(errorMsg);
            }
        }
        itinerary.setRefundInfo(RefundingUtils.createRefundInfo(refundToken, message.getReason()));
        context.setState(ETravellineItemState.IS_REFUNDED);
        RefundingUtils.scheduleServiceRefundedEvent(context, refundToken,
                message.getSkipFinEvents(), message.getMoneyRefundMode(), message.getRefundDescription());
        log.info("REFUNDED");

    }
}
