package ru.yandex.travel.orders.workflows.orderitem.travelline.handlers;

import lombok.extern.slf4j.Slf4j;
import org.apache.logging.log4j.util.Strings;

import ru.yandex.travel.hotels.common.orders.CancellationDetails;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.orders.entities.TravellineOrderItem;
import ru.yandex.travel.orders.services.hotels.Meters;
import ru.yandex.travel.orders.workflow.hotels.proto.TCancellationStart;
import ru.yandex.travel.orders.workflow.hotels.proto.TConfirmationStart;
import ru.yandex.travel.orders.workflow.hotels.travelline.proto.ETravellineItemState;
import ru.yandex.travel.orders.workflow.hotels.travelline.proto.TCancellationCommit;
import ru.yandex.travel.orders.workflow.hotels.travelline.proto.TConfirmationCommit;
import ru.yandex.travel.orders.workflow.hotels.travelline.proto.TReservationExpired;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.HandleEvent;

@Slf4j
public class ReservedStateHandler extends BaseTravellineHandler {
    public ReservedStateHandler(Meters meters) {
        super(meters);
    }

    @HandleEvent
    public void startCancellation(TCancellationStart message,
                                  StateContext<ETravellineItemState, TravellineOrderItem> context) {
        context.setState(ETravellineItemState.IS_CANCELLING);
        context.scheduleEvent(TCancellationCommit.newBuilder().build());
        log.info("scheduled cancellation");
        if (Strings.isNotEmpty(message.getReason())) {
            var itinerary = context.getWorkflowEntity().getItinerary();
            itinerary.setOrderCancellationDetails(CancellationDetails.create(message.getReason()));
            meters.incrementCancellationCounter(EPartnerId.PI_TRAVELLINE, message.getReason());
        }
    }

    @HandleEvent
    public void startGenericCancellation(ru.yandex.travel.orders.workflow.orderitem.generic.proto.TCancellationStart message,
                                         StateContext<ETravellineItemState, TravellineOrderItem> context) {
        startCancellation(TCancellationStart.newBuilder().build(), context);
    }

    @HandleEvent
    public void handleReservationExpired(TReservationExpired message,
                                         StateContext<ETravellineItemState, TravellineOrderItem> context) {
        log.info("Reservation expired. Scheduled cancellation");
        var itinerary = context.getWorkflowEntity().getItinerary();
        var reason = CancellationDetails.Reason.MONEY_ACQUIRE_ERROR;
        itinerary.setOrderCancellationDetails(CancellationDetails.create(reason));
        meters.incrementCancellationCounter(EPartnerId.PI_TRAVELLINE, reason);
        context.setState(ETravellineItemState.IS_CANCELLING);
        context.scheduleEvent(TCancellationCommit.newBuilder().build());
    }

    @HandleEvent
    public void startConfirmation(TConfirmationStart message,
                                  StateContext<ETravellineItemState, TravellineOrderItem> context) {
        startGenericConfirmation(null, context);
    }

    @HandleEvent
    public void startGenericConfirmation(ru.yandex.travel.orders.workflow.orderitem.generic.proto.TConfirmationStart message,
                                         StateContext<ETravellineItemState, TravellineOrderItem> context) {
        context.setState(ETravellineItemState.IS_CONFIRMING);
        context.scheduleEvent(TConfirmationCommit.newBuilder().build());
        log.info("scheduled confirmation");
    }
}
