package ru.yandex.travel.orders.workflows.orderitem.travelline.handlers;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Instant;
import java.util.Objects;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.hotels.common.orders.CancellationDetails;
import ru.yandex.travel.hotels.common.orders.MealData;
import ru.yandex.travel.hotels.common.orders.TravellineHotelItinerary;
import ru.yandex.travel.hotels.common.partners.base.CallContext;
import ru.yandex.travel.hotels.common.partners.travelline.TravellineClient;
import ru.yandex.travel.hotels.common.partners.travelline.exceptions.ReturnedErrorException;
import ru.yandex.travel.hotels.common.partners.travelline.model.BookingStatus;
import ru.yandex.travel.hotels.common.partners.travelline.model.ErrorType;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelReservationRequest;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelReservationResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.RatePlan;
import ru.yandex.travel.hotels.common.partners.travelline.model.RespHotelReservation;
import ru.yandex.travel.hotels.common.partners.travelline.model.ServiceKind;
import ru.yandex.travel.hotels.common.partners.travelline.model.VatTaxType;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.orders.commons.proto.EVat;
import ru.yandex.travel.orders.entities.TravellineOrderItem;
import ru.yandex.travel.orders.services.hotels.Meters;
import ru.yandex.travel.orders.workflow.hotels.travelline.proto.ETravellineItemState;
import ru.yandex.travel.orders.workflow.hotels.travelline.proto.TCancellationCommit;
import ru.yandex.travel.orders.workflow.hotels.travelline.proto.TReservationCommit;
import ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled;
import ru.yandex.travel.orders.workflow.order.proto.TServiceReserved;
import ru.yandex.travel.orders.workflows.orderitem.travelline.TravellineConfigurationProperties;
import ru.yandex.travel.orders.workflows.orderitem.travelline.TravellineFiscalItemCreator;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.HandleEvent;

@Slf4j
public class ReservingStateHandler extends BaseTravellineHandler {
    private final TravellineClient client;
    private final TravellineConfigurationProperties properties;
    private final TravellineFiscalItemCreator fiscalItemCreator;

    public ReservingStateHandler(TravellineClient client, TravellineConfigurationProperties properties, Meters meters,
                                 TravellineFiscalItemCreator fiscalItemCreator) {
        super(meters);
        this.client = client;
        this.properties = properties;
        this.fiscalItemCreator = fiscalItemCreator;
    }


    @HandleEvent
    public void handleReservationCommit(TReservationCommit message,
                                        StateContext<ETravellineItemState, TravellineOrderItem> context) {
        TravellineHotelItinerary itinerary = context.getWorkflowEntity().getItinerary();
        itinerary.setServiceId(context.getWorkflowEntity().getId().toString());
        // TODO(tivelkov): check whether we need to specify and check expiresAtInstant here
        log.info("checking itinerary for existence");
        RespHotelReservation reservation;
        var readReservationResponse = wrap(() -> client
                .withCallContext(getCallContext(context.getWorkflowEntity(), CallContext.CallPhase.ORDER_RESERVATION))
                .readReservationSync(itinerary.getYandexNumber()), true);
        if (readReservationResponse != null) {
            Preconditions.checkArgument(readReservationResponse.getHotelReservations().size() == 1,
                    "Unexpected number of hotel reservations");
            reservation = readReservationResponse.getHotelReservations().get(0);
            log.info("existing reservation found with reservation code #{}",
                    reservation.getReservationNumber());
        } else {
            log.info("calling Travelline HotelReservation API");
            var request = HotelReservationRequest.create(
                    itinerary.getOffer(),
                    itinerary.getSelectedPlacement(),
                    itinerary.getYandexNumber(),
                    itinerary.getGuests(),
                    itinerary.getCustomerPhone(),
                    itinerary.getCustomerEmail());

            HotelReservationResponse createReservationResponse;
            try {
                createReservationResponse = wrap(() -> client
                        .withCallContext(getCallContext(context.getWorkflowEntity(),
                                CallContext.CallPhase.ORDER_RESERVATION))
                        .createReservationSync(request), true);
            } catch (ReturnedErrorException err) {
                var soldOutErr = err.getErrorOfType(ErrorType.SOLD_OUT, ErrorType.ARRIVAL_DATE_IS_NOT_AVAILABLE);
                if (soldOutErr != null) {
                    log.error("Sold out: {}", soldOutErr);
                    itinerary.setOrderCancellationDetails(CancellationDetails.create(CancellationDetails.Reason.SOLD_OUT));
                    meters.incrementCancellationCounter(EPartnerId.PI_TRAVELLINE, CancellationDetails.Reason.SOLD_OUT);
                    context.setState(ETravellineItemState.IS_CANCELLED);
                    context.scheduleExternalEvent(context.getWorkflowEntity().getOrderWorkflowId(),
                            TServiceCancelled.newBuilder().setServiceId(context.getWorkflowEntity().getId().toString()).build());
                    return;
                } else {
                    throw err;
                }
            }
            Preconditions.checkArgument(createReservationResponse.getHotelReservations().size() == 1,
                    "Unexpected number of hotel reservations");
            reservation = createReservationResponse.getHotelReservations().get(0);
            log.info("new reservation created with reservation code#{}",
                    reservation.getReservationNumber());
        }
        Preconditions.checkArgument(reservation.getStatus() == BookingStatus.PENDING,
                String.format("Unexpected reservation status '%s'", reservation.getStatus()));
        var totalReserved = BigDecimal.valueOf(reservation.getTotal().getPriceBeforeTax()).setScale(0,
                RoundingMode.HALF_UP);
        var totalExpected = itinerary.getFiscalPrice().getNumberStripped().setScale(0, RoundingMode.HALF_UP);
        itinerary.setTravellineNumber(reservation.getReservationNumber());
        context.getWorkflowEntity().setProviderId(itinerary.getTravellineNumber());
        if (totalExpected.compareTo(totalReserved) != 0) {
            log.error("Price mismatch after reservation");
            itinerary.setOrderCancellationDetails(CancellationDetails.create(CancellationDetails.Reason.PRICE_CHANGED));
            context.setState(ETravellineItemState.IS_CANCELLING);
            meters.incrementCancellationCounter(EPartnerId.PI_TRAVELLINE, CancellationDetails.Reason.PRICE_CHANGED);
            context.scheduleEvent(TCancellationCommit.newBuilder().build());
        } else {
            log.info("RESERVED");
            context.setState(ETravellineItemState.IS_RESERVED);
            itinerary.setActualPrice(Money.of(BigDecimal.valueOf(reservation.getTotal().getPriceBeforeTax()).setScale(2,
                    RoundingMode.HALF_UP), itinerary.getFiscalPrice().getCurrency()));
            VatTaxType taxOverride = reservation.getRoomStays().stream()
                    .flatMap(rs -> rs.getRatePlans().stream())
                    .map(RatePlan::getVatTaxType)
                    .filter(Objects::nonNull)
                    .distinct()
                    .collect(Collectors.collectingAndThen(
                            Collectors.toList(),
                            l -> l.size() == 1? l.get(0) : null));
            itinerary.setVatTaxOverride(taxOverride);
            itinerary.setMealData(generateMealData(reservation,
                    context.getWorkflowEntity().getAgreement().getVatType().getProtoValue()));
            Instant expiresAt =
                    Instant.now().plus(properties.getReservationDuration());
            itinerary.setExpiresAtInstant(expiresAt);
            context.getWorkflowEntity().setExpiresAt(expiresAt);
            fiscalItemCreator.addFiscalItems(context.getWorkflowEntity());
            context.scheduleExternalEvent(context.getWorkflowEntity().getOrderWorkflowId(),
                    TServiceReserved.newBuilder().setServiceId(context.getWorkflowEntity().getId().toString()).build());
        }
    }

    private MealData generateMealData(RespHotelReservation reservation, EVat defaultVat) {
        if (!properties.getBilling().isGenerateMealItems()) {
            return null;
        }
        return MealData.builder().items(
                reservation.getRoomStays().stream()
                        .flatMap(rs -> rs.getServices().stream())
                        .filter(s -> s.isInclusive() && s.getKind() == ServiceKind.MEAL && s.getPrice() != null && s.getPrice().getPriceBeforeTax() > 0)
                        .map(s -> MealData.MealItem.builder()
                                .mealName(s.getName())
                                .mealPrice(Money.of(s.getPrice().getPriceBeforeTax(), ProtoCurrencyUnit.RUB))
                                .mealVat(s.getVatTaxType() != null ? s.getVatTaxType().toProto() : defaultVat)
                                .build())
                        .collect(Collectors.toList())
        ).build();
    }
}
