package ru.yandex.travel.orders.workflows.payments.invoice.handlers;

import java.time.Instant;
import java.util.List;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.google.protobuf.Message;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.PendingInvoice;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.workflow.order.proto.TClearingInProcess;
import ru.yandex.travel.orders.workflow.order.proto.TInvoiceCleared;
import ru.yandex.travel.orders.workflow.order.proto.TInvoicePaymentStarted;
import ru.yandex.travel.orders.workflow.order.proto.TMoneyAcquired;
import ru.yandex.travel.orders.workflow.payments.proto.EPaymentState;
import ru.yandex.travel.orders.workflows.invoice.trust.InvoiceUtils;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Slf4j
public class BasePendingInvoiceHandler extends AnnotatedStatefulWorkflowEventHandler<EPaymentState, PendingInvoice> {
    protected static void fullyRefundPaidAttempt(StateContext<EPaymentState, PendingInvoice> ctx) {
        @SuppressWarnings("SuspiciousMethodCalls")
        List<Invoice> paidInvoices = ctx.getWorkflowEntity().getPaymentAttempts().stream()
                .filter(pa -> !TrustInvoice.UNPAID_INVOICE_STATES.contains(pa.getInvoiceState()))
                .collect(Collectors.toList());
        Preconditions.checkState(paidInvoices.size() == 1, "Unexpected amount of paid invoices");
        ctx.scheduleExternalEvent(paidInvoices.get(0).getWorkflow().getId(),
                InvoiceUtils.buildFullRefund(paidInvoices.get(0)));
    }

    @HandleEvent
    public void handle(TInvoicePaymentStarted message, StateContext<EPaymentState, PendingInvoice> ctx) {
        propagateMessageToOwner(message, ctx);
    }

    @HandleEvent
    public void handle(TClearingInProcess message, StateContext<EPaymentState, PendingInvoice> ctx) {
        propagateMessageToOwner(message, ctx);
    }

    @HandleEvent
    public void handle(TInvoiceCleared message, StateContext<EPaymentState, PendingInvoice> ctx) {
        propagateMessageToOwner(message, ctx);
    }

    protected void propagateMessageToOwner(Message message, StateContext<EPaymentState, PendingInvoice> ctx) {
        log.info("Forwarding the event of class {} to owner workflow", message.getClass().getSimpleName());
        sendMessageToOwner(message, ctx);
    }

    protected void sendMessageToOwner(Message message, StateContext<EPaymentState, PendingInvoice> ctx) {
        ctx.scheduleExternalEvent(ctx.getWorkflowEntity().getOwnerOrder().getWorkflow().getId(), message);
    }

    protected void finishPayment(StateContext<EPaymentState, PendingInvoice> ctx) {
        ctx.setState(EPaymentState.PS_FULLY_PAID);
        ctx.getWorkflowEntity().setClosedAt(Instant.now());
        ctx.scheduleExternalEvent(ctx.getWorkflowEntity().getOwnerWorkflowId(),
                TMoneyAcquired.newBuilder().setPaymentId(ctx.getWorkflowEntity().getId().toString())
                        .build());
    }
}
