package ru.yandex.travel.orders.workflows.payments.invoice.handlers;

import java.time.Instant;

import com.google.protobuf.Message;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.PendingInvoice;
import ru.yandex.travel.orders.services.payments.InvoiceFactory;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.order.proto.TInvoicePaymentStarted;
import ru.yandex.travel.orders.workflow.order.proto.TPaymentCancelled;
import ru.yandex.travel.orders.workflow.payments.proto.EPaymentState;
import ru.yandex.travel.orders.workflow.payments.proto.TCancelPayment;
import ru.yandex.travel.orders.workflow.payments.proto.TStartPayment;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.HandleEvent;

@Slf4j
@RequiredArgsConstructor
public class InvoicePendingStateHandler extends BasePendingInvoiceHandler {

    private final InvoiceFactory invoiceFactory;


    @HandleEvent
    public void handleStartPayment(TStartPayment message, StateContext<EPaymentState, PendingInvoice> ctx) {
        if (ctx.getWorkflowEntity().getTotalAmount().isZero()) {
            log.info("Attempt to start payment of zero invoice, immediately returning as success");
            ctx.getWorkflowEntity().getOwnerOrder().setUserActionScheduled(false);
            finishPayment(ctx);

            return;
        }

        if (ctx.getWorkflowEntity().getPaymentAttempts().stream()
                .anyMatch(i -> i.getInvoiceState() != ETrustInvoiceState.IS_PAYMENT_NOT_AUTHORIZED)) {
            throw new IllegalStateException("Unable to create a payment for a payment invoice which already has an " +
                    "active payment");
        }
        Message paymentTestContext = null;
        switch (message.getPaymentTestContextsCase()) {
            case AVIAPAYMENTTESTCONTEXT:
                paymentTestContext = message.getAviaPaymentTestContext();
                break;
            case PAYMENTTESTCONTEXT:
                paymentTestContext = message.getPaymentTestContext();
                break;
        }
        log.info("Will create a new payment attempt for a PendingInvoice");
        Invoice invoice = invoiceFactory.createInvoice(
                ctx.getWorkflowEntity().getOwnerOrder(),
                InvoiceFactory.invoiceTypeFromPaymentType(message.getPaymentProvider()),
                ctx.getWorkflowEntity(), message.getSource(),
                message.getReturnUrl(), message.getConfirmationReturnUrl(), message.getPaymentMethodId(),
                paymentTestContext,
                message.getNewCommonPaymentWebForm());
        invoice.setOrderWorkflowId(ctx.getWorkflowId());
        ctx.scheduleExternalEvent(ctx.getWorkflowEntity().getOwnerWorkflowId(),
                TInvoicePaymentStarted.newBuilder().build());
        ctx.getWorkflowEntity().getOwnerOrder().setUserActionScheduled(false);
        ctx.setState(EPaymentState.PS_PAYMENT_IN_PROGRESS);
    }

    @HandleEvent
    public void handleCancelPayment(TCancelPayment message, StateContext<EPaymentState, PendingInvoice> ctx) {
        ctx.setState(EPaymentState.PS_CANCELLED);
        ctx.getWorkflowEntity().setClosedAt(Instant.now());
        ctx.scheduleExternalEvent(ctx.getWorkflowEntity().getOwnerWorkflowId(), TPaymentCancelled.newBuilder().build());
    }

}
