package ru.yandex.travel.orders.workflows.payments.jobs;

import java.util.Collection;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.support.DefaultTransactionDefinition;

import ru.yandex.travel.spring.tx.ForcedRollbackTxManagerWrapper;
import ru.yandex.travel.task_processor.AbstractTaskKeyProvider;
import ru.yandex.travel.task_processor.TaskKeyProvider;
import ru.yandex.travel.task_processor.TaskProcessor;
import ru.yandex.travel.task_processor.TaskProcessorProperties;

@Configuration
@EnableConfigurationProperties(PaymentJobsConfigurationProperties.class)
@RequiredArgsConstructor
public class PaymentJobsConfiguration {
    private final PaymentTasksService paymentTasksService;
    private final PaymentJobsConfigurationProperties properties;
    private final ForcedRollbackTxManagerWrapper forcedRollbackTxManagerWrapper;

    @Bean
    @ConditionalOnProperty("payment-schedule.jobs.refund-expired-schedules.enabled")
    public TaskProcessor<UUID> expiredPaymentsRefundInitiatingProcessor() {
        TaskKeyProvider<UUID> taskKeyProvider = new AbstractTaskKeyProvider<>() {
            @Override
            public Collection<UUID> getPendingTaskKeys(int maxResultSize) {
                return paymentTasksService.fetchExpiringScheduleIds(getLockedTaskKeys(), maxResultSize);
            }

            @Override
            public long getPendingTasksCount() {
                return paymentTasksService.countExpiringScheduleIds(getLockedTaskKeys());
            }
        };
        DefaultTransactionDefinition transactionDefinition = new DefaultTransactionDefinition();
        transactionDefinition.setName("DeferredScheduleRefundExpired");
        transactionDefinition.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        transactionDefinition.setIsolationLevel(TransactionDefinition.ISOLATION_REPEATABLE_READ);
        TransactionDefinition txDefinition = new DefaultTransactionDefinition(transactionDefinition);

        return new TaskProcessor<>(taskKeyProvider, paymentTasksService::startExpiredRefund,
                forcedRollbackTxManagerWrapper, txDefinition,
                TaskProcessorProperties.builder()
                        .name("DeferredScheduleRefundExpired")
                        .daemonPoolThreads(true)
                        .gracefulPoolShutdown(true)
                        .poolSize(properties.getRefundExpiredSchedules().getNumberOfItems())
                        .initialStartDelay(properties.getRefundExpiredSchedules().getInitialStartDelay())
                        .scheduleRate(properties.getRefundExpiredSchedules().getPeriod())
                        .build());
    }

    @Bean
    @ConditionalOnProperty("payment-schedule.jobs.notify-expiring-schedules-by-email.enabled")
    public TaskProcessor<UUID> expiringPaymentsEmailReminderProcessor() {
        TaskKeyProvider<UUID> taskKeyProvider = new AbstractTaskKeyProvider<>() {
            @Override
            public Collection<UUID> getPendingTaskKeys(int maxResultSize) {
                return paymentTasksService.fetchScheduleItemIdsToNotifyByEmail(getLockedTaskKeys(), maxResultSize);
            }

            @Override
            public long getPendingTasksCount() {
                return paymentTasksService.countScheduleItemIdsToNotifyByEmail(getLockedTaskKeys());
            }
        };
        DefaultTransactionDefinition transactionDefinition = new DefaultTransactionDefinition();
        transactionDefinition.setName("DeferredScheduleEmailNotify");
        transactionDefinition.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        transactionDefinition.setIsolationLevel(TransactionDefinition.ISOLATION_REPEATABLE_READ);
        TransactionDefinition txDefinition = new DefaultTransactionDefinition(transactionDefinition);

        return new TaskProcessor<>(taskKeyProvider, paymentTasksService::notifyPaymentRequiredByEmail,
                forcedRollbackTxManagerWrapper, txDefinition,
                TaskProcessorProperties.builder()
                        .name("DeferredScheduleEmailNotify")
                        .daemonPoolThreads(true)
                        .gracefulPoolShutdown(true)
                        .poolSize(properties.getNotifyExpiringSchedulesByEmail().getNumberOfItems())
                        .initialStartDelay(properties.getNotifyExpiringSchedulesByEmail().getInitialStartDelay())
                        .scheduleRate(properties.getNotifyExpiringSchedulesByEmail().getPeriod())
                        .build());
    }

    @Bean
    @ConditionalOnProperty("payment-schedule.jobs.notify-expiring-schedules-by-ticket.enabled")
    public TaskProcessor<UUID> expiringPaymentsTicketReminderProcessor() {
        TaskKeyProvider<UUID> taskKeyProvider = new AbstractTaskKeyProvider<>() {
            @Override
            public Collection<UUID> getPendingTaskKeys(int maxResultSize) {
                return paymentTasksService.fetchScheduleItemIdsToNotifyByTicket(getLockedTaskKeys(), maxResultSize);
            }

            @Override
            public long getPendingTasksCount() {
                return paymentTasksService.countScheduleItemIdsToNotifyByTicket(getLockedTaskKeys());
            }
        };
        DefaultTransactionDefinition transactionDefinition = new DefaultTransactionDefinition();
        transactionDefinition.setName("DeferredScheduleTicketNotify");
        transactionDefinition.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        transactionDefinition.setIsolationLevel(TransactionDefinition.ISOLATION_REPEATABLE_READ);
        TransactionDefinition txDefinition = new DefaultTransactionDefinition(transactionDefinition);

        return new TaskProcessor<>(taskKeyProvider, paymentTasksService::notifyPaymentRequiredByTicket,
                forcedRollbackTxManagerWrapper, txDefinition,
                TaskProcessorProperties.builder()
                        .name("DeferredScheduleTicketNotify")
                        .daemonPoolThreads(true)
                        .gracefulPoolShutdown(true)
                        .poolSize(properties.getNotifyExpiringSchedulesByTicket().getNumberOfItems())
                        .initialStartDelay(properties.getNotifyExpiringSchedulesByTicket().getInitialStartDelay())
                        .scheduleRate(properties.getNotifyExpiringSchedulesByTicket().getPeriod())
                        .build());
    }

    @Bean
    @ConditionalOnProperty("payment-schedule.jobs.auto-pay-schedules.enabled")
    public TaskProcessor<UUID> autoPaymentProcessor() {
        TaskKeyProvider<UUID> taskKeyProvider = new AbstractTaskKeyProvider<>() {
            @Override
            public Collection<UUID> getPendingTaskKeys(int maxResultSize) {
                return paymentTasksService.fetchScheduleItemsToAutoStartPayment(getLockedTaskKeys(), maxResultSize);
            }

            @Override
            public long getPendingTasksCount() {
                return paymentTasksService.countScheduleItemsToAutoStartPayment(getLockedTaskKeys());
            }
        };
        DefaultTransactionDefinition transactionDefinition = new DefaultTransactionDefinition();
        transactionDefinition.setName("DeferredScheduleAutoPayment");
        transactionDefinition.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        transactionDefinition.setIsolationLevel(TransactionDefinition.ISOLATION_REPEATABLE_READ);
        TransactionDefinition txDefinition = new DefaultTransactionDefinition(transactionDefinition);

        return new TaskProcessor<>(taskKeyProvider, paymentTasksService::startAutoPayment,
                forcedRollbackTxManagerWrapper, txDefinition,
                TaskProcessorProperties.builder()
                        .name("DeferredScheduleAutoPayment")
                        .daemonPoolThreads(true)
                        .gracefulPoolShutdown(true)
                        .poolSize(properties.getAutoPaySchedules().getNumberOfItems())
                        .initialStartDelay(properties.getAutoPaySchedules().getInitialStartDelay())
                        .scheduleRate(properties.getAutoPaySchedules().getPeriod())
                        .build());
    }
}
