package ru.yandex.travel.orders.workflows.payments.jobs;

import java.time.Instant;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.PaymentSchedule;
import ru.yandex.travel.orders.entities.PaymentScheduleItem;
import ru.yandex.travel.orders.entities.Ticket;
import ru.yandex.travel.orders.management.StarTrekService;
import ru.yandex.travel.orders.repository.PaymentScheduleItemRepository;
import ru.yandex.travel.orders.repository.PaymentScheduleRepository;
import ru.yandex.travel.orders.repository.TicketRepository;
import ru.yandex.travel.orders.services.NotificationHelper;
import ru.yandex.travel.orders.workflow.notification.proto.TSend;
import ru.yandex.travel.orders.workflow.payments.proto.EPaymentState;
import ru.yandex.travel.orders.workflow.payments.schedule.proto.TPaymentScheduleExpired;
import ru.yandex.travel.orders.workflow.payments.schedule.proto.TStartAutoPayment;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.WorkflowMessageSender;

@Service
@Slf4j
@RequiredArgsConstructor
public class PaymentTasksService {
    private static final UUID ZERO_UUID = UUID.fromString("00000000-0000-0000-0000-000000000000");
    private final PaymentScheduleRepository scheduleRepository;
    private final WorkflowMessageSender workflowMessageSender;
    private final PaymentScheduleItemRepository scheduleItemRepository;
    private final NotificationHelper notificationHelper;
    private final TicketRepository ticketRepository;
    private final StarTrekService starTrekService;

    @TransactionMandatory
    public List<UUID> fetchExpiringScheduleIds(Set<UUID> active, int maxResultSize) {
        active = ensureSetIsNotEmpty(active);
        return scheduleRepository.findNonExpiredSchedulesInGivenStatesExpiringBeforeInstant(
                Set.of(EPaymentState.PS_PARTIALLY_PAID),
                Set.of(EPaymentState.PS_INVOICE_PENDING),
                Instant.now(),
                active,
                PageRequest.of(0, maxResultSize))
                .stream()
                .map(PaymentSchedule::getId)
                .collect(Collectors.toList());
    }

    @TransactionMandatory
    public List<UUID> fetchScheduleItemIdsToNotifyByEmail(Set<UUID> active, int maxResultSize) {
        active = ensureSetIsNotEmpty(active);
        return scheduleItemRepository.findItemsForEmailReminderWithNoNotifications(
                Instant.now(),
                active,
                PageRequest.of(0, maxResultSize))
                .stream()
                .map(PaymentScheduleItem::getId)
                .collect(Collectors.toList());
    }

    @TransactionMandatory
    public long countScheduleItemIdsToNotifyByEmail(Set<UUID> active) {
        active = ensureSetIsNotEmpty(active);
        return scheduleItemRepository.countItemsForEmailReminderWithNoNotifications(Instant.now(), active);
    }

    @TransactionMandatory
    public List<UUID> fetchScheduleItemIdsToNotifyByTicket(Set<UUID> active, int maxResultSize) {
        active = ensureSetIsNotEmpty(active);
        return scheduleItemRepository.findItemsForTicketReminderWithNoNotifications(
                Instant.now(),
                active,
                PageRequest.of(0, maxResultSize))
                .stream()
                .map(PaymentScheduleItem::getId)
                .collect(Collectors.toList());
    }

    @TransactionMandatory
    public long countScheduleItemIdsToNotifyByTicket(Set<UUID> active) {
        active = ensureSetIsNotEmpty(active);
        return scheduleItemRepository.countItemsForTicketReminderWithNoNotifications(Instant.now(), active);
    }

    @TransactionMandatory
    public long countExpiringScheduleIds(Set<UUID> active) {
        active = ensureSetIsNotEmpty(active);
        return scheduleRepository.countNonExpiredSchedulesInGivenStatesExpiringBeforeInstant(
                Set.of(EPaymentState.PS_PARTIALLY_PAID),
                Set.of(EPaymentState.PS_INVOICE_PENDING),
                Instant.now(),
                active);
    }

    @TransactionMandatory
    public List<UUID> fetchScheduleItemsToAutoStartPayment(Set<UUID> active, int maxResultSize) {
        active = ensureSetIsNotEmpty(active);
        return scheduleItemRepository
                .findItemsForAutoPayment(Instant.now(), active, PageRequest.of(0, maxResultSize))
                .stream()
                .map(PaymentScheduleItem::getId)
                .collect(Collectors.toList());
    }

    @TransactionMandatory
    public long countScheduleItemsToAutoStartPayment(Set<UUID> active) {
        active = ensureSetIsNotEmpty(active);
        return scheduleItemRepository.countItemsForAutoPayment(Instant.now(), active);
    }

    @TransactionMandatory
    public void startExpiredRefund(UUID scheduleId) {
        PaymentSchedule schedule = scheduleRepository.getOne(scheduleId);
        try (var ignored = NestedMdc.forEntity(schedule.getOrder())) {
            log.info("Notifying schedule {} of its expiration", scheduleId);
            schedule.setExpired(true);
            workflowMessageSender.scheduleEvent(schedule.getWorkflow().getId(),
                    TPaymentScheduleExpired.newBuilder().build());
        }
    }

    @TransactionMandatory
    public void notifyPaymentRequiredByEmail(UUID scheduleItemId) {
        PaymentScheduleItem item = scheduleItemRepository.getOne(scheduleItemId);
        try (var ignored = NestedMdc.forEntity(item.getSchedule().getOrder())) {
            log.info("Initiating reminder email workflow");
            UUID emailWorkflowId =
                    notificationHelper.createWorkflowForHotelOrderRequiresPayment((HotelOrder) item.getSchedule().getOrder());
            workflowMessageSender.scheduleEvent(emailWorkflowId, TSend.newBuilder().build());
            item.setReminderEmailSent(true);
        }
    }

    @TransactionMandatory
    public void notifyPaymentRequiredByTicket(UUID scheduleItemId) {
        PaymentScheduleItem item = scheduleItemRepository.getOne(scheduleItemId);
        try (var ignored = NestedMdc.forEntity(item.getSchedule().getOrder())) {
            log.info("Creating tracker ticket");
            UUID ticketUid = starTrekService.createIssueToRemindAboutExpiringDeferredOrder(item, workflowMessageSender);
            Ticket ticket = ticketRepository.getOne(ticketUid);
            item.setReminderTicket(ticket);
        }
    }

    @TransactionMandatory
    public void startAutoPayment(UUID scheduleItemId) {
        PaymentScheduleItem item = scheduleItemRepository.getOne(scheduleItemId);
        try (var ignored = NestedMdc.forEntity(item.getSchedule().getOrder())) {
            log.info("Starting automatic payment");
            workflowMessageSender.scheduleEvent(item.getSchedule().getWorkflow().getId(),
                    TStartAutoPayment.newBuilder()
                            .setItemId(item.getId().toString())
                            .build());
            item.setAutoPaymentStarted(true);
        }
    }

    private Set<UUID> ensureSetIsNotEmpty(Set<UUID> uuidSet) {
        Set<UUID> notEmptySet = uuidSet;
        if (notEmptySet == null || notEmptySet.size() == 0) {
            notEmptySet = new HashSet<>();
            notEmptySet.add(ZERO_UUID);
        }
        return notEmptySet;
    }

}
