package ru.yandex.travel.orders.workflows.payments.schedule.handlers;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;

import lombok.extern.slf4j.Slf4j;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.travel.orders.entities.PaymentSchedule;
import ru.yandex.travel.orders.entities.PaymentScheduleItem;
import ru.yandex.travel.orders.entities.PendingInvoice;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.orders.workflow.order.proto.TCommentIssue;
import ru.yandex.travel.orders.workflow.order.proto.TInvoiceRefunded;
import ru.yandex.travel.orders.workflow.order.proto.TMoneyAcquireErrorOccurred;
import ru.yandex.travel.orders.workflow.order.proto.TMoneyAcquired;
import ru.yandex.travel.orders.workflow.order.proto.TPaymentCanNotBeCancelled;
import ru.yandex.travel.orders.workflow.order.proto.TPaymentCancelled;
import ru.yandex.travel.orders.workflow.payments.proto.EPaymentState;
import ru.yandex.travel.orders.workflow.payments.proto.TCancelPayment;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;

@IgnoreEvents(types = {TMoneyAcquireErrorOccurred.class, TInvoiceRefunded.class})
@Slf4j
public class CancellationInProgressStateHandler extends BasePaymentScheduleHandler {

    @HandleEvent
    public void handleMoneyAcquired(TMoneyAcquired event, StateContext<EPaymentState, PaymentSchedule> ctx) {
        if (event.getPaymentId().equals(ctx.getWorkflowEntity().getInitialPendingInvoice().getId().toString())) {
            log.info("Acquired money for initial payment while in cancellation in progress. " +
                    "Will cancel and refund the payment.");
            ctx.scheduleExternalEvent(ctx.getWorkflowEntity().getInitialPendingInvoice().getWorkflow().getId(),
                    TCancelPayment.newBuilder().setRefundIfAlreadyPaid(true).build());
        }
        Map<String, PendingInvoice> deferredInvoiceMap = ctx.getWorkflowEntity().getItems().stream()
                .map(PaymentScheduleItem::getPendingInvoice)
                .collect(Collectors.toMap(i -> i.getId().toString(), Function.identity()));
        if (deferredInvoiceMap.containsKey(event.getPaymentId())) {
            log.info("Acquired money for deferred payment {} while in cancellation in progress. " +
                            "Will cancel and refund the payment.",
                    event.getPaymentId());
            ctx.scheduleExternalEvent(deferredInvoiceMap.get(event.getPaymentId()).getWorkflow().getId(),
                    TCancelPayment.newBuilder().setRefundIfAlreadyPaid(true).build());
        }
    }

    @HandleEvent
    public void handlePaymentCancelled(TPaymentCancelled event, StateContext<EPaymentState, PaymentSchedule> ctx) {
        checkCancellationComplete(ctx);
    }

    @HandleEvent
    public void handlePaymentCanNotBeCancelled(TPaymentCanNotBeCancelled event,
                                               StateContext<EPaymentState, PaymentSchedule> ctx) {
        if (ctx.getWorkflowEntity().getAllowPaidItemsWhenCancelling()) {
            checkCancellationComplete(ctx);
        } else {
            throw new IllegalStateException("TPaymentCanNotBeCancelled encountered on cancellation " +
                    "when paid items should not be left");
        }
    }

    private void checkCancellationComplete(StateContext<EPaymentState, PaymentSchedule> ctx) {
        Set<EPaymentState> closedStates;
        if (ctx.getWorkflowEntity().getAllowPaidItemsWhenCancelling()) {
            closedStates = Set.of(EPaymentState.PS_FULLY_PAID, EPaymentState.PS_CANCELLED);
        } else {
            closedStates = Set.of(EPaymentState.PS_CANCELLED);
        }

        List<Tuple2<UUID, EPaymentState>> nonClosedIds = ctx.getWorkflowEntity().getAllInvoices().stream()
                .filter(pendingInvoice -> !closedStates.contains(pendingInvoice.getState()))
                .map(pendingInvoice -> Tuple2.tuple(pendingInvoice.getId(), pendingInvoice.getState()))
                .collect(Collectors.toList());

        if (nonClosedIds.size() == 0) {
            ctx.getWorkflowEntity().getItems().stream()
                    .map(PaymentScheduleItem::getReminderTicket)
                    .filter(Objects::nonNull)
                    .forEach(t -> {
                        if (ctx.getWorkflowEntity().getOrder().getEntityState() == EHotelOrderState.OS_CONFIRMED) {
                            ctx.scheduleExternalEvent(t.getWorkflow().getId(),
                                    TCommentIssue.newBuilder()
                                            .setText("Заказ отменился автоматически из-за неоплаты. " +
                                                    "Вероятно, стоит об этом сообщить пользователю")
                                            .build());
                        } else {
                            ctx.scheduleExternalEvent(t.getWorkflow().getId(),
                                    TCommentIssue.newBuilder()
                                            .setText("Заказ был отменен пользователем")
                                            .setCloseIssue(true)
                                            .build());
                        }
                    });
            log.info("All deferred payments are closed, cancelling the schedule");
            ctx.setState(EPaymentState.PS_CANCELLED);
            ctx.scheduleExternalEvent(ctx.getWorkflowEntity().getOwnerWorkflowId(),
                    TPaymentCancelled.newBuilder().build());
        } else {
            log.info("{} deferred payments are still non-cancelled: {}", nonClosedIds.size(), nonClosedIds);
        }
    }

}
