package ru.yandex.travel.orders.workflows.payments.schedule.handlers;

import java.time.Instant;
import java.time.ZoneId;
import java.util.Comparator;
import java.util.Optional;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.PaymentSchedule;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.services.payments.TrustClientProvider;
import ru.yandex.travel.orders.services.payments.TrustUserInfo;
import ru.yandex.travel.orders.services.payments.model.TrustBoundPaymentMethod;
import ru.yandex.travel.orders.services.payments.model.TrustBoundPaymentMethodType;
import ru.yandex.travel.orders.services.payments.model.TrustPaymentMethodsResponse;
import ru.yandex.travel.orders.services.payments.schedule.HotelPaymentScheduleBuilderProperties;
import ru.yandex.travel.orders.workflow.order.proto.TInvoicePaymentStarted;
import ru.yandex.travel.orders.workflow.order.proto.TMoneyAcquireErrorOccurred;
import ru.yandex.travel.orders.workflow.order.proto.TMoneyAcquired;
import ru.yandex.travel.orders.workflow.payments.proto.EPaymentState;
import ru.yandex.travel.orders.workflow.payments.proto.TCancelPayment;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;

@Slf4j
@IgnoreEvents(types = TInvoicePaymentStarted.class)
@RequiredArgsConstructor
public class InitialPaymentInProgressStateHandler extends BasePaymentScheduleHandler {

    private final TrustClientProvider trustClientProvider;
    private final HotelPaymentScheduleBuilderProperties properties;

    @HandleEvent
    public void handleMoneyAcquire(TMoneyAcquired message,
                                   StateContext<EPaymentState, PaymentSchedule> ctx) {
        if (message.getPaymentId().equals(ctx.getWorkflowEntity().getInitialPendingInvoice().getId().toString())) {
            Preconditions.checkState(ctx.getWorkflowEntity().getInitialPendingInvoice().getState() == EPaymentState.PS_FULLY_PAID,
                    "Initial invoice is not paid");
        }
        log.info("Initial money acquired");
        Invoice lastAttempt = ctx.getWorkflowEntity().getInitialPendingInvoice().getLastAttempt();
        if (lastAttempt instanceof TrustInvoice) {
            checkAndSetAutoPayment(ctx.getWorkflowEntity(), (TrustInvoice) lastAttempt);
        }

        ctx.setState(EPaymentState.PS_PARTIALLY_PAID);
        ctx.scheduleExternalEvent(ctx.getWorkflowEntity().getOwnerWorkflowId(),
                TMoneyAcquired.newBuilder().setPaymentId(ctx.getWorkflowEntity().getId().toString()).build());
    }

    @HandleEvent
    public void handleMoneyAcquireError(TMoneyAcquireErrorOccurred message, StateContext<EPaymentState,
            PaymentSchedule> context) {
        log.info("Initial invoice payment was cancelled due to an error, returning back to invoice pending state");
        context.setState(EPaymentState.PS_INVOICE_PENDING);
    }

    @HandleEvent
    public void handleCancelPayment(TCancelPayment message, StateContext<EPaymentState, PaymentSchedule> ctx) {
        ctx.setState(EPaymentState.PS_CANCELLATION_IN_PROGRESS);
        ctx.getWorkflowEntity().setAllowPaidItemsWhenCancelling(!message.getRefundIfAlreadyPaid());
        ctx.scheduleExternalEvent(ctx.getWorkflowEntity().getInitialPendingInvoice().getWorkflow().getId(), message);
        ctx.getWorkflowEntity().getItems().forEach(i ->
                ctx.scheduleExternalEvent(i.getPendingInvoice().getWorkflow().getId(), message));
    }


    private void checkAndSetAutoPayment(PaymentSchedule schedule, TrustInvoice initialPayment) {
        if (schedule.getOrder().getExperiments().isAutomaticDeferredCharge()) {
            if (Strings.isNullOrEmpty(initialPayment.getPassportId())) {
                log.info("User is not authorized, no automatic payment possible");
                return;
            }
            if (initialPayment.getActualPaymentMethodId().startsWith("card-")) {
                log.info("Initial payment is done with a bound card, will try to initiate automatic payment");
                TrustClient client =
                        trustClientProvider.getTrustClientForPaymentProfile(initialPayment.getPaymentProfile());
                TrustUserInfo userInfo = new TrustUserInfo(
                        initialPayment.getPassportId(),
                        initialPayment.getUserIp()
                );
                TrustPaymentMethodsResponse response = client.getPaymentMethods(userInfo);
                Optional<TrustBoundPaymentMethod> methodOptional = response.getBoundPaymentMethods().stream()
                        .filter(pm -> pm.getPaymentMethod() == TrustBoundPaymentMethodType.CARD &&
                                pm.getId().equals(initialPayment.getActualPaymentMethodId()))
                        .findFirst();
                if (methodOptional.isPresent()) {
                    TrustBoundPaymentMethod method = methodOptional.get();
                    Instant lastPaymentInstant = schedule.getItems().stream()
                            .map(i -> i.getPaymentEndsAt().minus(properties.getAutoPaymentInterval()))
                            .max(Comparator.comparing(Instant::toEpochMilli))
                            .orElseThrow();
                    int paymentYear = lastPaymentInstant.atZone(ZoneId.systemDefault()).toLocalDate().getYear();
                    int paymentMonth = lastPaymentInstant.atZone(ZoneId.systemDefault()).toLocalDate().getMonthValue();
                    if (method.getExpirationYear() > paymentYear || (method.getExpirationYear() == paymentYear && method.getExpirationMonth() >= paymentMonth)) {
                        log.info("Will use payment method {} as auto-payment for this schedule", method.getId());
                        schedule.getItems().forEach(i -> {
                            i.setEmailReminderAt(null);
                            i.setAutoPaymentAt(i.getPaymentEndsAt().minus(properties.getAutoPaymentInterval()));
                            i.setBoundPaymentMethod(method);
                            i.setAutoPaymentStarted(false);
                        });
                    } else {
                        log.info("The bound card expires before the payment should occur, automatic payment is not " +
                                "possible");
                    }
                } else {
                    log.info("Payment method is not found for the user");
                }

            } else {
                log.info("Initial payment is done with an unbound card, no automatic payment possible");
            }
        }
    }
}
