package ru.yandex.travel.orders.workflows.payments.schedule.handlers;

import java.util.Comparator;
import java.util.Objects;
import java.util.UUID;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.entities.PaymentSchedule;
import ru.yandex.travel.orders.entities.PaymentScheduleItem;
import ru.yandex.travel.orders.services.finances.FinancialEventService;
import ru.yandex.travel.orders.services.payments.PaymentProfile;
import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.services.payments.TrustClientProvider;
import ru.yandex.travel.orders.services.payments.TrustUserInfo;
import ru.yandex.travel.orders.services.payments.model.TrustBoundPaymentMethodType;
import ru.yandex.travel.orders.services.payments.model.TrustPaymentMethodsResponse;
import ru.yandex.travel.orders.services.payments.schedule.HotelPaymentScheduleBuilderProperties;
import ru.yandex.travel.orders.workflow.order.proto.TCommentIssue;
import ru.yandex.travel.orders.workflow.order.proto.TInvoicePaymentStarted;
import ru.yandex.travel.orders.workflow.order.proto.TMoneyAcquireErrorOccurred;
import ru.yandex.travel.orders.workflow.order.proto.TMoneyAcquired;
import ru.yandex.travel.orders.workflow.payments.proto.EPaymentProvider;
import ru.yandex.travel.orders.workflow.payments.proto.EPaymentState;
import ru.yandex.travel.orders.workflow.payments.proto.TCancelPayment;
import ru.yandex.travel.orders.workflow.payments.proto.TConfirmPayment;
import ru.yandex.travel.orders.workflow.payments.proto.TPublish;
import ru.yandex.travel.orders.workflow.payments.proto.TStartPayment;
import ru.yandex.travel.orders.workflow.payments.schedule.proto.TPaymentScheduleExpired;
import ru.yandex.travel.orders.workflow.payments.schedule.proto.TStartAutoPayment;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;

@Slf4j
@IgnoreEvents(types = {TInvoicePaymentStarted.class})
@RequiredArgsConstructor
public class PartiallyPaidStateHandler extends BasePaymentScheduleHandler {
    private final TrustClientProvider trustClientProvider;
    private final HotelPaymentScheduleBuilderProperties properties;
    private final FinancialEventService financialEventService;

    @HandleEvent
    public void handleConfirm(TConfirmPayment message, StateContext<EPaymentState, PaymentSchedule> ctx) {
        log.info("Order services are confirmed, scheduling deferred payments");
        ctx.getWorkflowEntity().getItems().forEach(i -> ctx.scheduleExternalEvent(i.getPendingInvoice().getWorkflow().getId(), TPublish.newBuilder().build()));
    }

    @HandleEvent
    public void handleStartPayment(TStartPayment message, StateContext<EPaymentState, PaymentSchedule> ctx) {
        Preconditions.checkState(ctx.getWorkflowEntity().getInitialPendingInvoice().getState() == EPaymentState.PS_FULLY_PAID,
                "Unexpected state of initial invoice");
        Preconditions.checkState(ctx.getWorkflowEntity().getItems().stream()
                        .noneMatch(i -> i.getPendingInvoice().getState() == EPaymentState.PS_PAYMENT_IN_PROGRESS),
                "Unable to start payment when some of other payments are in progress");

        log.info("StartPayment arrived to a confirmed schedule, will find a deferred payment to start");
        UUID pendingInvoiceWorkflowId = ctx.getWorkflowEntity().getItems().stream()
                .filter(i -> i.getPendingInvoice().getState() == EPaymentState.PS_INVOICE_PENDING)
                .min(Comparator.comparing(PaymentScheduleItem::getPaymentEndsAt))
                .orElseThrow(() -> new IllegalStateException("No pending invoice found"))
                .getPendingInvoice().getWorkflow().getId();
        ctx.scheduleExternalEvent(pendingInvoiceWorkflowId, message);
    }

    @HandleEvent
    public void handleMoneyAcquired(TMoneyAcquired message, StateContext<EPaymentState, PaymentSchedule> ctx) {
        log.info("Deferred payment for invoice {} is done", message.getPaymentId());
        if (ctx.getWorkflowEntity().getAllInvoices().stream().allMatch(i -> i.getState() == EPaymentState.PS_FULLY_PAID)) {
            ctx.getWorkflowEntity().getItems().stream()
                    .filter(i -> i.getPendingInvoice().getId().toString().equals(message.getPaymentId()))
                    .map(PaymentScheduleItem::getReminderTicket)
                    .filter(Objects::nonNull)
                    .forEach(t -> ctx.scheduleExternalEvent(t.getWorkflow().getId(),
                            TCommentIssue.newBuilder()
                                    .setText("Счет оплачен, все хорошо :)")
                                    .setCloseIssue(true)
                                    .build()));
            log.info("All deferred payments are fully paid, marking schedule as fully paid as well");
            ctx.setState(EPaymentState.PS_FULLY_PAID);
            ctx.scheduleExternalEvent(ctx.getWorkflowEntity().getOwnerWorkflowId(),
                    TMoneyAcquired.newBuilder().setPaymentId(ctx.getWorkflowEntity().getId().toString()).build());
            ctx.getWorkflowEntity().getOrder().getOrderItems().forEach(oi -> financialEventService.registerFullyPaidPaymentSchedule(oi, ctx.getWorkflowEntity()));
        }
    }

    @HandleEvent
    public void handleMoneyAcquireErrorOccurred(TMoneyAcquireErrorOccurred message, StateContext<EPaymentState,
            PaymentSchedule> ctx) {
        log.info("Deferred payment has failed");
        ctx.getWorkflowEntity().getItems().stream()
                .filter(i -> i.getAutoPaymentAt() != null && i.getEmailReminderAt() == null)
                .forEach(i -> i.setEmailReminderAt(i.getAutoPaymentAt()));
    }

    @HandleEvent
    public void handleCancelPayment(TCancelPayment message, StateContext<EPaymentState, PaymentSchedule> ctx) {
        ctx.setState(EPaymentState.PS_CANCELLATION_IN_PROGRESS);
        ctx.getWorkflowEntity().setAllowPaidItemsWhenCancelling(!message.getRefundIfAlreadyPaid());
        ctx.scheduleExternalEvent(ctx.getWorkflowEntity().getInitialPendingInvoice().getWorkflow().getId(), message);
        ctx.getWorkflowEntity().getItems().forEach(i ->
                ctx.scheduleExternalEvent(i.getPendingInvoice().getWorkflow().getId(), message));
    }

    @HandleEvent
    public void handlePaymentScheduleExpired(TPaymentScheduleExpired message, StateContext<EPaymentState,
            PaymentSchedule> ctx) {
        ctx.setState(EPaymentState.PS_CANCELLATION_IN_PROGRESS);
        ctx.getWorkflowEntity().setAllowPaidItemsWhenCancelling(true);
        ctx.getWorkflowEntity().getItems().forEach(i ->
                ctx.scheduleExternalEvent(i.getPendingInvoice().getWorkflow().getId(),
                        TCancelPayment.newBuilder().build()));
    }

    @HandleEvent
    public void handleStartAutoPayment(TStartAutoPayment message, StateContext<EPaymentState, PaymentSchedule> ctx) {
        PaymentScheduleItem item = ctx.getWorkflowEntity().getItems()
                .stream()
                .filter(i -> i.getId().toString().equals(message.getItemId()) &&
                        i.getPendingInvoice().getState() == EPaymentState.PS_INVOICE_PENDING)
                .findFirst()
                .orElseThrow(() -> new IllegalStateException(String.format("No schedule item with pending invoice and" +
                        " id %s found", message.getItemId())));
        if (item.getBoundPaymentMethod() == null) {
            log.warn("No bound payment method for schedule item " + message.getItemId());
            return;
        }
        TrustClient client =
                trustClientProvider.getTrustClientForPaymentProfile(PaymentProfile.HOTEL);
        TrustUserInfo userInfo = new TrustUserInfo(item.getBoundPaymentMethod().getOrigUid(), null);
        TrustPaymentMethodsResponse response = client.getPaymentMethods(userInfo);
        if (response.getBoundPaymentMethods().stream()
                .noneMatch(pm -> pm.getPaymentMethod() == TrustBoundPaymentMethodType.CARD &&
                        pm.getId().equals(item.getBoundPaymentMethod().getId()))) {
            log.warn("Bound payment method is no longer available, automatic payment is not possible");
            //restoring email reminder
            item.setEmailReminderAt(item.getPaymentEndsAt().minus(properties.getEmailInterval()));
            return;
        }
        log.info("Will schedule automatic payment for pending invoice {}", item.getPendingInvoice().getId());
        ctx.scheduleExternalEvent(item.getPendingInvoice().getWorkflow().getId(),
                TStartPayment.newBuilder()
                        .setPaymentProvider(EPaymentProvider.PP_TRUST)
                        .setPaymentMethodId(item.getBoundPaymentMethod().getId()).build());
    }
}
