package ru.yandex.travel.orders.workflows.plus.topup.handlers;

import javax.money.CurrencyUnit;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.orders.entities.FiscalItemType;
import ru.yandex.travel.orders.entities.YandexPlusTopup;
import ru.yandex.travel.orders.entities.finances.FinancialEventPaymentScheme;
import ru.yandex.travel.orders.services.finances.FinancialEventService;
import ru.yandex.travel.orders.services.payments.PaymentProfile;
import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.services.payments.TrustClientProvider;
import ru.yandex.travel.orders.services.payments.TrustUserInfo;
import ru.yandex.travel.orders.services.payments.model.TrustBoundPaymentMethod;
import ru.yandex.travel.orders.services.payments.model.TrustResponseStatus;
import ru.yandex.travel.orders.services.payments.model.plus.TrustCreateAccountRequest;
import ru.yandex.travel.orders.services.payments.model.plus.TrustCreateAccountResponse;
import ru.yandex.travel.orders.services.payments.model.plus.TrustCreateTopupResponse;
import ru.yandex.travel.orders.services.payments.model.plus.TrustTopupPassParams;
import ru.yandex.travel.orders.services.payments.model.plus.TrustTopupPayload;
import ru.yandex.travel.orders.services.payments.model.plus.TrustTopupRequest;
import ru.yandex.travel.orders.services.plus.YandexPlusPayloadService;
import ru.yandex.travel.orders.workflow.plus.proto.EYandexPlusTopupState;
import ru.yandex.travel.orders.workflow.plus.proto.TTopupStart;
import ru.yandex.travel.orders.workflow.plus.proto.TTopupStartPayment;
import ru.yandex.travel.orders.workflows.plus.topup.YandexPlusPromoProperties;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Service
@Slf4j
@RequiredArgsConstructor
public class NewStateHandler extends AnnotatedStatefulWorkflowEventHandler<EYandexPlusTopupState, YandexPlusTopup> {
    private final TrustClientProvider trustClientProvider;
    private final YandexPlusPromoProperties properties;
    private final FinancialEventService financialEventService;
    private final YandexPlusPayloadService payloadService;

    @HandleEvent
    public void handleTopupStart(TTopupStart event, StateContext<EYandexPlusTopupState, YandexPlusTopup> context) {
        YandexPlusTopup topup = context.getWorkflowEntity();
        // only rouble yandex accounts are supported at the moment
        CurrencyUnit currency = topup.getCurrency();

        TrustClient trustClient = trustClientProvider.getTrustClientForPaymentProfile(topup.getPaymentProfile());
        TrustUserInfo userInfo = new TrustUserInfo(topup.getPassportId(), topup.getUserIp());

        TrustBoundPaymentMethod method = trustClient.getPaymentMethods(userInfo).getBoundYandexAccountMethod(currency);
        String plusAccountId;
        if (method != null) {
            plusAccountId = method.getId();
        } else {
            log.info("User {} doesn't have a Yandex Plus account, creating a new one", userInfo.getUid());
            TrustCreateAccountResponse accountRsp = trustClient.createAccount(TrustCreateAccountRequest.builder()
                    .currency(currency.getCurrencyCode())
                    .build(), userInfo);
            Preconditions.checkState(accountRsp.getStatus() == TrustResponseStatus.SUCCESS,
                    "Failed to create a new Yandex Plus account for passport id %s with currency %s; status %s",
                    userInfo.getUid(), currency, accountRsp.getStatus());
            plusAccountId = accountRsp.getPaymentMethodId();
        }

        TrustTopupRequest topupRequest = TrustTopupRequest.builder()
                .paymethodId(plusAccountId)
                .productId(FiscalItemType.HOTELS_YANDEX_PLUS_CASHBACK.getTrustId())
                .amount(topup.getAmount().longValue())
                .currency(currency.getCurrencyCode())
                .passParams(TrustTopupPassParams.builder()
                        .payload(getTrustTopupPayload(topup))
                        .build())
                .build();

        try (NestedMdc ignored = NestedMdc.forOptionalEntity(topup.getOrderItem())) {
            log.info("Starting Yandex Plus topup for {} points", topup.getAmount());
            TrustCreateTopupResponse topupBasket = trustClient.createTopup(topupRequest, userInfo);
            topup.setPurchaseToken(topupBasket.getPurchaseToken());
        }

        context.setState(EYandexPlusTopupState.PS_STARTING_PAYMENT);
        context.scheduleEvent(TTopupStartPayment.newBuilder().build());
    }

    private TrustTopupPayload getTrustTopupPayload(YandexPlusTopup topup) {
        if (topup.getOrderItem() != null) {
            return payloadService.createTopupPayloadForService(topup.getOrderItem());
        } else {
            Preconditions.checkState(properties.isToursEnabled(), "Topups for tours are not enabled yet");
            Preconditions.checkState(topup.getPaymentProfile() == PaymentProfile.HOTEL, "Topups are allowed only for hotels (tours use hotels profile too)");
            Preconditions.checkState(topup.getTotalAmountForPayload() != null);
            Preconditions.checkState(topup.getCommissionAmountForPayload() != null);
            Preconditions.checkState(topup.getVatCommissionAmountForPayload() != null);
            var finData = YandexPlusPayloadService.ServiceFinancialData.builder()
                    .total(topup.getTotalAmountForPayload())
                    .fee(topup.getCommissionAmountForPayload())
                    .feeVat(topup.getVatCommissionAmountForPayload())
                    .build();
            return payloadService.createTopupPayloadForTours(finData, FinancialEventPaymentScheme.HOTELS);
        }
    }
}
