package ru.yandex.travel.orders.workflows.plus.topup.handlers;

import java.time.Instant;

import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.entities.YandexPlusTopup;
import ru.yandex.travel.orders.services.payments.model.TrustBasketStatusResponse;
import ru.yandex.travel.orders.workflow.invoice.proto.TTrustPaymentStatusChanged;
import ru.yandex.travel.orders.workflow.plus.proto.EYandexPlusTopupState;
import ru.yandex.travel.orders.workflow.plus.proto.TStoreFinEvents;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Slf4j
@RequiredArgsConstructor
public class WaitingPaymentStateHandler
        extends AnnotatedStatefulWorkflowEventHandler<EYandexPlusTopupState, YandexPlusTopup> {
    @HandleEvent
    public void handleTrustPaymentStatusChanged(TTrustPaymentStatusChanged event,
                                                StateContext<EYandexPlusTopupState, YandexPlusTopup> ctx) {
        YandexPlusTopup topup = ctx.getWorkflowEntity();
        TrustBasketStatusResponse basketStatus = ProtoUtils.fromTJson(event.getBasketStatus(),
                TrustBasketStatusResponse.class);

        topup.setTrustPaymentId(basketStatus.getTrustPaymentId());
        switch (basketStatus.getPaymentStatus()) {
            case CLEARED:
                topup.setAuthorizedAt(Instant.now());
                ctx.setState(EYandexPlusTopupState.PS_CLEARED);
                try (NestedMdc ignored = NestedMdc.forOptionalEntity(topup.getOrderItem())) {
                    log.info("{} Yandex Plus points have been sent to the user", topup.getAmount());
                }
                // todo(tlg-13): not sending the events right now as there is no use in them
                // and to be extra careful and not to crash any order in manual processing
                //ctx.scheduleExternalEvent(topup.getOrderItem().getWorkflow().getId(),
                //        TPlusPointsTopupCompleted.newBuilder().setTopupId(topup.getId().toString()).build());
                if (!topup.isSkipFinancialEvents()) {
                    ctx.scheduleEvent(TStoreFinEvents.newBuilder().build());
                } else {
                    try (NestedMdc ignored = NestedMdc.forOptionalEntity(topup.getOrderItem())) {
                        log.info("Skipping financial events for topup ({})", topup.getId());
                    }
                }
                break;
            case NOT_AUTHORIZED:
            case CANCELED:
                try (NestedMdc ignored = NestedMdc.forOptionalEntity(topup.getOrderItem())) {
                    log.warn("Payment for topup {} not cleared. Failure reason: {}", topup.getId(),
                            topup.getAuthorizationErrorCode());
                }
                ctx.setState(EYandexPlusTopupState.PS_NOT_CLEARED);
                //ctx.scheduleExternalEvent(topup.getOrderItem().getWorkflow().getId(),
                //        TPlusPointsTopupCompletedWithError.newBuilder().setTopupId(topup.getId().toString()).build());
                if (!Strings.isNullOrEmpty(basketStatus.getPaymentRespCode())) {
                    topup.setAuthorizationErrorCode(basketStatus.getPaymentRespCode());
                    topup.setAuthorizationErrorDesc(basketStatus.getPaymentRespDesc());
                }
                // re-sending the event to the state without an appropriate handler simply to fail the whole workflow;
                // manual investigation is required in such case
                ctx.scheduleEvent(event);
                break;
            default:
                break;
        }
    }
}
