package ru.yandex.travel.orders.workflows.plus.topup.jobs;

import java.time.Instant;
import java.util.Collection;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.repository.YandexPlusTopupRepository;
import ru.yandex.travel.orders.services.payments.TrustClientProvider;
import ru.yandex.travel.orders.services.payments.TrustUserInfo;
import ru.yandex.travel.orders.services.payments.model.TrustBasketStatusResponse;
import ru.yandex.travel.orders.workflow.invoice.proto.TTrustPaymentStatusChanged;
import ru.yandex.travel.orders.workflow.plus.proto.EYandexPlusTopupState;
import ru.yandex.travel.orders.workflows.plus.topup.YandexPlusPromoProperties;
import ru.yandex.travel.task_processor.AbstractSimpleTask;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.workflow.WorkflowMessageSender;

import static ru.yandex.travel.orders.repository.JpaRepositoryUtils.NO_EXCLUDE_UUIDS;
import static ru.yandex.travel.orders.repository.JpaRepositoryUtils.firstPage;

@Service
@Slf4j
@RequiredArgsConstructor
public class YandexPlusTopupRefreshService extends AbstractSimpleTask<UUID> {

    private final YandexPlusTopupRepository yandexPlusTopupRepository;

    private final WorkflowMessageSender workflowMessageSender;

    private final TrustClientProvider trustClientProvider;

    private final YandexPlusPromoProperties properties;

    @Override
    @TransactionMandatory
    public Collection<UUID> getPendingTaskKeys(int maxResultSize) {
        return yandexPlusTopupRepository.getTopupIdsAwaitingRefreshInStateWithExclusions(Instant.now(),
                EYandexPlusTopupState.PS_WAIT_FOR_PAYMENT, EWorkflowState.WS_RUNNING,
                getLockedTaskKeysSafe(NO_EXCLUDE_UUIDS), firstPage(maxResultSize));
    }

    @Override
    @TransactionMandatory
    public long getPendingTasksCount() {
        return yandexPlusTopupRepository.countTopupsAwaitingRefreshInStateWithExclusions(Instant.now(),
                EYandexPlusTopupState.PS_WAIT_FOR_PAYMENT, EWorkflowState.WS_RUNNING,
                getLockedTaskKeysSafe(NO_EXCLUDE_UUIDS));
    }

    @Override
    @TransactionMandatory
    public void process(UUID invoiceId) {
        var topup = yandexPlusTopupRepository.getOne(invoiceId);
        try (var ignored = NestedMdc.forEntity(topup.getId(), topup.getEntityType())) {
            if (!topup.isBackgroundJobActive()) {
                log.info("Background job not active, returning");
                return;
            }

            TrustBasketStatusResponse trustBasketStatusResponse =
                    trustClientProvider.getTrustClientForPaymentProfile(topup.getPaymentProfile()).getBasketStatus(
                            topup.getPurchaseToken(), new TrustUserInfo(topup.getPassportId(), topup.getUserIp())
                    );

            switch (trustBasketStatusResponse.getPaymentStatus()) {
                case CLEARED:
                case NOT_AUTHORIZED:
                    topup.setBackgroundJobActive(false);
                    topup.setNextCheckStatusAt(null);
                    workflowMessageSender.scheduleEvent(topup.getWorkflow().getId(),
                            TTrustPaymentStatusChanged.newBuilder()
                                    .setBasketStatus(ProtoUtils.toTJson(trustBasketStatusResponse))
                                    .build());
                    break;
                default:
                    topup.rescheduleNextCheckStatusAt(properties.getAwaitPaymentCheckPeriod());
                    break;
            }
        } catch (Exception e) {
            log.error("Exception getting trust status while waiting payment", e);
            topup.rescheduleNextCheckStatusAt(properties.getAwaitPaymentCheckPeriod());
        }
    }
}
