package ru.yandex.travel.orders.workflows.supervisors.order;

import java.util.UUID;

import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.entities.WellKnownWorkflowEntityType;
import ru.yandex.travel.orders.grpc.helpers.ProtoChecks;
import ru.yandex.travel.orders.management.StarTrekService;
import ru.yandex.travel.workflow.MessagingContext;
import ru.yandex.travel.workflow.TWorkflowCrashed;
import ru.yandex.travel.workflow.WorkflowMaintenanceService;
import ru.yandex.travel.workflow.base.AnnotatedWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Slf4j
@RequiredArgsConstructor
public class OrderSupervisorWorkflowEventHandler extends AnnotatedWorkflowEventHandler<Void> {
    private final StarTrekService starTrekService;
    private final WorkflowMaintenanceService workflowMaintenanceService;

    @HandleEvent
    public void handleWorkflowCrashedEvent(TWorkflowCrashed event, MessagingContext<Void> ctx) {
        UUID entityId;
        UUID workflowId;

        try {
            entityId = ProtoChecks.checkStringIsUuid("entity id", event.getEntityId());
            workflowId = ProtoChecks.checkStringIsUuid("workflow id", event.getWorkflowId());
        } catch (Exception e) {
            log.error("Error parsing entity/workflow ids", e);
            return; // we can't move further without proper ids
        }

        try {
            log.info("Pausing workflows supervised by {}", workflowId);
            workflowMaintenanceService.pauseSupervisedRunningWorkflows(workflowId);
        } catch (Exception e) {
            log.error("Pausing workflows supervised by {} failed", workflowId, e);
        }

        String entityType = event.getEntityType();
        if (Strings.isNullOrEmpty(entityType)) {
            log.error("Missing entity type while handling the crashed workflow {}", workflowId);
            return;
        }
        try {
            if (entityType.equals(WellKnownWorkflowEntityType.HOTEL_ORDER.getDiscriminatorValue()) ||
                    entityType.equals(WellKnownWorkflowEntityType.AEROFLOT_ORDER.getDiscriminatorValue()) ||
                    entityType.equals(WellKnownWorkflowEntityType.TRAIN_ORDER.getDiscriminatorValue()) ||
                    entityType.equals(WellKnownWorkflowEntityType.GENERIC_ORDER.getDiscriminatorValue())) {
                Long eventId = event.getEventId();
                starTrekService.createIssueForGeneralOrderError(entityId, eventId, ProtoUtils.toInstant(event.getHappenedAt()), ctx);
            } else {
                log.info("Unknown entity type {} while handling the crashed workflow {}", event.getEntityType(),
                        workflowId);
                return;
            }
        } catch (Exception e) {
            // We're extra cautious to avoid failure of order supervisor workflow
            log.error("Error creating ticket for failed workflow {}", workflowId, e);
        }
    }
}
