package ru.yandex.travel.orders.workflows.voucher.handlers;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.entities.Voucher;
import ru.yandex.travel.orders.workflow.voucher.proto.EVoucherState;
import ru.yandex.travel.orders.workflow.voucher.proto.TGenerateVoucher;
import ru.yandex.travel.orders.workflow.voucher.proto.TVoucherStateUpdateError;
import ru.yandex.travel.orders.workflow.voucher.proto.TVoucherStateUpdated;
import ru.yandex.travel.orders.workflows.voucher.VoucherWorkflowConfigurationProperties;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Slf4j
@RequiredArgsConstructor
public class CreatingStateHandler extends AnnotatedStatefulWorkflowEventHandler<EVoucherState, Voucher> {
    private final VoucherWorkflowConfigurationProperties configurationProperties;
    private final VoucherWorkflowService voucherWorkflowService;

    @HandleEvent
    public void handleVoucherStateUpdated(TVoucherStateUpdated event, StateContext<EVoucherState, Voucher> context) {
        Voucher voucher = context.getWorkflowEntity();
        boolean recreated = voucher.getVoucherUrl() != null;
        voucher.setVoucherUrl(event.getVoucherUrl());
        voucherWorkflowService.handleVoucherGenerated(context, recreated);
    }

    @HandleEvent
    public void handleStateUpdateError(TVoucherStateUpdateError event, StateContext<EVoucherState, Voucher> context) {
        Voucher voucher = context.getWorkflowEntity();
        if (voucher.getPayload().getGenerationTry() < configurationProperties.getMaxGenerateTries()) {
            voucher.getPayload().setGenerationTry(voucher.getPayload().getGenerationTry() + 1);
            voucherWorkflowService.startVoucherGeneration(context);
        } else {
            voucher.setState(EVoucherState.VS_FAILED);
            if (voucher.isCrashWorkflowOnError()) {
                throw new RuntimeException("Generation tries limit exceeded");
            }
        }
    }

    @HandleEvent
    public void handleGenerateVoucher(TGenerateVoucher event, StateContext<EVoucherState, Voucher> context) {
        Voucher voucher = context.getWorkflowEntity();
        Preconditions.checkState(voucher.getNextCheckCreatedAt() == null, "Generation process already running");
        voucher.getPayload().setGenerationTry(voucher.getPayload().getGenerationTry() + 1);
        log.info("Start generation, try number = {}", voucher.getPayload().getGenerationTry());
        voucherWorkflowService.startVoucherGeneration(context);
    }
}
