package ru.yandex.travel.orders.workflows.voucher.handlers;

import java.time.Instant;
import java.time.temporal.ChronoUnit;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.orders.entities.Voucher;
import ru.yandex.travel.orders.services.pdfgenerator.PdfGeneratorService;
import ru.yandex.travel.orders.services.pdfgenerator.model.PdfGenerateBusesTicketsRequest;
import ru.yandex.travel.orders.services.pdfgenerator.model.PdfGenerateHotelsBusinessTripDocRequest;
import ru.yandex.travel.orders.services.pdfgenerator.model.PdfGenerateHotelsVoucherRequest;
import ru.yandex.travel.orders.workflow.voucher.proto.EVoucherState;
import ru.yandex.travel.orders.workflow.voucher.proto.EVoucherType;
import ru.yandex.travel.orders.workflow.voucher.proto.TVoucherCreated;
import ru.yandex.travel.orders.workflow.voucher.proto.TVoucherRecreated;
import ru.yandex.travel.orders.workflows.voucher.VoucherWorkflowConfigurationProperties;
import ru.yandex.travel.workflow.StateContext;

@Service
@RequiredArgsConstructor
@Slf4j
public class VoucherWorkflowService {
    private final VoucherWorkflowConfigurationProperties voucherWorkflowConfigurationProperties;
    private final PdfGeneratorService pdfGeneratorService;

    public void generateVoucher(StateContext<EVoucherState, Voucher> context) {
        Voucher voucher = context.getWorkflowEntity();
        if (voucher.getVoucherType() != EVoucherType.VT_UNKNOWN) {
            voucher.getPayload().setGenerationTry(1);
            startVoucherGeneration(context);
        } else {
            throw new RuntimeException(String.format("Can't generate voucher with type %s",
                    voucher.getVoucherType().toString()));
        }
    }

    public void startVoucherGeneration(StateContext<EVoucherState, Voucher> context) {
        Voucher voucher = context.getWorkflowEntity();
        voucher.getPayload().setGenerationStarted(Instant.now().truncatedTo(ChronoUnit.SECONDS));
        switch (voucher.getVoucherType()) {
            case VT_HOTELS:
                pdfGeneratorService.generateHotelsVoucher(
                        new PdfGenerateHotelsVoucherRequest(voucher.getOrderId().toString(), voucher.getVoucherName()));
                break;
            case VT_BUSES_TICKET:
                pdfGeneratorService.generateBusesTickets(
                        new PdfGenerateBusesTicketsRequest(voucher.getOrderId().toString(), voucher.getVoucherName()));
                break;
            case VT_HOTELS_BUSINESS_TRIP:
                pdfGeneratorService.generateHotelsBusinessTripDoc(
                        new PdfGenerateHotelsBusinessTripDocRequest(voucher.getOrderId().toString(), voucher.getVoucherName()));
                break;
            default:
                throw new RuntimeException(String.format("Can't generate voucher with type %s",
                        voucher.getVoucherType().toString()));
        }
        voucher.setNextCheckCreatedAt(Instant.now().plus(voucherWorkflowConfigurationProperties.getCheckCreatedPeriod()));
        voucher.getPayload().setCheckCreatedTill(Instant.now().plus(voucherWorkflowConfigurationProperties.getCheckCreatedTimeout()));
        context.setState(EVoucherState.VS_CREATING);
    }

    public void handleVoucherGenerated(StateContext<EVoucherState, Voucher> context, boolean recreated) {
        Voucher voucher = context.getWorkflowEntity();
        String documentUrl = voucher.getVoucherUrl();
        Preconditions.checkState(!Strings.isNullOrEmpty(documentUrl), "Voucher url can't be empty");
        context.setState(EVoucherState.VS_CREATED);
        if (recreated) {
            context.scheduleExternalEvent(voucher.getOrderWorkflowId(),
                    TVoucherRecreated.newBuilder()
                            .setVoucherUrl(documentUrl)
                            .setVoucherType(voucher.getVoucherType()).build());
        } else {
            context.scheduleExternalEvent(voucher.getOrderWorkflowId(),
                    TVoucherCreated.newBuilder()
                            .setVoucherUrl(documentUrl)
                            .setVoucherType(voucher.getVoucherType()).build());
        }
    }
}
