package ru.yandex.travel.orders.workflows.voucher.jobs;


import java.time.Instant;
import java.util.Collection;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.orders.entities.Voucher;
import ru.yandex.travel.orders.repository.VoucherRepository;
import ru.yandex.travel.orders.services.pdfgenerator.PdfGeneratorService;
import ru.yandex.travel.orders.services.pdfgenerator.PdfNotFoundException;
import ru.yandex.travel.orders.services.pdfgenerator.model.PdfStateResponse;
import ru.yandex.travel.orders.workflow.voucher.proto.EVoucherState;
import ru.yandex.travel.orders.workflow.voucher.proto.TVoucherStateUpdateError;
import ru.yandex.travel.orders.workflow.voucher.proto.TVoucherStateUpdated;
import ru.yandex.travel.orders.workflows.voucher.VoucherWorkflowConfigurationProperties;
import ru.yandex.travel.task_processor.AbstractSimpleTask;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.workflow.WorkflowMessageSender;

@Service
@Slf4j
@RequiredArgsConstructor
public class VoucherCheckStateTask extends AbstractSimpleTask<UUID> {
    private final VoucherRepository voucherRepository;
    private final VoucherWorkflowConfigurationProperties voucherWorkflowConfigurationProperties;
    private final WorkflowMessageSender workflowMessageSender;
    private final PdfGeneratorService pdfGeneratorService;

    @TransactionMandatory
    public Collection<UUID> getPendingTaskKeys(int maxResultSize) {
        return voucherRepository.getVouchersNextCheckCreatedAt(Instant.now(),
                EVoucherState.VS_CREATING, EWorkflowState.WS_RUNNING,
                PageRequest.of(0, maxResultSize));
    }

    @TransactionMandatory
    public long getPendingTasksCount() {
        return voucherRepository.countVouchersNextCheckCreatedAt(Instant.now(),
                EVoucherState.VS_CREATING, EWorkflowState.WS_RUNNING);
    }

    @TransactionMandatory
    public void process(UUID id) {
        Voucher voucher = voucherRepository.getOne(id);

        if (voucher.getState() == EVoucherState.VS_CREATING &&
                voucher.getNextCheckCreatedAt() != null &&
                voucher.getNextCheckCreatedAt().isBefore(Instant.now())
        ) {
            try (var ignored = NestedMdc.forEntity(voucher)) {
                try {
                    PdfStateResponse stateRsp = pdfGeneratorService.getState(voucher.getVoucherName());
                    if (!voucher.getPayload().getGenerationStarted().isAfter(stateRsp.getLastModified())) {
                        voucher.setNextCheckCreatedAt(null);
                        workflowMessageSender.scheduleEvent(voucher.getWorkflow().getId(),
                                TVoucherStateUpdated.newBuilder().setVoucherUrl(stateRsp.getUrl()).build());
                        return;
                    } else {
                        log.info("voucher not ready with LastModified={}", stateRsp.getLastModified().toString());
                    }
                } catch (PdfNotFoundException e) {
                    log.info("voucher not ready");
                } catch (RuntimeException e) {
                    log.error("voucher get state error", e);
                }
                if (voucher.getPayload().getCheckCreatedTill().isBefore(Instant.now())) {
                    // timeout
                    voucher.setNextCheckCreatedAt(null);
                    workflowMessageSender.scheduleEvent(voucher.getWorkflow().getId(),
                            TVoucherStateUpdateError.newBuilder().build());
                } else {
                    voucher.setNextCheckCreatedAt(Instant.now().plus(voucherWorkflowConfigurationProperties.getCheckCreatedPeriod()));
                }
            }
        }
    }
}
