CREATE OR REPLACE FUNCTION cancel_dolphin_order_crashed_while_waiting_for_confirmation(orderPrettyId varchar, dolphinCode varchar) RETURNS varchar AS
$$
DECLARE
    orderId           uuid;
    serviceId         uuid;
    invoiceId         uuid;
    orderWorkflowId   uuid;
    serviceWorkflowId uuid;
    invoiceWorkflowId uuid;
    orderState        int;
    serviceState      int;
    invoiceState      int;
    orderWfState      int;
    serviceWfState    int;
    invoiceWfState    int;
    orderWfType       varchar;
    serviceWfType     varchar;
    invoiceWfType     varchar;

BEGIN
    /*
    Данная функция предназначена для отмены Дельфиньего заказа, упавшего во время ожидания подтверждения при том, что
    заказ успел создаться на стороне партнера, но этот факт не сохранился на нашей стороне. В этой ситуации на стороне
    партнера есть бронирование, находящеейся в каком-либо статусе, а на нашей стороне НЕ известен код этого
    бронирования, а деньги за него взяты с клиента.
    Для отмены такого заказа необходимо предварительно выяснить код бронирования не стороне партнера, после чего
    выполнить эту функцию, передав этот код в качестве параметра, чтобы аннулировать бронирование у партнера и вернуть
    деньги пользователю.

    Эту функцию стоит выполнять либо если заказ на стороне партенра не находится в подтвержденном состоянии, либо если
    за время прошедшее со времени ошибки пользователь успел отказаться от бронирования. В остальных случаях имеет смысл
    восстанавливать заказ.

    Функция делает следующее:
    1) Проверяет, что заказ действительно дельфиний и действительно упал во время ождания потверждения
    2) Сохраняет в услугу переданный код подтвреждения у партнера
    3) Переводит услугу в состояние IS_CANCELLING
    4) Отправляет в workflow услуги событие TCancellationCommit
    5) Восстанавливает все workflow (услуги, инвойса и заказа)

    После успешного выполнения функции оркестратор сам запустит выполнение обработчика события TCancellationCommit
    workflow услуги, находящейся в состоянии IS_CANCELLING. Этот обработчик аннулирует услугу у партнера,
    переведет услугу в состояние IS_CANCELLED и просигнализиурет в workflow отельного заказа, что услуга отменена. Это,
    в свою очередь, приведет к тому, что workflow заказа инициирует возврат денег по услугам и в конечном счете отменит
    заказ.
    */
    SELECT id INTO STRICT orderId FROM orders WHERE pretty_id = orderPrettyId;
    SELECT id INTO STRICT serviceId FROM order_items WHERE order_id = orderId;
    SELECT id INTO STRICT invoiceId FROM invoices WHERE order_id = orderId;
    SELECT workflow_id INTO STRICT orderWorkflowId FROM orders WHERE id = orderId;
    SELECT workflow_id INTO STRICT serviceWorkflowId FROM order_items WHERE id = serviceId;
    SELECT workflow_id INTO STRICT invoiceWorkflowId FROM invoices WHERE order_id = orderId;
    RAISE NOTICE 'Will cancel order %, service %, invoice %. Order WF - %, service Wf - %, invoice WF - %',
        orderId, serviceId, invoiceId, orderWorkflowId, serviceWorkflowId, invoiceWorkflowId;

    SELECT state INTO STRICT orderState FROM orders where id = orderId;
    SELECT state INTO STRICT serviceState FROM order_items where id = serviceId;
    SELECT state INTO STRICT invoiceState FROM invoices where id = invoiceId;
    SELECT state INTO STRICT orderWfState FROM workflows where id = orderWorkflowId;
    SELECT state INTO STRICT serviceWfState FROM workflows where id = serviceWorkflowId;
    SELECT state INTO STRICT invoiceWfState FROM workflows where id = invoiceWorkflowId;
    SELECT entity_type INTO STRICT orderWfType FROM workflows where id = orderWorkflowId;
    SELECT entity_type INTO STRICT serviceWfType FROM workflows where id = serviceWorkflowId;
    SELECT entity_type INTO STRICT invoiceWfType FROM workflows where id = invoiceWorkflowId;

    -- checking current states
    assert orderState = 5; -- waiting_confirmation
    assert serviceState = 4; -- confirming
    assert invoiceState = 3; -- hold
    assert serviceWfState = 4; -- crashed
    assert orderWfState = 4; -- crashed
    assert invoiceWfState = 2; -- paused
    assert orderWfType = 'hotel_order';
    assert serviceWfType = 'dolphin_order_item';
    assert invoiceWfType = 'trust_invoice';

    -- move service to CANCELLING state and set dolphin code
    UPDATE order_items SET state = 3, dolphin_order_code=dolphinCode WHERE id = serviceId;

    -- Schedule TCancellationCommit event
    INSERT INTO workflow_events(id, workflow_id, created_at, data, class_name, state)
    VALUES (nextval('workflow_events_id_seq'),
            serviceWorkflowId,
            now(),
            '',
            'ru.yandex.travel.orders.workflow.hotels.dolphin.proto.TCancellationCommit',
            1);

    -- recover WFs
    UPDATE workflows SET state = 1 WHERE id in (serviceWorkflowId, orderWorkflowId, invoiceWorkflowId);

    return 'Заказ успешно отменен';
END;
$$ LANGUAGE plpgsql;


CREATE OR REPLACE FUNCTION cancel_dolphin_order_crashed_while_in_waitlist(orderPrettyId varchar) RETURNS varchar AS
$$
    /*
     Данная функция предназначена для отмены Дельфиньего заказа, упавшего в WaitList, то есть находящегося в ситуации,
     когда на стороне партнера есть бронирование, находящеейся в WAIT_LIST статусе, а на нашей стороне известен код этого
     бронирования - и взяты деньги за него.
     Для отмены такого заказа необходимо аннулировать бронирование у партнера и вернуть деньги пользователю.

     Функция делает следующее:
     1) Проверяет, что заказ действительно дельфиний и действительно упал в Waitlist
     2) Переводит услугу в состояние IS_CANCELLING
     3) Отправляет в workflow услуги событие TCancellationCommit
     4) Восстанавливает все workflow (услуги, инвойса и заказа)

     После успешного выполнения функции оркестратор сам запустит выполнение обработчика события TCancellationCommit
     workflow услуги, находящейся в состоянии IS_CANCELLING. Этот обработчик аннулирует услугу у партнера,
     переведет услугу в состояние IS_CANCELLED и просигнализиурет в workflow отельного заказа, что услуга отменена. Это,
     в свою очередь, приведет к тому, что workflow заказа инициирует возврат денег по услугам и в конечном счете отменит
     заказ.
     */
DECLARE
    orderId           uuid;
    serviceId         uuid;
    invoiceId         uuid;
    orderWorkflowId   uuid;
    serviceWorkflowId uuid;
    invoiceWorkflowId uuid;
    orderState        int;
    serviceState      int;
    invoiceState      int;
    orderWfState      int;
    serviceWfState    int;
    invoiceWfState    int;
    orderWfType       varchar;
    serviceWfType     varchar;
    invoiceWfType     varchar;

BEGIN
    SELECT id INTO STRICT orderId FROM orders WHERE pretty_id = orderPrettyId;
    SELECT id INTO STRICT serviceId FROM order_items WHERE order_id = orderId;
    SELECT id INTO STRICT invoiceId FROM invoices WHERE order_id = orderId;
    SELECT workflow_id INTO STRICT orderWorkflowId FROM orders WHERE id = orderId;
    SELECT workflow_id INTO STRICT serviceWorkflowId FROM order_items WHERE id = serviceId;
    SELECT workflow_id INTO STRICT invoiceWorkflowId FROM invoices WHERE order_id = orderId;
    RAISE NOTICE 'Will cancel order %, service %, invoice %. Order WF - %, service Wf - %, invoice WF - %',
        orderId, serviceId, invoiceId, orderWorkflowId, serviceWorkflowId, invoiceWorkflowId;

    SELECT state INTO STRICT orderState FROM orders where id = orderId;
    SELECT state INTO STRICT serviceState FROM order_items where id = serviceId;
    SELECT state INTO STRICT invoiceState FROM invoices where id = invoiceId;
    SELECT state INTO STRICT orderWfState FROM workflows where id = orderWorkflowId;
    SELECT state INTO STRICT serviceWfState FROM workflows where id = serviceWorkflowId;
    SELECT state INTO STRICT invoiceWfState FROM workflows where id = invoiceWorkflowId;
    SELECT entity_type INTO STRICT orderWfType FROM workflows where id = orderWorkflowId;
    SELECT entity_type INTO STRICT serviceWfType FROM workflows where id = serviceWorkflowId;
    SELECT entity_type INTO STRICT invoiceWfType FROM workflows where id = invoiceWorkflowId;

    -- checking current states
    assert orderState = 5; -- waiting_confirmation
    assert serviceState = 100; -- confirming
    assert invoiceState = 3; -- hold
    assert serviceWfState = 4; -- crashed
    assert orderWfState = 4; -- crashed
    assert invoiceWfState = 2; -- paused
    assert orderWfType = 'hotel_order';
    assert serviceWfType = 'dolphin_order_item';
    assert invoiceWfType = 'trust_invoice';

    -- move service to CANCELLING state and set dolphin code
    UPDATE order_items SET state = 3 WHERE id = serviceId;

    -- Schedule TCancellationCommit event
    INSERT INTO workflow_events(id, workflow_id, created_at, data, class_name, state)
    VALUES (nextval('workflow_events_id_seq'),
            serviceWorkflowId,
            now(),
            '',
            'ru.yandex.travel.orders.workflow.hotels.dolphin.proto.TCancellationCommit',
            1);

    -- recover WFs
    UPDATE workflows SET state = 1 WHERE id in (serviceWorkflowId, orderWorkflowId, invoiceWorkflowId);
    return 'Заказ успешно отменен';
END;
$$ LANGUAGE plpgsql;


CREATE OR REPLACE FUNCTION restore_dolphin_order_crashed_while_waiting_for_confirmation(orderPrettyId varchar) RETURNS varchar AS
$$
DECLARE
    orderId           uuid;
    serviceId         uuid;
    invoiceId         uuid;
    orderWorkflowId   uuid;
    serviceWorkflowId uuid;
    invoiceWorkflowId uuid;
    orderState        int;
    serviceState      int;
    invoiceState      int;
    orderWfState      int;
    serviceWfState    int;
    invoiceWfState    int;
    orderWfType       varchar;
    serviceWfType     varchar;
    invoiceWfType     varchar;
    eventId           int;

BEGIN
    /*
    Данная функция предназначена для восстановления Дельфиньего заказа, упавшего во время ожидания подтверждения при
    том, что заказ успел создаться на стороне партнера, но этот факт не сохранился на нашей стороне. В этой ситуации на
    стороне партнера есть бронирование, находящеейся в каком-либо статусе, а на нашей стороне НЕ известен код этого
    бронирования, а деньги за него взяты с клиента.
    Для восстановления заказа необходимо завершить подтверждение заказа с нашей стороны, в том числе привязать к услуге
    код заказа на стороне партнера.

    Эту функцию стоит выполнять либо если заказ на стороне партенра находится в подтвержденном состоянии (в противном
    случае заказа снова упадет) и нет цели вернуть деньги пользователю. В соатльных случаях заказ надо отменять,
    используя функцию cancel_dolphin_order_crashed_while_waiting_for_confirmation

    Функция делает следующее:
    1) Проверяет, что заказ действительно дельфиний и действительно упал во время ождания потверждения
    2) Отменяет обработку последнего события TConfirmationCommit
    3) Восстанавливает все workflow (услуги, инвойса и заказа)

    После успешного выполнения функции оркестратор повторно запустит выполнение обработчика события TConfirmationCommit
    workflow услуги, находящейся в состоянии IS_CONFIRMING. Этот обработчик запросит у партнера все известные
    бронироания, найдет бронирование соответствующее текущему заказу, привяжет его код - и продолжит выполнение
    workflow дальше - создаст ваучер, отправит его по почте и тп.
    */
    SELECT id INTO STRICT orderId FROM orders WHERE pretty_id = orderPrettyId;
    SELECT id INTO STRICT serviceId FROM order_items WHERE order_id = orderId;
    SELECT id INTO STRICT invoiceId FROM invoices WHERE order_id = orderId;
    SELECT workflow_id INTO STRICT orderWorkflowId FROM orders WHERE id = orderId;
    SELECT workflow_id INTO STRICT serviceWorkflowId FROM order_items WHERE id = serviceId;
    SELECT workflow_id INTO STRICT invoiceWorkflowId FROM invoices WHERE order_id = orderId;
    RAISE NOTICE 'Will restore order %, service %, invoice %. Order WF - %, service Wf - %, invoice WF - %',
        orderId, serviceId, invoiceId, orderWorkflowId, serviceWorkflowId, invoiceWorkflowId;

    SELECT state INTO STRICT orderState FROM orders where id = orderId;
    SELECT state INTO STRICT serviceState FROM order_items where id = serviceId;
    SELECT state INTO STRICT invoiceState FROM invoices where id = invoiceId;
    SELECT state INTO STRICT orderWfState FROM workflows where id = orderWorkflowId;
    SELECT state INTO STRICT serviceWfState FROM workflows where id = serviceWorkflowId;
    SELECT state INTO STRICT invoiceWfState FROM workflows where id = invoiceWorkflowId;
    SELECT entity_type INTO STRICT orderWfType FROM workflows where id = orderWorkflowId;
    SELECT entity_type INTO STRICT serviceWfType FROM workflows where id = serviceWorkflowId;
    SELECT entity_type INTO STRICT invoiceWfType FROM workflows where id = invoiceWorkflowId;

    -- checking current states
    assert orderState = 5; -- waiting_confirmation
    assert serviceState = 4; -- confirming
    assert invoiceState = 3; -- hold
    assert serviceWfState = 4; -- crashed
    assert orderWfState = 4; -- crashed
    assert invoiceWfState = 2; -- paused
    assert orderWfType = 'hotel_order';
    assert serviceWfType = 'dolphin_order_item';
    assert invoiceWfType = 'trust_invoice';

    -- re-process the latest crashed TConfirmationCommit event in service workflow
    SELECT id
    INTO strict eventId
    FROM workflow_events
    WHERE class_name = 'ru.yandex.travel.orders.workflow.hotels.dolphin.proto.TConfirmationCommit'
      AND workflow_id = serviceWorkflowId
      AND state = 3
    ORDER BY created_at DESC
    LIMIT 1;
    RAISE NOTICE 'Will re-process event % of workflow %', eventId, serviceWorkflowId;
    UPDATE workflow_events SET state = 1 WHERE id = eventId;

    -- recover WFs
    UPDATE workflows SET state = 1 WHERE id in (serviceWorkflowId, orderWorkflowId, invoiceWorkflowId);
    return 'Заказ успешно восстановлен';
END;
$$ LANGUAGE plpgsql;
