DROP FUNCTION cancel_dolphin_order_crashed_while_waiting_for_confirmation;
DROP FUNCTION cancel_dolphin_order_crashed_while_in_waitlist;

CREATE OR REPLACE FUNCTION restore_dolphin_order_while_in_manual_confirmation(orderPrettyId varchar) RETURNS varchar AS
$$
    /*
     Данная функция предназначена для продолжения работы Дельфиньего заказа, находящегося в статусе MANUAL_CONFIRMATION.
     Она переводит заказ обратно в автоматическую обработку, синхронизируя статус услуги со статусом бронирования у
     партнера. Иначе говоря, если на момент возвращения заказ у партнера подтвержден, то и услуга будет переведена в
     IS_CONFIRMED, после чего заказ подтвердиться, выпишутся документы и тп. Если же заказ у партенра аннулирован, то и
     услуга будет переведена в IS_CANCELLED, деньги возвращены, заказ отменен (важно: это именно cancellation, а не
     refund). Если же заказ у партнера все еще в статусе WaitList, то workflow снова перейдет в IS_MANUAL_CONFIRMATION.

     Перед выполеннием функции имеет смысл либо аннулировать, либо подтвердить заказ на стороне партнера.

     Функция делает следующее:
     1) Проверяет, что заказ действительно дельфиний и услуга действительно упал в IS_MANUAL_CONFIRMATION
     2) Переводит услугу в состояние IS_POLLING_FOR_STATUS
     3) Переводит заказ в состояние OS_WAITING_CONFIRMATION
     4) Отправляет в workflow услуги событие TRefresh
     5) Восстанавливает все workflow (инвойса и заказа)

     */
DECLARE
    orderId           uuid;
    serviceId         uuid;
    invoiceId         uuid;
    orderWorkflowId   uuid;
    serviceWorkflowId uuid;
    invoiceWorkflowId uuid;
    orderState        int;
    serviceState      int;
    invoiceState      int;
    orderWfState      int;
    serviceWfState    int;
    invoiceWfState    int;
    orderWfType       varchar;
    serviceWfType     varchar;
    invoiceWfType     varchar;

BEGIN
    SELECT id INTO STRICT orderId FROM orders WHERE pretty_id = orderPrettyId;
    SELECT id INTO STRICT serviceId FROM order_items WHERE order_id = orderId;
    SELECT id INTO STRICT invoiceId FROM invoices WHERE order_id = orderId;
    SELECT workflow_id INTO STRICT orderWorkflowId FROM orders WHERE id = orderId;
    SELECT workflow_id INTO STRICT serviceWorkflowId FROM order_items WHERE id = serviceId;
    SELECT workflow_id INTO STRICT invoiceWorkflowId FROM invoices WHERE order_id = orderId;
    RAISE NOTICE 'Will restore order %, service %, invoice %. Order WF - %, service Wf - %, invoice WF - %',
        orderId, serviceId, invoiceId, orderWorkflowId, serviceWorkflowId, invoiceWorkflowId;

    SELECT state INTO STRICT orderState FROM orders where id = orderId;
    SELECT state INTO STRICT serviceState FROM order_items where id = serviceId;
    SELECT state INTO STRICT invoiceState FROM invoices where id = invoiceId;
    SELECT state INTO STRICT orderWfState FROM workflows where id = orderWorkflowId;
    SELECT state INTO STRICT serviceWfState FROM workflows where id = serviceWorkflowId;
    SELECT state INTO STRICT invoiceWfState FROM workflows where id = invoiceWorkflowId;
    SELECT entity_type INTO STRICT orderWfType FROM workflows where id = orderWorkflowId;
    SELECT entity_type INTO STRICT serviceWfType FROM workflows where id = serviceWorkflowId;
    SELECT entity_type INTO STRICT invoiceWfType FROM workflows where id = invoiceWorkflowId;

    -- checking current states
    assert orderState = 11; -- manual_processing
    assert serviceState = 10; -- manual_confirmation
    assert invoiceState = 3; -- hold
    assert orderWfState = 1; -- running
    assert serviceWfState = 2; -- paused
    assert invoiceWfState = 2; -- paused
    assert orderWfType = 'hotel_order';
    assert serviceWfType = 'dolphin_order_item';
    assert invoiceWfType = 'trust_invoice';
    -- move service to IS_POLLING_FOR_STATUS state
    UPDATE order_items SET state = 100 WHERE id = serviceId;

    -- move order back to to OS_WAITING_CONFIRMATION state
    UPDATE orders SET state = 5 WHERE id = orderId;

    -- Schedule TRefresh event
    INSERT INTO workflow_events(id, workflow_id, created_at, data, class_name, state)
    VALUES (nextval('workflow_events_id_seq'),
            serviceWorkflowId,
            now(),
            '',
            'ru.yandex.travel.orders.workflow.hotels.dolphin.proto.TRefresh',
            1);

    -- recover WFs
    UPDATE workflows SET state = 1 WHERE id in (serviceWorkflowId, invoiceWorkflowId);
    return 'Заказ успешно восстановлен';
END;
$$ LANGUAGE plpgsql;


CREATE OR REPLACE FUNCTION cancel_dolphin_order_while_in_manual_confirmation(orderPrettyId varchar) RETURNS varchar AS
$$
    /*
     Данная функция предназначена для отмены Дельфиньего заказа, находящегося в статусе MANUAL_CONFIRMATION.

     Если для услуги известен order_code на стороне партнера (то есть есть у Дельфина есть бронирование), то данная
     функция переведет услугу в состояние IS_CANCELLING и инициирует отмену, отправив услуге сообщение
     TCancellationCommit. Это приведет к аннулированию бронирования на стороне партнера, после чего услуга перейдет в
     состояние IS_CANCELLED и просигнализиурет в workflow отельного заказа, что услуга отменена. Это, в свою очередь,
     приведет к тому, что workflow заказа инициирует возврат денег по услугам и в конечном счете отменит заказ.

     Если же order_code партнерского заказа не известен (то есть у Дельфина бронирование создасться не успело), то
     данная функция сразу переведет услугу в IS_CANCELLED и отправит в в workflow отельного заказа сообщение
     TServiceCancelled. Это приведет к тому, что workflow заказа инициирует возврат денег по услугам и в конечном счете
     отменит заказ.

     Перед выполеннием функции очень важно убедиться, что на стороне партнера действительно нет заказа. В противном
     случае лучше восставить заказ (ака повторить его создание) вызовом функции
     restore_dolphin_order_crashed_while_waiting_for_confirmation и потом уже обычным образом отменять его,
     если необходимо.

     Функция делает следующее:
     1) Проверяет, что заказ действительно дельфиний и услуга действительно упал в IS_MANUAL_CONFIRMATION
     2) Переводит заказ в состояние OS_WAITING_CONFIRMATION
     3) Смотрит, есть ли у заказа order_code
       - Если да, то переводит услугу в состояние IS_CANCELLING и отправляет ей событие TCancellationCommit
       - Если нет, то переводит услугу в состояние IS_CANCELLED и отправляет заказу событие TServiceCancelled
     4) Восстанавливает все workflow (услуги, инвойса и заказа)
     */
DECLARE
    orderId           uuid;
    serviceId         uuid;
    invoiceId         uuid;
    orderWorkflowId   uuid;
    serviceWorkflowId uuid;
    invoiceWorkflowId uuid;
    orderState        int;
    serviceState      int;
    invoiceState      int;
    orderWfState      int;
    serviceWfState    int;
    invoiceWfState    int;
    orderWfType       varchar;
    serviceWfType     varchar;
    invoiceWfType     varchar;
    orderCode         varchar;

BEGIN
    SELECT id INTO STRICT orderId FROM orders WHERE pretty_id = orderPrettyId;
    SELECT id INTO STRICT serviceId FROM order_items WHERE order_id = orderId;
    SELECT id INTO STRICT invoiceId FROM invoices WHERE order_id = orderId;
    SELECT workflow_id INTO STRICT orderWorkflowId FROM orders WHERE id = orderId;
    SELECT workflow_id INTO STRICT serviceWorkflowId FROM order_items WHERE id = serviceId;
    SELECT workflow_id INTO STRICT invoiceWorkflowId FROM invoices WHERE order_id = orderId;
    RAISE NOTICE 'Will restore order %, service %, invoice %. Order WF - %, service Wf - %, invoice WF - %',
        orderId, serviceId, invoiceId, orderWorkflowId, serviceWorkflowId, invoiceWorkflowId;

    SELECT state INTO STRICT orderState FROM orders where id = orderId;
    SELECT state INTO STRICT serviceState FROM order_items where id = serviceId;
    SELECT state INTO STRICT invoiceState FROM invoices where id = invoiceId;
    SELECT state INTO STRICT orderWfState FROM workflows where id = orderWorkflowId;
    SELECT state INTO STRICT serviceWfState FROM workflows where id = serviceWorkflowId;
    SELECT state INTO STRICT invoiceWfState FROM workflows where id = invoiceWorkflowId;
    SELECT entity_type INTO STRICT orderWfType FROM workflows where id = orderWorkflowId;
    SELECT entity_type INTO STRICT serviceWfType FROM workflows where id = serviceWorkflowId;
    SELECT entity_type INTO STRICT invoiceWfType FROM workflows where id = invoiceWorkflowId;

    -- checking current states
    assert orderState = 11; -- manual_processing
    assert serviceState = 10; -- manual_confirmation
    assert invoiceState = 3; -- hold
    assert orderWfState = 1; -- running
    assert serviceWfState = 2; -- paused
    assert invoiceWfState = 2; -- paused
    assert orderWfType = 'hotel_order';
    assert serviceWfType = 'dolphin_order_item';
    assert invoiceWfType = 'trust_invoice';

    -- move order back to to OS_WAITING_CONFIRMATION state
    UPDATE orders SET state = 5 WHERE id = orderId;

    -- есть ли заказ у партнера?
    SELECT dolphin_order_code INTO STRICT orderCode from order_items where id = serviceId;
    IF orderCode is not null THEN
        RAISE NOTICE 'Заказ с кодом % будет аннулирован у партнера и отменен', orderCode;
        -- move service to CANCELLING state
        UPDATE order_items SET state = 3 WHERE id = serviceId;

        -- Schedule TCancellationCommit event
        INSERT INTO workflow_events(id, workflow_id, created_at, data, class_name, state)
        VALUES (nextval('workflow_events_id_seq'),
                serviceWorkflowId,
                now(),
                '',
                'ru.yandex.travel.orders.workflow.hotels.dolphin.proto.TCancellationCommit',
                1);
    ELSE
        RAISE NOTICE 'Заказ будет сразу отменен';
        -- move service to CANCELLED state
        UPDATE order_items SET state = 7 WHERE id = serviceId;

        -- Schedule TServiceCancelled event
        INSERT INTO workflow_events(id, workflow_id, created_at, data, class_name, state)
        VALUES (nextval('workflow_events_id_seq'),
                orderWorkflowId,
                now(),
                '',
                'ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled',
                1);

    END IF;
    -- recover WFs
    UPDATE workflows SET state = 1 WHERE id in (serviceWorkflowId, invoiceWorkflowId);
    return 'Заказ успешно отменен';
END;
$$ LANGUAGE plpgsql;
