/*
    Usage:
        -- real cancellation
        select cancel_aeroflot_order('___YA_9999_9999_9999___');
        -- dry run
        select cancel_aeroflot_order('___YA_9999_9999_9999___', /*do_update=*/false);

        -- To cancel all broken for a period:
        select cancel_aeroflot_order((select o.pretty_id from orders o where o.id = ods.order_id), true), pretty_id
        from order_search_documents ods
        where broken = true and order_type = 2/*aeroflot*/
            and order_created_at between '2021-01-20' and '2021-02-02' order by created_at desc limit 15

    To drop:
        drop function cancel_aeroflot_order(varchar, bool);
*/
CREATE or replace function cancel_aeroflot_order(order_pretty_id varchar, do_update bool default true) returns varchar
language plpgsql
as $$
declare
    _order_id uuid;
    _item_id uuid;
    _invoice_id uuid;
    order_state_single int;
    order_state varchar;
    pending_events_desc varchar;
    wf record;
    updated_rows int;
begin

    select id, state from orders where pretty_id = order_pretty_id and order_type = 'aeroflot_order' into _order_id, order_state_single;
    select id from order_items where order_id = _order_id into _item_id;
    select id from invoices where order_id = _order_id into _invoice_id;

    if _order_id is null then
        raise notice 'No such Aeroflot order - %', order_pretty_id;
        return 'not found';
    end if;
    if order_state_single = 6/*OS_CONFIRMED*/ or order_state_single = 7/*OS_CANCELLED*/ then
        return 'already completed';
    end if;

    select 'order:' || coalesce((select get_enum_name('%EAeroflotOrderState', state) from orders where id = _order_id), '<null>')
       || ', service: ' || coalesce((select get_enum_name('%EAeroflotItemState', state) from order_items oi where oi.order_id = _order_id), '<null>')
       || ', invoice: ' || coalesce((select get_enum_name('%EAeroflotInvoiceState', state) from invoices i where i.order_id = _order_id), '<null>')
    into order_state;

    RAISE NOTICE 'Order id %, state %', _order_id, order_state;

    if do_update then
        update orders set state = 7 /*OS_CANCELLED*/ where id = _order_id;
        update order_items set state = 8 /*IS_CANCELLED*/ where order_id = _order_id;
        update invoices set state = 8 /*IS_CANCELLED*/ where order_id = _order_id;
        RAISE NOTICE '    ... changed states to: %',
             (select 'order:' || coalesce((select get_enum_name('%EAeroflotOrderState', state) from orders where id = _order_id), '<null>')
            || ', service: ' || coalesce((select get_enum_name('%EAeroflotItemState', state) from order_items oi where oi.order_id = _order_id), '<null>')
            || ', invoice: ' || coalesce((select get_enum_name('%EAeroflotInvoiceState', state) from invoices i where i.order_id = _order_id), '<null>'));
    end if;

    for wf in (select * from workflows where entity_id in (_order_id, _item_id, _invoice_id))
    loop
        select STRING_AGG(regexp_replace(class_name, '([^.]+\.)+', '')||':'||get_enum_name('%EWorkflowEventState', state), ', ')
        from workflow_events w where workflow_id = wf.id and state != 2/*PROCESSED*/ into pending_events_desc;

        RAISE NOTICE 'Wf for entity %: workflow id %,%state %,%pending events [%]', wf.entity_type, wf.id,
                chr(10)||'    ', get_enum_name('%EWorkflowState', wf.state),
                chr(10)||'    ', pending_events_desc;

        if do_update then
            if wf.state != 1 /*WS_RUNNING*/ then
                update workflows set state = 1/*RUNNING*/ where id = wf.id;
                get diagnostics updated_rows = row_count;
                RAISE NOTICE '    ... restoring the workflow - %', updated_rows;
            end if;
            if pending_events_desc is not null then
                update workflow_events set state = 2 /*WES_PROCESSED*/ where workflow_id = wf.id and state != 2/*PROCESSED*/;
                get diagnostics updated_rows = row_count;
                RAISE NOTICE '    ... dropping the pending events - %', updated_rows;
            end if;
        end if;
    end loop;

    if do_update is null or not do_update then
        RAISE NOTICE 'Dry run, not doing anything';
    end if;
    if do_update then
        insert into order_changes(id, order_id, created_at)
        select nextval('order_change_id_seq'), _order_id, now();
        RAISE NOTICE 'Scheduling re-indexing';
    end if;

    return 'ok';

end; $$
;
