/*
    Usage:
        select restore_workflows_recursively(workflow_id) from orders where pretty_id = '___PRETTY_ID___';
        select restore_workflows_recursively(id) from workflows where entity_id = '___ENTITY_UUID___';
        select restore_workflows_recursively('___ANY_ENTITY_WORKFLOW_ID___');
        -- dry run
        select restore_workflows_recursively('___ANY_ENTITY_WORKFLOW_ID___', /*do_update=*/false);
    To drop:
        drop function restore_workflows_recursively(uuid, bool);
*/
CREATE or replace function restore_workflows_recursively(_workflow_id uuid, do_update bool default true) returns varchar
as $$
DECLARE
    DECLARE
    workflows_cursor CURSOR (workflow_id uuid) FOR WITH RECURSIVE child_workflows(id, supervisor_id, state)
                                                                   AS (SELECT w.id, w.supervisor_id, w.state
                                                                       FROM workflows w
                                                                       where id = workflow_id
                                                                       UNION ALL
                                                                       SELECT w.id, w.supervisor_id, w.state
                                                                       FROM workflows w,
                                                                            child_workflows n
                                                                       WHERE n.id = w.supervisor_id)
                                                select id
                                                from child_workflows
                                                where state in ('2'/*paused*/, '4'/*crashed*/);
    wid uuid;
BEGIN

    -- order workflow id
    OPEN workflows_cursor(_workflow_id);

    LOOP
        FETCH workflows_cursor INTO wid;
        EXIT WHEN NOT FOUND;

        -- excluding order_supervisor & generic_error_supervisor which recursively supervise ALL workflows
        IF wid in ('7cb3f93c-7ae6-44f1-b2a4-6596e180879d', 'd2a7d7a8-01d6-400c-8be5-0edfbe49c38e') THEN
            RAISE EXCEPTION 'Never restore supervisor % recursively!', wid;
        END IF;

        RAISE NOTICE 'Restoring workflow %', wid;

        if do_update then
            -- setting workflow crashed event to processed
            update workflow_events set state = '2' /*processed*/ where workflow_id = wid and state = '3'/*crashed*/
                                                                   and class_name = 'ru.yandex.travel.workflow.TWorkflowCrashed';
            update workflow_events set state = '1' /*new*/ where workflow_id = wid and state = '3'/*crashed*/;

            -- resuming workflow
            update workflows set state = '1' /*running*/ where id = wid;
        else
            RAISE NOTICE 'Dry run, not doing anything';
        end if;

    END LOOP;

    CLOSE workflows_cursor;

    return 'OK';
END;
$$
    LANGUAGE plpgsql;
