-- Dashboard "Partner payouts", tab "Payment Balance"
-- https://datalens.yandex-team.ru/zv7fpa6w5014w-partner-payouts?tab=60D

-- DROP VIEW view_for_hotels_payment_balance;
CREATE OR REPLACE VIEW view_for_hotels_payment_balance
AS
WITH financial_events_direct AS (
    SELECT fe.billing_contract_id    AS financial_contract_id,
           fe.hotel_money,
           fe.partner_total          AS total_ordered,
           fe.partner_refunded_total AS total_refunded,
           fe.payout_at,
           fe.order_item_id,
           fe.id
    FROM view_financial_events_calculated fe
             JOIN order_items oi ON oi.item_type in ('travelline', 'bnovo') AND oi.id = fe.order_item_id
    WHERE fe.type <> 'yandex_account_topup_payment'
),
     billing_order_details_signed AS (
         SELECT *,
                -- to keep assumptions about the intervals in one place, we'll extract it here
                CASE
                    WHEN status = 'started' AND created_at > CURRENT_TIMESTAMP - interval '1 day'
                        THEN signed_sum
                    ELSE 0
                    END AS money_being_processed_atm,
                CASE
                    WHEN status = 'done'
                        THEN signed_sum
                    ELSE 0
                    END AS have_been_paid_to_hotel_sum,
                CASE
                    -- corresponds to oebs_payment_status = returned
                    WHEN status = 'cancelled' THEN signed_sum
                    -- it appears that such payments are likely to have failed.
                    WHEN status = 'started' AND updated_at < CURRENT_TIMESTAMP - interval '1 day'
                        THEN signed_sum
                    ELSE 0
                    END AS money_failed,
                CASE
                    WHEN status = 'cancelled' AND created_at > CURRENT_TIMESTAMP - interval '1 week'
                        THEN signed_sum
                    ELSE 0
                    END AS money_failed_recently
         FROM (
                  SELECT bod.id,
                         bod.contract_id,
                         CASE
                             WHEN bod.transaction_type = 'payment' THEN bod.sum
                             WHEN bod.transaction_type = 'refund' THEN -bod.sum
                             END AS signed_sum,
                         bod.yt_id,
                         bo.oebs_payment_status,
                         bo.status,
                         bo.created_at,
                         bo.updated_at
                  FROM bank_order_details bod
                           JOIN bank_orders bo
                                ON bo.payment_batch_id = bod.payment_batch_id
                                    -- by some reason we often have another bank order with "created" status
                                    AND bo.bank_order_id is not null AND bo.bank_order_id <> ''
                       -- we exclude fee, topup and other verticals-specific payments
                  WHERE bod.payment_type in ('cost', 'yandex_account_withdraw')
              ) AS billing_order_details_signed_subselect
     ),
     billing_transactions_signed AS
         (
             SELECT yt_id,
                    id,
                    source_financial_event_id,
                    CASE
                        WHEN transaction_type = 'payment' THEN value_amount
                        WHEN transaction_type = 'refund' THEN -value_amount
                        END AS signed_sum,
                    exported_to_yt,
                    exported_to_yt_at,
                    payout_at
             FROM billing_transactions
                  -- we care only about hotel money here
             WHERE payment_type = 'cost'
         )
SELECT
    -- add new fields only to the bottom of the list or Datalens will blow up
    contract_id,
    balance,
    "ordered - refunded",
    ordered,
    refunded,
    sent_to_billing,
    fe_count,
    not_yet_sent_to_billing,
    billing_count,
    paid_to_hotel,
    money_being_processed_atm,
    money_failed_recently,
    missed_by_billing,
    missed_by_billing_events,
    abs(COALESCE(impeccable_billing_balance, 0))                        AS balance_abs,
    balance - COALESCE(missed_by_billing, 0)                            AS balance_error_compensated, -- having not 0 in that place might indicate an error in calculation or inconsistency in data
    money_failed,
    possibly_failed_in_billing_with_no_info,
    possibly_failed_in_billing_with_no_info_events,
    array_to_string(missed_by_billing_events_array, ', ')               AS missed_by_billing_events_array,
    array_to_string(possibly_failed_in_billing_with_no_info_events_array,
                    ', ')                                               AS possibly_failed_in_billing_with_no_info_events_array,
    COALESCE(impeccable_billing_balance, 0)                             AS impeccable_billing_balance,
    array_to_string(financial_events_witn_no_billing_counterpart, ', ') AS financial_events_witn_no_billing_counterpart,
    array_to_string(hotel_names, '; ')                                  AS hotel_names,
    array_to_string(hotel_inns, '; ')                                   AS hotel_inns
FROM (
         SELECT per_billing.contract_id,
                per_financial_events.sent_to_billing -
                (per_billing.have_been_paid_to_hotel_sum + per_billing.money_being_processed_atm) AS balance,

                per_financial_events.hotel_money_sum                                              AS "ordered - refunded",
                per_financial_events.total_ordered_sum                                            AS ordered,
                per_financial_events.total_refunded_sum                                           AS refunded,
                per_financial_events.sent_to_billing                                              AS sent_to_billing,
                per_financial_events.financial_events_count                                       AS fe_count,
                per_financial_events.to_be_paid                                                   AS not_yet_sent_to_billing,
                per_financial_events.hotel_names,
                per_financial_events.hotel_inns,

                per_billing.billing_order_details_count                                           AS billing_count,
                per_billing.have_been_paid_to_hotel_sum                                           AS paid_to_hotel,
                per_billing.money_being_processed_atm                                             AS money_being_processed_atm,
                per_billing.money_failed_recently                                                 AS money_failed_recently,
                per_billing.money_failed                                                          AS money_failed,

                error_1.missed_by_billing,
                error_1.missed_by_billing_events,
                error_1.missed_by_billing_events_array,

                error_2.possibly_failed_in_billing_with_no_info,
                error_2.possibly_failed_in_billing_with_no_info_events,
                error_2.possibly_failed_in_billing_with_no_info_events_array,

                impeccable_billing.no_billing_counterpart                                         AS impeccable_billing_balance,
                impeccable_billing.financial_events_witn_no_billing_counterpart
         FROM (
                  -- ============= `per_financial_events` ==============
                  SELECT fe.financial_contract_id,
                         count(*)                                AS financial_events_count,
                         sum(fe.hotel_money)                     AS hotel_money_sum,
                         sum(fe.total_ordered)                   AS total_ordered_sum,
                         sum(fe.total_refunded)                  AS total_refunded_sum,
                         sum(CASE
                                 WHEN fe.payout_at >= now() THEN fe.hotel_money
                                 ELSE 0
                             END)                                AS to_be_paid,
                         sum(CASE
                                 WHEN fe.payout_at < now() THEN fe.hotel_money
                                 ELSE 0
                             END)                                AS sent_to_billing,
                         array_agg(DISTINCT vfhoi.hotel_name)    AS hotel_names,
                         array_agg(DISTINCT vfhoi.agreement_inn) AS hotel_inns
                  FROM financial_events_direct fe
                           JOIN view_for_hotel_order_item_payload_details vfhoi
                                ON vfhoi.order_item_id = fe.order_item_id
                  GROUP BY 1
              ) AS per_financial_events
                  JOIN
              (
                  -- ============= `per_billing` ==============
                  SELECT bod.contract_id,
                         sum(signed_sum)                  AS total_per_billing,
                         sum(have_been_paid_to_hotel_sum) AS have_been_paid_to_hotel_sum,
                         sum(money_being_processed_atm)   AS money_being_processed_atm,
                         sum(money_failed)                AS money_failed,
                         sum(money_failed_recently)       AS money_failed_recently,
                         count(*)                         AS billing_order_details_count
                  FROM billing_order_details_signed bod
                  GROUP BY 1
              ) per_billing ON per_billing.contract_id::text = per_financial_events.financial_contract_id::text
                  LEFT JOIN (
             -- ============= `error_1` ==============
             -- when we've got no info from billing for a specific yt_id.
             -- at the moment on our side it indistinguishable whether we lost a record from billing
             -- or the hotel owns us money. Having 0 or a result similar to negative balance is fine.
             -- --------------------
             SELECT sum(fe.hotel_money) AS missed_by_billing,
                    count(*)            AS missed_by_billing_events,
                    array_agg(id)       AS missed_by_billing_events_array,
                    fe.financial_contract_id
             FROM financial_events_direct fe
             WHERE id in (
                 SELECT DISTINCT __fe.id
                 FROM financial_events_direct __fe
                          LEFT JOIN billing_transactions bt
                                    ON bt.source_financial_event_id = __fe.id AND bt.exported_to_yt
                                        AND bt.payment_type = 'cost'
                          LEFT JOIN billing_order_details_signed bod ON bod.yt_id = bt.yt_id
                 WHERE __fe.payout_at < now()
                   AND bod.id is null
             )
             GROUP BY fe.financial_contract_id
         ) AS error_1 ON error_1.financial_contract_id = per_financial_events.financial_contract_id
                  LEFT JOIN (
             -- ============= `error_2` =====================
             -- the last bank order is "in process" for too long
             -- similar to money_failed in "per_billing" block, but deduplicated per fin events
             SELECT sum(fe.hotel_money) AS possibly_failed_in_billing_with_no_info,
                    count(*)            AS possibly_failed_in_billing_with_no_info_events,
                    array_agg(fe.id)    AS possibly_failed_in_billing_with_no_info_events_array,
                    fe.financial_contract_id
             FROM financial_events_direct fe
             WHERE (fe.id IN (SELECT DISTINCT bt.source_financial_event_id
                              FROM billing_transactions bt
                                       JOIN billing_order_details_signed bod ON bod.yt_id = bt.yt_id
                              WHERE bt.exported_to_yt
                                AND bt.payment_type::text = 'cost'::text
                                AND bt.payout_at < now()
                                AND bod.status::text <> 'done'::text
                                AND bod.created_at < (CURRENT_TIMESTAMP - '1 day'::interval)
                                AND NOT (bt.yt_id IN (SELECT __bod.yt_id
                                                      FROM billing_order_details_signed __bod
                                                      WHERE __bod.status::text = 'done'::text))))
             GROUP BY fe.financial_contract_id
         ) AS error_2 ON error_2.financial_contract_id = per_financial_events.financial_contract_id
                  LEFT JOIN (
             -- ============= `impeccable_billing` =====================
             -- it's another approach to calculating billing balance:
             -- we consider that the last successful payment makes balance = 0
             -- and therefore it's enough to sum only the transactions after that.
             -- This will allow us to ignore the old errors.
             SELECT sum(bt.signed_sum)        AS no_billing_counterpart,
                    count(*)                  AS financial_events_witn_no_billing_counterpart_count,
                    array_agg(DISTINCT fe.id) AS financial_events_witn_no_billing_counterpart,
                    fe.financial_contract_id
             FROM (
                      SELECT max(yt_id) AS yt_id, contract_id
                      FROM billing_order_details_signed
                           -- successful or in progress transactions
                      WHERE money_being_processed_atm <> 0
                         OR have_been_paid_to_hotel_sum <> 0
                      GROUP BY contract_id
                  ) last_successfull_payment
                      RIGHT JOIN financial_events_direct fe on fe.financial_contract_id = last_successfull_payment.contract_id
                      RIGHT JOIN billing_transactions_signed bt on bt.yt_id > COALESCE(last_successfull_payment.yt_id, 0)
                 AND bt.source_financial_event_id = fe.id
             GROUP BY fe.financial_contract_id
         ) AS impeccable_billing
                            ON impeccable_billing.financial_contract_id = per_financial_events.financial_contract_id
         -- -----------------------------------------------------------------------------
     ) AS sub_select;

GRANT SELECT ON view_for_hotels_payment_balance TO orders_analytic_public;
