------------------------- MODULE ExpediaBookingModel ------------------------
(* ExpediaBookingModel specifies how we believe Expedia processes
 * all the bookings made through their API.
 *)

VARIABLE state

AllStates == {
  "ES_None",                 \* There was no activity so far.
  "ES_PriceMismatch",        \* The booking was not created due to a changed price.
  "ES_SoldOut",              \* The booking was not created due to a sold out inventory.
  "ES_InvalidData",          \* The booking was not created due to an invalid data.
  "ES_OnHold",               \* The booking was put on hold.
  "ES_Expired",              \* The held booking timed out while waiting to be confirmed.
  "ES_Cancelled",            \* The held booking was cancelled explicitly.
  "ES_Confirmed",            \* The held booking was confirmed.
  "ES_ConfirmedAndCancelled" \* The confirmed booking was cancelled.
}

-----------------------------------------------------------------------------
(***************************************************************************)
(* Actions                                                                 *)
(***************************************************************************)

LOCAL Transition(from, to) == (state = from) /\ (state' = to)

LOCAL Enabled404 == state \in {
  "ES_PriceMismatch", "ES_SoldOut", "ES_InvalidData",
  "ES_Expired", "ES_Cancelled"
}

(* Possible actions for the 'Hold' call. *)

\* 201 Created
HoldCall201 == Transition("ES_None", "ES_OnHold")
\* 400 Bad Request
HoldCall400 == Transition("ES_None", "ES_InvalidData")
\* 409 Price Mismatch
HoldCall409 == Transition("ES_None", "ES_PriceMismatch")
\* 410 Sold Out
HoldCall410 == Transition("ES_None", "ES_SoldOut")
\* TODO(sandello):
\*   > The booking link from your previous Price Check response expires after a short period.
\*   > If you receive an HTTP 503 error upon your first attempt, the link has likely expired.
\*   > Obtain a new link and attempt your booking again.

(* Possible actions for the 'CancelItinerary' call. *)

\* 202 Accepted
\* https://developer.expediapartnersolutions.com/documentation/rapid-manage-booking-docs-2-2/
\*   > Request is valid but the state of the booking is unknown.
\*   > Make a retrieve call to verify the booking has been canceled properly.
CancelItineraryCall202 ==
  \/ Transition("ES_OnHold", "ES_OnHold")
  \/ Transition("ES_OnHold", "ES_Cancelled")
\* 204 No Content
\* https://developer.expediapartnersolutions.com/documentation/rapid-manage-booking-docs-2-2/
\*   > Booking has successfully canceled.
\* https://developer.expediapartnersolutions.com/reference/hold-resume
\*   > Cancelling a held booking that is still on hold will release the inventory from hold
\*   > and return an HTTP 204 status
CancelItineraryCall204 == Transition("ES_OnHold", "ES_Cancelled")
\* 400 Bad Request / type==itinerary_level_cancel_not_supported
\*   > Cancelling a held booking that has already been successfully resumed
\*   > will return an "itinerary level cancel not supported" error message.
\*   > Make a Retrieve call and use the room-level links provided in the response
\*   > to cancel individual rooms.
CancelItineraryCall400_ItineraryLevelCancelNotSupported == (state = "ES_Confirmed") /\ UNCHANGED state
\* 404 Not Found
\* https://developer.expediapartnersolutions.com/reference/hold-resume
\*   > Cancelling a held booking that has already been cancelled (rolled back by request)
\*   > or has timed out (automatically rolled back) will return the "itinerary not found"
\*   > error documented earlier on this page.
CancelItineraryCall404 == Enabled404 /\ UNCHANGED state

(* Possible actions for the 'Resume' call. *)

\* 202 Accepted
\* https://developer.expediapartnersolutions.com/documentation/rapid-booking-docs-2-2/
\*   > Request is valid but the state of the booking is unknown.
\*   > Make a retrieve call to verify the booking has been resumed properly.
ResumeCall202 ==
  \/ Transition("ES_OnHold", "ES_OnHold")
  \/ Transition("ES_OnHold", "ES_Confirmed")
\* 204 No Content
\* https://developer.expediapartnersolutions.com/documentation/rapid-booking-docs-2-2/
\*   > Held booking has successfully resumed.
ResumeCall204 == Transition("ES_OnHold", "ES_Confirmed")
\* 400 Bad Request / type==already_committed
\* https://developer.expediapartnersolutions.com/reference/hold-resume
\*   > Resuming an on-hold booking that has already been successfully resumed
\*   > will return an error stating the booking is already committed.
ResumeCall400_AlredyCommitted == (state = "ES_Confirmed") /\ UNCHANGED state
\* 404 Not Found
\* https://developer.expediapartnersolutions.com/reference/hold-resume
\*   > Resuming an on-hold booking that has been cancelled (rolled back by request)
\*   > or has timed out (automatically rolled back) will return the same "itinerary not found"
\*   > error detailed in the previous retrieval section.
ResumeCall404 == Enabled404 /\ UNCHANGED state

(* Possible actions for the 'CancelRoom' call. *)

\* 202 Accepted
\* https://developer.expediapartnersolutions.com/documentation/rapid-manage-booking-docs-2-2/
\*   > Request is valid but the state of the booking is unknown.
\*   > Make a retrieve call to verify the booking has been canceled properly.
CancelRoomCall202 ==
  \/ Transition("ES_Confirmed", "ES_Confirmed")
  \/ Transition("ES_Confirmed", "ES_ConfirmedAndCancelled")
\* 204 No Content
\* https://developer.expediapartnersolutions.com/documentation/rapid-manage-booking-docs-2-2/
\*   > Booking has successfully canceled.
CancelRoomCall204 == Transition("ES_Confirmed", "ES_ConfirmedAndCancelled")
\* 404 Not Found
\* https://developer.expediapartnersolutions.com/documentation/rapid-manage-booking-docs-2-2/
CancelRoomCall404 == Enabled404 /\ UNCHANGED state

(* Possible actions that happen on Expedia side. *)
BackgroundNext ==
  \/ Transition("ES_OnHold", "ES_Expired")
  \* TODO(sandello): Support changes from the Voyager and the callbacks.

-----------------------------------------------------------------------------

IsNonexistent == state \in {"ES_PriceMismatch", "ES_SoldOut", "ES_InvalidData"}

IsCancelled == state \in {"ES_Expired", "ES_Cancelled"}

IsConfirmed == state \in {"ES_Confirmed"}

-----------------------------------------------------------------------------
(***************************************************************************)
(* Exploratory Specification                                               *)
(***************************************************************************)

ModelInit == (state = "ES_None")

ModelNext ==
  \/ HoldCall201 \/ HoldCall400 \/ HoldCall409 \/ HoldCall410
  \/ CancelItineraryCall202 \/ CancelItineraryCall204
  \/ CancelItineraryCall400_ItineraryLevelCancelNotSupported \/ CancelItineraryCall404
  \/ ResumeCall202 \/ ResumeCall204
  \/ ResumeCall400_AlredyCommitted \/ ResumeCall404
  \/ CancelRoomCall202 \/ CancelRoomCall204 \/ CancelRoomCall404
  \/ BackgroundNext

ModelSpec == ModelInit /\ [][ModelNext]_<<state>>

ModelTypeInvariant == (state \in AllStates)

THEOREM ModelSpec => ModelTypeInvariant

=============================================================================
\* Modification History
\* Last modified Tue Feb 05 11:37:05 MSK 2019 by sandello
\* Created Tue Jan 29 21:25:40 MSK 2019 by sandello
