------------------------- MODULE ExpediaServiceModel ------------------------
(* ExpediaServiceModel specifies the automata for the Expedia order item.
 * Note that the model requires an external entity to enqueue events.
 * Otherwise, the model will deadlock.
 *)

(* Well-known TODOs:
 * - Missing specification for the Refund flow.
 * - Missing constraints on itinerary data synchronization.
 *)

EXTENDS Naturals, Sequences

CONSTANT IsSharedRelay
ASSUME IsSharedRelay \in BOOLEAN

\* The service is modelled as a workflow instance.
VARIABLES state, queue, relay
\* The remote state of the booking.
VARIABLES remoteExpediaState

vars == <<state, queue, relay, remoteExpediaState>>

AllStates == {
  "EWS_New",
  "EWS_Reserving",
  "EWS_Reserved",
  "EWS_Cancelling",
  "EWS_Cancelled",
  "EWS_Confirming",
  "EWS_Confirmed",
  "EWS_Revoked",
  \* Failed state.
  "EWS_UnhandlerError"
}

AllConsumedEvents == {
  "SE_Reserve",
  "SE_ReserveCommit",
  "SE_Cancel",
  "SE_CancelCommit",
  "SE_Confirm",
  "SE_ConfirmCommit",
  "SE_Revoke"
}

AllProducedEvents == {
  "SE_Reserved",
  "SE_Cancelled",
  "SE_Confirmed",
  "SE_Revoked",
  \* Failed message.
  "SE_DeathLetter"
}

-----------------------------------------------------------------------------

E == INSTANCE ExpediaBookingModel WITH state <- remoteExpediaState

-----------------------------------------------------------------------------
(***************************************************************************)
(* Helper Actions                                                          *)
(***************************************************************************)

LOCAL Fail ==
  /\ state' = "EWS_UnhandlerError"
  /\ queue' = queue
  /\ relay' = Append(relay, "SE_DeathLetter")
  /\ UNCHANGED remoteExpediaState

LOCAL Retry == UNCHANGED <<state, queue, relay>>

Enqueue(event) ==
  /\ queue' = Append(queue, event)
  /\ IF IsSharedRelay
     THEN UNCHANGED <<state, remoteExpediaState>>
     ELSE UNCHANGED <<state, relay, remoteExpediaState>>

-----------------------------------------------------------------------------
(***************************************************************************)
(* State Handlers                                                          *)
(***************************************************************************)

ProcessEvent_NewStateHandler ==
  /\ state = "EWS_New"
  /\ Len(queue) > 0
  /\ LET event == Head(queue) IN
     CASE event = "SE_Reserve" ->
       /\ state' = "EWS_Reserving"
       /\ queue' = Append(Tail(queue), "SE_ReserveCommit")
       /\ relay' = relay
       /\ UNCHANGED remoteExpediaState
       [] OTHER -> Fail

ProcessEvent_ReservingStateHandler ==
  /\ state = "EWS_Reserving"
  /\ Len(queue) > 0
  /\ LET event == Head(queue) IN
     CASE event = "SE_ReserveCommit" ->
       \* Implementation should make the 'Hold' call
       \* and decide what to do depending on the result.
       \* If the call failed with 4xx then just cancel the service.
       \* If the call failed with 5xx then retry infinitely.
       (* One of the two outcomes are possible. *)
       \* The booking was created.
       \/ /\ (E!HoldCall201)
          /\ state' = "EWS_Reserved"
          /\ queue' = Tail(queue)
          /\ relay' = Append(relay, "SE_Reserved")
       \* The booking was not created.
       \/ /\ (E!HoldCall400 \/ E!HoldCall409 \/ E!HoldCall410)
          /\ state' = "EWS_Cancelled"
          /\ queue' = Tail(queue)
          /\ relay' = Append(relay, "SE_Cancelled")
       [] OTHER -> Fail

ProcessEvent_ReservedStateHandler ==
  /\ state = "EWS_Reserved"
  /\ Len(queue) > 0
  /\ LET event == Head(queue) IN
     CASE event = "SE_Cancel" ->
       /\ state' = "EWS_Cancelling"
       /\ queue' = Append(Tail(queue), "SE_CancelCommit")
       /\ UNCHANGED <<relay, remoteExpediaState>>
       [] event = "SE_Confirm" ->
       /\ state' = "EWS_Confirming"
       /\ queue' = Append(Tail(queue), "SE_ConfirmCommit")
       /\ UNCHANGED <<relay, remoteExpediaState>>
       [] OTHER -> Fail

ProcessEvent_CancellingStateHandler ==
  /\ state = "EWS_Cancelling"
  /\ Len(queue) > 0
  /\ LET event == Head(queue) IN
     CASE event = "SE_CancelCommit" ->
       \* The implementation should make the 'Cancel' call
       \* and decide what to do depending on the result.
       \* If the call failed then retry infinitely.
       (* Next subactions must be weakly fair (see `LivenessCancel`). *)
       \* The booking was cancelled.
       \/ /\ (E!CancelItineraryCall204 \/ E!CancelItineraryCall404)
          /\ state' = "EWS_Cancelled"
          /\ queue' = Tail(queue)
          /\ relay' = Append(relay, "SE_Cancelled")
       \* The booking was confirmed; this is unexpected.
       \/ /\ (E!CancelItineraryCall400_ItineraryLevelCancelNotSupported)
          /\ Fail
       (* Next subactions may stutter. *)
       \* The booking state is unknown; shall retrieve the booking.
       \/ /\ (E!CancelItineraryCall202)
          /\ Retry
       (* A stub handler for 4xx & 5xx retries. *)
       \/ /\ UNCHANGED remoteExpediaState
          /\ Retry
       [] OTHER -> Fail

ProcessEvent_CancelledStateHandler ==
  /\ state = "EWS_Cancelled"
  /\ Len(queue) > 0
  /\ LET event == Head(queue) IN
     Fail

ProcessEvent_ConfirmingStateHandler ==
  /\ state = "EWS_Confirming"
  /\ Len(queue) > 0
  /\ LET event == Head(queue) IN
     CASE event = "SE_ConfirmCommit" ->
       \* The implementation should make the 'Resume' call
       \* and decide what to do depending on the result.
       \* If the call failed then retry infinitely.
       (* Next subactions must be weakly fair (see `LivenessConfirm`). *)
       \* The booking was confirmed.
       \/ /\ (E!ResumeCall204 \/ E!ResumeCall400_AlredyCommitted)
          /\ state' = "EWS_Confirmed"
          /\ queue' = Tail(queue)
          /\ relay' = Append(relay, "SE_Confirmed")
       \* The booking was cancelled.
       \/ /\ (E!ResumeCall404)
          /\ state' = "EWS_Cancelled"
          /\ queue' = Tail(queue)
          /\ relay' = Append(relay, "SE_Cancelled")
       (* Next subactions may stutter. *)
       \* The booking state is unknown; shall retrieve the booking.
       \/ /\ (E!ResumeCall202)
          /\ Retry
       [] OTHER -> Fail

ProcessEvent_ConfirmedStateHandler ==
  /\ state = "EWS_Confirmed"
  /\ Len(queue) > 0
  /\ LET event == Head(queue) IN
     CASE event = "TRevoke" ->
       \* The booking was refunded.
       \/ /\ (E!CancelRoomCall202 \/ E!CancelRoomCall204)
          /\ state' = "EWS_Revoked"
          /\ queue' = Tail(queue)
          /\ relay' = Append(relay, "SE_Revoked")
       \* The booking was cancelled; this is unexpected.
       \/ /\ (E!CancelRoomCall404)
          /\ Fail
       \/ Fail \* TODO(sandello): Handle 2xx & 4xx.
       [] OTHER -> Fail

ProcessEvent_RevokedStateHandler ==
  /\ state = "EWS_Revoked"
  /\ Len(queue) > 0
  /\ LET event == Head(queue) IN
     Fail

ProcessEvent ==
  \/ ProcessEvent_NewStateHandler
  \/ ProcessEvent_ReservingStateHandler
  \/ ProcessEvent_ReservedStateHandler
  \/ ProcessEvent_CancellingStateHandler
  \/ ProcessEvent_CancelledStateHandler
  \/ ProcessEvent_ConfirmingStateHandler
  \/ ProcessEvent_ConfirmedStateHandler
  \/ ProcessEvent_RevokedStateHandler

-----------------------------------------------------------------------------

ModelInit ==
  /\ state = "EWS_New"
  /\ queue = <<>>
  /\ relay = <<>>
  /\ E!ModelInit

ModelNext ==
  \/ ProcessEvent
  \/ E!BackgroundNext /\ UNCHANGED <<state, queue, relay>>

ModelTypeInvariant ==
  /\ state \in AllStates
  /\ queue \in Seq(AllConsumedEvents)
  \* XXX(sandello): When the model is used with a shared relay,
  \* this invariant does not hold. (Think of the relay being the order queue).
  /\ ~IsSharedRelay => relay \in Seq(AllProducedEvents)
  /\ E!ModelTypeInvariant

=============================================================================
\* Modification History
\* Last modified Fri Feb 08 15:30:10 MSK 2019 by sandello
\* Created Thu Jan 31 14:31:09 MSK 2019 by sandello
