--------------------------- MODULE HotelOrderModel --------------------------

EXTENDS Naturals, Sequences

VARIABLES orderState, orderQueue
VARIABLES serviceState, serviceQueue
VARIABLES invoiceState, invoiceQueue
VARIABLES remoteExpediaState, remoteTrustState

vars == <<orderState, orderQueue,
          serviceState, serviceQueue,
          invoiceState, invoiceQueue,
          remoteExpediaState, remoteTrustState>>

AllStates == {
  "OWS_New",
  "OWS_WaitForServicesReserved",
  "OWS_WaitForPayment",
  "OWS_WaitForServicesConfirmed",
  "OWS_Confirmed",
  "OWS_Cancelled",
  \* Failed state.
  "OWS_UnhandledError"
}

AllEvents == {
  "OE_StartReservation",
  "OE_StartPayment"
}

-----------------------------------------------------------------------------

Service == INSTANCE ExpediaServiceModel WITH
  state <- serviceState,
  queue <- serviceQueue,
  relay <- orderQueue,
  remoteExpediaState <- remoteExpediaState,
  IsSharedRelay <- TRUE

Invoice == INSTANCE TrustInvoiceModel WITH
  state <- invoiceState,
  queue <- invoiceQueue,
  relay <- orderQueue,
  remoteTrustState <- remoteTrustState,
  IsSharedRelay <- TRUE

-----------------------------------------------------------------------------
(***************************************************************************)
(* Helper Actions                                                          *)
(***************************************************************************)

LOCAL UnchangedOrder ==
  UNCHANGED <<orderState, orderQueue>>
LOCAL UnchangedService ==
  UNCHANGED <<serviceState, serviceQueue, remoteExpediaState>>
LOCAL UnchangedInvoice ==
  UNCHANGED <<invoiceState, invoiceQueue, remoteTrustState>>
LOCAL UnchangedServiceAndInvoice ==
  UnchangedService /\ UnchangedInvoice

Fail ==
  /\ orderState' = "OWS_UnhandledError"
  /\ orderQueue' = orderQueue
  /\ UnchangedServiceAndInvoice

Retry ==
  /\ UNCHANGED vars

-----------------------------------------------------------------------------
(***************************************************************************)
(* External Actions                                                        *)
(***************************************************************************)

GRPC_Reserve ==
  /\ orderState = "OWS_New"
  /\ UNCHANGED orderState
  /\ orderQueue' = Append(orderQueue, "OE_StartReservation")
  /\ UnchangedServiceAndInvoice

GRPC_StartPayment ==
  /\ orderState = "OWS_WaitForPayment"
  /\ UNCHANGED orderState
  /\ orderQueue' = Append(orderQueue, "OE_StartPayment")
  /\ UnchangedServiceAndInvoice

-----------------------------------------------------------------------------
(***************************************************************************)
(* State Handlers                                                          *)
(***************************************************************************)

ProcessEvent_NewStateHandler ==
  /\ orderState = "OWS_New"
  /\ Len(orderQueue) > 0
  /\ LET event == Head(orderQueue) IN
     CASE event = "OE_StartReservation" ->
       /\ orderState' = "OWS_WaitForServicesReserved"
       /\ orderQueue' = Tail(orderQueue)
       /\ Service!Enqueue("SE_Reserve")
       /\ UnchangedInvoice
       [] OTHER -> Fail

ProcessEvent_WaitForServicesReservedStateHandler ==
  /\ orderState = "OWS_WaitForServicesReserved"
  /\ Len(orderQueue) > 0
  /\ LET event == Head(orderQueue) IN
     CASE event = "SE_Reserved" ->
       /\ orderState' = "OWS_WaitForPayment"
       /\ orderQueue' = Tail(orderQueue)
       /\ UnchangedServiceAndInvoice
       [] event = "SE_Cancelled" ->
       /\ orderState' = "OWS_Cancelled"
       /\ orderQueue' = Tail(orderQueue)
       /\ UnchangedServiceAndInvoice
       [] OTHER -> Fail

ProcessEvent_WaitForPaymentStateHandler ==
  /\ orderState = "OWS_WaitForPayment"
  /\ Len(orderQueue) > 0
  /\ LET event == Head(orderQueue) IN
     CASE event = "OE_StartPayment" ->
       /\ UNCHANGED orderState
       /\ orderQueue' = Tail(orderQueue)
       /\ Invoice!Enqueue("IE_TPaymentInitialized")
       /\ UnchangedService
       [] event = "IE_MoneyAuthorized" ->
       /\ orderState' = "OWS_WaitForServicesConfirmed"
       /\ orderQueue' = Tail(orderQueue)
       /\ Service!Enqueue("SE_Confirm")
       /\ UnchangedInvoice
       [] event = "IE_MoneyNotAuthorized" ->
       /\ Fail \* TODO(sandello): Manual processing? Cancel everything?
       [] OTHER -> Fail

ProcessEvent_WaitForServicesConfirmedStateHandler ==
  /\ orderState = "OWS_WaitForServicesConfirmed"
  /\ Len(orderQueue) > 0
  /\ LET event == Head(orderQueue) IN
     CASE event = "SE_Confirmed" ->
       /\ orderState' = "OWS_Confirmed"
       /\ orderQueue' = Tail(orderQueue)
       /\ UnchangedServiceAndInvoice
       [] event = "SE_Cancelled" ->
       /\ Fail \* TODO(sandello): Return some money.
       [] OTHER -> Fail

ProcessEvent_ConfirmedStateHandler ==
  /\ orderState = "OWS_Confirmed"
  /\ Len(orderQueue) > 0
  /\ LET event == Head(orderQueue) IN
     CASE event = "..." ->
       /\ Retry
       [] OTHER -> Fail

ProcessEvent_CancelledStateHandler ==
  /\ orderState = "OWS_Cancelled"
  /\ Len(orderQueue) > 0
  /\ LET event == Head(orderQueue) IN
     CASE event = "..." ->
       /\ Retry
       [] OTHER -> Fail

ProcessEvent ==
  \/ ProcessEvent_NewStateHandler
  \/ ProcessEvent_WaitForServicesReservedStateHandler
  \/ ProcessEvent_WaitForPaymentStateHandler
  \/ ProcessEvent_WaitForServicesConfirmedStateHandler
  \/ ProcessEvent_ConfirmedStateHandler
  \/ ProcessEvent_CancelledStateHandler

-----------------------------------------------------------------------------

ScheduleEvent ==
  /\ invoiceState = "IWS_WaitForPayment"
  /\ Len(invoiceQueue) = 0
  /\ UnchangedOrder
  /\ UnchangedService
  /\ Invoice!Enqueue("IE_TRefreshPaymentStatus")

-----------------------------------------------------------------------------

ModelInit ==
  /\ orderState = "OWS_New"
  /\ orderQueue = <<>>
  /\ Service!ModelInit
  /\ Invoice!ModelInit

ModelNext ==
  \/ ProcessEvent
  \/ ScheduleEvent
  \/ Service!ModelNext /\ UNCHANGED orderState /\ UnchangedInvoice
  \/ Invoice!ModelNext /\ UNCHANGED orderState /\ UnchangedService

ModelTypeInvariant ==
  /\ orderState \in AllStates
  /\ orderQueue \in Seq(AllEvents \union Service!AllProducedEvents \union Invoice!AllProducedEvents)
  /\ Service!ModelTypeInvariant
  /\ Invoice!ModelTypeInvariant

=============================================================================
\* Modification History
\* Last modified Fri Feb 08 15:29:08 MSK 2019 by sandello
\* Created Thu Jan 31 14:31:09 MSK 2019 by sandello
