-------------------------- MODULE TrustInvoiceModel -------------------------
(* TrustInvoiceModel specifies the automata for the Trust invoice.
 * Note that the model requires an external entity to enqueue events.
 * Otherwise, the model will deadlock.
 *)

(* Well-known TODOs:
 * - Missing specification for the Refund flow.
 * - Missing constraints on itinerary data synchronization.
 * - Failures are notoriously badly modelled here.
 *)

EXTENDS Naturals, Sequences

CONSTANT IsSharedRelay
ASSUME IsSharedRelay \in BOOLEAN

\* The invoice is modelled as a workflow instance.
VARIABLES state, queue, relay
\* The remote state of the payment.
VARIABLES remoteTrustState

vars == <<state, queue, relay, remoteTrustState>>

AllStates == {
  "IWS_New",
  "IWS_Created",
  "IWS_TrustOrdersCreated",
  "IWS_WaitForPayment",
  "IWS_Hold",
  "IWS_PaymentNotAuthorized",
  "IWS_Clear",
  "IWS_Cancelled",

  "IWS_WaitRefund",
  "IWS_PartRefund",
  "IWS_Refund",
  "IWS_Done",
  \* Failed state.
  "IWS_UnhandlerError"
}

AllConsumedEvents == {
  "IE_TPaymentInitialized",
  "IE_TCreateBasket",
  "IE_TPaymentStarted",
  "IE_TRefreshPaymentStatus",
  "IE_TPaymentCompleted",
  "IE_TPaymentCompletedWithError",
  "IE_TPaymentCreated",
  "IE_TPaymentClear",
  "IE_TPaymentUnhold"
}

AllProducedEvents == {
  "IE_MoneyAuthorized",
  "IE_MoneyNotAuthorized",
  \* Failed message.
  "IE_DeathLetter"
}

-----------------------------------------------------------------------------

T == INSTANCE TrustPaymentModel WITH state <- remoteTrustState

-----------------------------------------------------------------------------
(***************************************************************************)
(* Helper Actions                                                          *)
(***************************************************************************)

LOCAL Fail ==
  /\ state' = "IWS_UnhandlerError"
  /\ queue' = queue
  /\ relay' = Append(relay, "IE_DeathLetter")
  /\ UNCHANGED remoteTrustState

LOCAL Retry == UNCHANGED <<state, queue, relay>>

Enqueue(event) ==
  /\ queue' = Append(queue, event)
  /\ IF IsSharedRelay
     THEN UNCHANGED <<state, remoteTrustState>>
     ELSE UNCHANGED <<state, relay, remoteTrustState>>

-----------------------------------------------------------------------------
(***************************************************************************)
(* State Handlers                                                          *)
(***************************************************************************)

ProcessEvent_NewStateHandler ==
  /\ state = "IWS_New"
  /\ Len(queue) > 0
  /\ LET event == Head(queue) IN
     CASE event = "IE_TPaymentInitialized" ->
       \/ /\ T!CreateOrdersCall
          /\ state' = "IWS_TrustOrdersCreated"
          /\ queue' = Append(Tail(queue), "IE_TCreateBasket")
          /\ UNCHANGED relay
       \* \/ Fail TODO(sandello): Implement me.
       [] OTHER -> Fail
       
ProcessEvent_TrustOrdersCreatedStateHandler ==
  /\ state = "IWS_TrustOrdersCreated"
  /\ Len(queue) > 0
  /\ LET event == Head(queue) IN
     CASE event = "IE_TCreateBasket" ->
       \/ /\ T!CreateBasketCall
          /\ state' = "IWS_Created"
          /\ queue' = Append(Tail(queue), "IE_TPaymentStarted")
          /\ UNCHANGED relay 

ProcessEvent_CreatedStateHandler ==
  /\ state = "IWS_Created"
  /\ Len(queue) > 0
  /\ LET event == Head(queue) IN
     CASE event = "IE_TPaymentStarted" ->
       \/ /\ T!StartPaymentCall
          /\ state' = "IWS_WaitForPayment"
          /\ queue' = Tail(queue)
          /\ UNCHANGED relay
       \* \/ Fail TODO(sandello): Implement me.
       [] OTHER -> Fail

ProcessEvent_WaitForPaymentStateHandler ==
  /\ state = "IWS_WaitForPayment"
  /\ Len(queue) > 0
  /\ LET event == Head(queue) IN
     CASE event = "IE_TRefreshPaymentStatus" ->
       \/ /\ T!IsAuthorized /\ UNCHANGED remoteTrustState
          /\ state' = "IWS_Hold"
          /\ queue' = Tail(queue)
          /\ relay' = Append(relay, "IE_MoneyAuthorized")
       \/ /\ T!IsNotAuthorized /\ UNCHANGED remoteTrustState
          /\ state' = "IWS_PaymentNotAuthorized"
          /\ queue' = Tail(queue)
          /\ relay' = Append(relay, "IE_MoneyNotAuthorized")
       \* \/ Fail TODO(sandello): Implement me.
       [] OTHER -> Fail

ProcessEvent_HoldStateHandler ==
  /\ state = "IWS_Hold"
  /\ Len(queue) > 0
  /\ LET event == Head(queue) IN
     CASE event = "IE_TPaymentClear" ->
       \/ /\ T!ClearCall
          /\ state' = "IWS_Clear"
          /\ queue' = Tail(queue)
          /\ UNCHANGED relay
       \* \/ Fail TODO(sandello): Implement me.
       [] event = "IE_TPaymentUnhold" ->
       \/ /\ T!CancelCall
          /\ state' = "IWS_Cancelled"
          /\ queue' = Tail(queue)
          /\ UNCHANGED relay
       \* \/ Fail TODO(sandello): Implement me.
       [] OTHER -> Fail

ProcessEvent_PaymentNotAuthorizedStateHandler ==
  /\ state = "IWS_PaymentNotAuthorized"
  /\ Len(queue) > 0
  /\ LET event == Head(queue) IN
     CASE event = "..." -> Fail
       [] OTHER -> Fail

ProcessEvent_ClearStateHandler ==
  /\ state = "IWS_Clear"
  /\ Len(queue) > 0
  /\ LET event == Head(queue) IN
     CASE event = "..." -> Fail
       [] OTHER -> Fail

ProcessEvent_CancelledStateHandler ==
  /\ state = "IWS_Cancelled"
  /\ Len(queue) > 0
  /\ LET event == Head(queue) IN
     CASE event = "..." -> Fail
       [] OTHER -> Fail

ProcessEvent_Unimplemented ==
  /\ state \in {"IWS_WaitRefund", "IWS_PartRefund", "IWS_Refund", "IWS_Done"}
  /\ Len(queue) > 0
  /\ Fail

ProcessEvent ==
  \/ ProcessEvent_NewStateHandler
  \/ ProcessEvent_CreatedStateHandler
  \/ ProcessEvent_WaitForPaymentStateHandler
  \/ ProcessEvent_HoldStateHandler
  \/ ProcessEvent_PaymentNotAuthorizedStateHandler
  \/ ProcessEvent_ClearStateHandler
  \/ ProcessEvent_CancelledStateHandler
  \/ ProcessEvent_Unimplemented
  (*
  \/ ProcessEvent_WaitRefundStateHandler
  \/ ProcessEvent_PartRefundStateHandler
  \/ ProcessEvent_RefundStateHandler
  \/ ProcessEvent_DoneStateHandler
  *)

-----------------------------------------------------------------------------

ModelInit ==
  /\ state = "IWS_New"
  /\ queue = <<>>
  /\ relay = <<>>
  /\ T!ModelInit

ModelNext ==
  \/ ProcessEvent
  \/ T!BackgroundNext /\ UNCHANGED <<state, queue, relay>>

ModelTypeInvariant ==
  /\ state \in AllStates
  /\ queue \in Seq(AllConsumedEvents)
  \* XXX(sandello): When the model is used with a shared relay,
  \* this invariant does not hold. (Think of the relay being the order queue).
  /\ ~IsSharedRelay => relay \in Seq(AllProducedEvents)
  /\ T!ModelTypeInvariant

ModelSpec == ModelInit \/ [][ModelNext]_vars

=============================================================================
\* Modification History
\* Last modified Thu Feb 14 11:29:36 MSK 2019 by mbobrov
\* Last modified Fri Feb 08 15:30:15 MSK 2019 by sandello
\* Created Tue Jan 29 15:25:01 MSK 2019 by mbobrov
