------------------------------- MODULE travel -------------------------------

EXTENDS Integers, Sequences

\* Order Workflow States          
AllOWS == {"OWS_New",
          "OWS_UnhandledError",
          "OWS_RegisterServices",
          "OWS_ReserveServices",
          "OWS_WaitForServicesReserved",
          "OWS_WaitForPayment",
          "OWS_ConfirmServices",
          "OWS_WaitForServicesConfirmed",
          "OWS_Confirmed",
          "OWS_CancelServices",
          "OWS_WaitForServicesCancelled",
          "OWS_Cancelled"}

\* Order Events
AllOE == {"OE_TStartReservation",
          "OE_TRegisterServices",
          "OE_TReserveServices",
          "OE_TConfirmServices",
          "OE_TCancelServices",
          "OE_TWaitServicesConfirmed",
          "OE_TRefreshServices",
          "OE_TServiceStateChanged",
          "OE_TWaitForPayment",
          "OE_TMoneyAcquired",
          "OE_TMoneyAcquireErrorOccurred",
          "OE_TCancellationFinished",
          "OE_TUnhandledErrorOccurred",
          "OE_TOrderExpired"}

\* Service States
AllSS == {"SS_New",
         "SS_Registered",
         "SS_Reserved",
         "SS_Confirmed",
         "SS_Cancelled",
         "SS_Refunded"}

\* Invoice States
AllIS == {"IS_New",
          "IS_UnhandledError",
          "IS_WaitForPayment",
          "IS_Authorized",
          "IS_NotAuthorized",
          "IS_Cancelled",
          "IS_Cleared",
          "IS_WaitForRefund",
          "IS_PartiallyRefunded",
          "IS_FullyRefunded"}
          
REFRESH_ORDER_STATES == {"OWS_WaitForServicesReserved",
                         "OWS_WaitForServicesConfirmed",
                         "OWS_WaitForServicesCancelled"}          

VARIABLES orderWorkflowState, orderWorkflowQueue, serviceState, invoiceState 

\* For stuttering
vars == <<orderWorkflowState, orderWorkflowQueue, serviceState, invoiceState>>

CONSTANTS Nil

-----------------------------------------------------------------------------

TypeInvariant == /\ orderWorkflowState \in AllOWS
                 /\ orderWorkflowQueue \in Seq(AllOE)
                 /\ serviceState \in AllSS
                 /\ invoiceState \in AllIS \cup {Nil}

GRPC_Reserve ==
  /\ orderWorkflowState = "OWS_New"
  /\ orderWorkflowQueue' = Append(orderWorkflowQueue, "OE_TStartReservation")
  /\ UNCHANGED <<orderWorkflowState, serviceState, invoiceState>>

GRPC_StartPayment ==
  /\ orderWorkflowState = "OWS_WaitForPayment"
  /\ invoiceState = Nil
  /\ invoiceState' = "IS_New" 
  /\ UNCHANGED <<orderWorkflowState, orderWorkflowQueue, serviceState>>

(* Temporary state draining messages from the queue *)
FailOrder ==
  /\ orderWorkflowState' = "OWS_UnhandledError"
  /\ UNCHANGED <<orderWorkflowQueue, serviceState, invoiceState>>

DropMessage ==
  /\ orderWorkflowQueue' = Tail(orderWorkflowQueue)
  /\ UNCHANGED <<orderWorkflowState, serviceState, invoiceState>>

ProcessOrderEvent_NewStateHandler(msg) ==
  CASE msg = "OE_TStartReservation" ->
      /\ orderWorkflowState' = "OWS_RegisterServices"
      /\ orderWorkflowQueue' = Append(Tail(orderWorkflowQueue), "OE_TRegisterServices")
      /\ UNCHANGED <<serviceState, invoiceState>>
  [] OTHER -> FailOrder
  
ProcessOrderEvent_RegisterServicesStateHandler(msg) ==
  CASE msg = "OE_TRegisterServices" ->
      /\ orderWorkflowState' = "OWS_ReserveServices"
      /\ serviceState' = "SS_Registered"  
      /\ orderWorkflowQueue' = Append(Tail(orderWorkflowQueue), "OE_TReserveServices")
      /\ UNCHANGED <<invoiceState>>
    [] msg = "OE_TStartReservation" ->
      /\ DropMessage   
    [] OTHER -> FailOrder
  
ProcessOrderEvent_ReserveServicesStateHandler(msg) ==
  CASE msg = "OE_TReserveServices" ->
      /\ orderWorkflowState' = "OWS_WaitForServicesReserved"
      /\ orderWorkflowQueue' = Tail(orderWorkflowQueue)
      /\ UNCHANGED <<serviceState, invoiceState>>
    [] OTHER -> FailOrder
  
ProcessOrderEvent_WaitForServicesReserved(msg) ==
  CASE msg = "OE_TRefreshServices" ->
      /\ orderWorkflowState' = "OWS_WaitForPayment"
      /\ orderWorkflowQueue' = Tail(orderWorkflowQueue)
      /\ serviceState' = "SS_Reserved"
      /\ UNCHANGED <<invoiceState>>
    [] OTHER -> FailOrder

ProcessOrderEvent_WaitForPayment(msg) ==
  \* think about money balance
  CASE msg = "OE_TMoneyAcquired" ->
      /\ orderWorkflowState' = "OWS_ConfirmServices"
      /\ orderWorkflowQueue' = Append(Tail(orderWorkflowQueue), "OE_TConfirmServices")
      /\ UNCHANGED <<serviceState, invoiceState>>
    [] msg = "OE_TMoneyAcquireErrorOccurred" ->
      /\ orderWorkflowState' = "OWS_CancelServices"
      /\ orderWorkflowQueue' = Append(Tail(orderWorkflowQueue), "OE_TCancelServices")
      /\ UNCHANGED <<serviceState, invoiceState>>
    [] msg = "OE_TRefreshServices" -> DropMessage       
    [] OTHER -> FailOrder

ProcessOrderEvent_ConfirmServices(msg) ==
  CASE msg = "OE_TConfirmServices" ->
      /\ orderWorkflowState' = "OWS_WaitForServicesConfirmed"
      /\ orderWorkflowQueue' = Tail(orderWorkflowQueue)
      /\ UNCHANGED <<serviceState, invoiceState>>
    [] OTHER -> FailOrder

ProcessOrderEvent_WaitForServicesConfirmed(msg) ==
  CASE msg = "OE_TRefreshServices" ->
      /\ orderWorkflowState' = "OWS_Confirmed"
      /\ orderWorkflowQueue' = Tail(orderWorkflowQueue)
      /\ serviceState' = "SS_Confirmed"
      /\ UNCHANGED <<invoiceState>>
    [] OTHER -> FailOrder
    
ProcessOrderEvent_CancelServices(msg) ==
  CASE msg = "OE_TCancelServices" ->
      /\ orderWorkflowState' = "OWS_WaitForServicesCancelled"
      /\ orderWorkflowQueue' = Tail(orderWorkflowQueue)
      /\ UNCHANGED <<serviceState, invoiceState>>
    [] OTHER -> FailOrder
    
ProcessOrderEvent_WaitForServicesCancelled(msg) ==
  CASE msg = "OE_TRefreshServices" ->
      /\ orderWorkflowState' = "OWS_Cancelled"
      /\ orderWorkflowQueue' = Tail(orderWorkflowQueue)
      /\ serviceState' = "SS_Cancelled"
      /\ UNCHANGED <<invoiceState>>
    [] OTHER -> FailOrder
    
\* it's terminal for the moment    
ProcessOrderEvent_Confirmed(msg) ==
  DropMessage
  
ProcessOrderEvent_Cancelled(msg) ==
  DropMessage 
    
ScheduleOrderEvent ==
  /\ IF orderWorkflowState \in REFRESH_ORDER_STATES THEN
       orderWorkflowQueue' = Append(orderWorkflowQueue, "OE_TRefreshServices")
     ELSE 
       UNCHANGED <<orderWorkflowQueue>>
  /\ UNCHANGED <<orderWorkflowState, serviceState, invoiceState>>

ProcessOrderEvent ==
  /\ Len(orderWorkflowQueue) > 0
  /\ LET msg == Head(orderWorkflowQueue)
     IN CASE orderWorkflowState = "OWS_New" -> ProcessOrderEvent_NewStateHandler(msg)
        [] orderWorkflowState = "OWS_RegisterServices" -> ProcessOrderEvent_RegisterServicesStateHandler(msg)
        [] orderWorkflowState = "OWS_ReserveServices" -> ProcessOrderEvent_ReserveServicesStateHandler(msg)
        [] orderWorkflowState = "OWS_WaitForServicesReserved" -> ProcessOrderEvent_WaitForServicesReserved(msg)
        [] orderWorkflowState = "OWS_WaitForPayment" -> ProcessOrderEvent_WaitForPayment(msg)
        [] orderWorkflowState = "OWS_ConfirmServices" -> ProcessOrderEvent_ConfirmServices(msg)
        [] orderWorkflowState = "OWS_WaitForServicesConfirmed" -> ProcessOrderEvent_WaitForServicesConfirmed(msg)
        [] orderWorkflowState = "OWS_Confirmed" -> ProcessOrderEvent_Confirmed(msg)
        [] orderWorkflowState = "OWS_CancelServices" -> ProcessOrderEvent_CancelServices(msg)
        [] orderWorkflowState = "OWS_WaitForServicesCancelled" -> ProcessOrderEvent_WaitForServicesCancelled(msg)
        [] orderWorkflowState = "OWS_Cancelled" -> ProcessOrderEvent_Cancelled(msg)
        [] OTHER -> FailOrder

HackForTrustSpecSuccess ==
  /\ invoiceState = "IS_New"
  /\ invoiceState' = "IS_Authorized"
  /\ orderWorkflowQueue' = Append(orderWorkflowQueue, "OE_TMoneyAcquired")
  /\ UNCHANGED <<orderWorkflowState, serviceState>>
 
  
HackForTrustSpecFailure ==
  /\ invoiceState = "IS_New"
  /\ invoiceState' = "IS_NotAuthorized"
  /\ orderWorkflowQueue' = Append(orderWorkflowQueue, "OE_TMoneyAcquireErrorOccurred")
  /\ UNCHANGED <<orderWorkflowState, serviceState>>  

  
  
-----------------------------------------------------------------------------
Init == /\ orderWorkflowState = "OWS_New"
        /\ orderWorkflowQueue = <<>>
        /\ serviceState = "SS_New"
        /\ invoiceState = Nil

Next ==
  \/ GRPC_Reserve
  \/ GRPC_StartPayment
  \/ ScheduleOrderEvent
  \/ ProcessOrderEvent
  \/ HackForTrustSpecSuccess
  \/ HackForTrustSpecFailure
  
Fairness == WF_vars(Next)

Spec == Init /\ [][Next]_vars /\ Fairness

DidStartReservation == LET Test(e) == (e = "OE_TStartReservation") IN Len(SelectSeq(orderWorkflowQueue, Test)) > 0
PaymentHappened == invoiceState = "IS_Authorized"
PaymentFailed == invoiceState = "IS_NotAuthorized" 

IsReserved ==
  /\ orderWorkflowState = "OWS_WaitForPayment"
  /\ serviceState = "SS_Reserved"
  
IsConfirmed ==
  /\ orderWorkflowState = "OWS_Confirmed"
  /\ serviceState = "SS_Confirmed"
  
IsCancelled ==
  /\ orderWorkflowState = "OWS_Cancelled"
  /\ serviceState = "SS_Cancelled"

LivenessReserved == DidStartReservation ~> IsReserved
LivenessConfirmed == PaymentHappened ~> <>[]IsConfirmed
LivenessCancelled == PaymentFailed ~> <>[]IsCancelled 

Liveness == /\ LivenessReserved
            /\ LivenessCancelled
            /\ LivenessConfirmed 

THEOREM Spec => Liveness 


=============================================================================
\* Modification History
\* Last modified Mon Jan 28 15:02:26 MSK 2019 by mbobrov
\* Last modified Fri Jan 25 17:09:37 MSK 2019 by sandello
\* Created Thu Jan 24 15:49:33 MSK 2019 by sandello