package ru.yandex.travel.orders.stress;

import java.time.Duration;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import io.micrometer.core.instrument.Timer;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.travel.commons.metrics.MetricsUtils;

@Slf4j
public class ScenarioRunner implements AutoCloseable {

    private final Logger errorLogger = LoggerFactory.getLogger("OrderErrors");

    private final ScenarioService scenarioService;

    private final ExecutorService executor;

    private final ScheduledExecutorService scheduledExecutor;

    private final ConcurrentMap<Outcome, Timer> executeTimersSuccess;
    private final ConcurrentMap<Outcome, Timer> executeTimersFailure;

    private final Counter successfullRuns;

    private final Counter failedRuns;

    private final Duration period;

    public ScenarioRunner(ScenarioService scenarioService,
                          int maxRunning, Duration period) {
        this.scenarioService = scenarioService;
        this.period = period;
        this.executor = Executors.newFixedThreadPool(maxRunning,
                new ThreadFactoryBuilder().setNameFormat("ScenarioRunner-%s").setDaemon(true).build());
        scheduledExecutor = Executors.newSingleThreadScheduledExecutor(
                new ThreadFactoryBuilder().setNameFormat("Scheduler-%s").setDaemon(true).build());

        executeTimersSuccess = new ConcurrentHashMap<>();
        executeTimersFailure = new ConcurrentHashMap<>();

        for (Outcome outcome : Outcome.values()) {
            executeTimersSuccess.put(outcome, Timer.builder(outcome.toString() + ".success.executionMs")
                    .publishPercentileHistogram(true)
                    .serviceLevelObjectives(MetricsUtils.mediumDurationSla())
                    .publishPercentiles(MetricsUtils.higherPercentiles())
                    .register(Metrics.globalRegistry));
            executeTimersFailure.put(outcome, Timer.builder(outcome.toString() + ".failure.executionMs")
                    .publishPercentileHistogram(true)
                    .serviceLevelObjectives(MetricsUtils.mediumDurationSla())
                    .publishPercentiles(MetricsUtils.higherPercentiles())
                    .register(Metrics.globalRegistry)
            );
        }

        successfullRuns = Counter.builder("scenario.runs.success").register(Metrics.globalRegistry);
        failedRuns = Counter.builder("scenario.runs.failed").register(Metrics.globalRegistry);
    }

    public void start() {
        scheduledExecutor.scheduleAtFixedRate(this::scheduleScenarioExecution, 100, period.toMillis(),
                TimeUnit.MILLISECONDS);
    }

    @Override
    public void close() throws Exception {
        MoreExecutors.shutdownAndAwaitTermination(scheduledExecutor, 2, TimeUnit.SECONDS);
        MoreExecutors.shutdownAndAwaitTermination(executor, 60, TimeUnit.SECONDS);
    }

    private void scheduleScenarioExecution() {
        try {
            executor.submit(this::executeScenario);
        } catch (Exception e) {
            log.error("Unexpected error planning scenario execution", e);
        }
    }

    private void executeScenario() {
        Outcome outcome = generateOutcome();
        log.info("Executing scenario with outcome {}", outcome);
        long startTime = System.currentTimeMillis();
        try {
            scenarioService.execute(outcome);
            successfullRuns.increment();
            executeTimersSuccess.get(outcome).record(System.currentTimeMillis() - startTime, TimeUnit.MILLISECONDS);
        } catch (Exception e) {
            errorLogger.error("Exception running scenario with outcome {}", outcome, e);
            failedRuns.increment();
            executeTimersFailure.get(outcome).record(System.currentTimeMillis() - startTime, TimeUnit.MILLISECONDS);
        }
    }

    private Outcome generateOutcome() {
        double o = Math.random();
        if (o < 0.1d) {
            return Outcome.FAILURE_RESERVE;
        } else if (o < 0.2d) {
            return Outcome.SUCCESS_RESERVE_FAILURE_PAYMENT;
        } else if (o < 0.4d) {
            return Outcome.FAILURE_CONFIRM;
        } else {
            return Outcome.SUCCESS;
        }
    }
}
