package ru.yandex.travel.orders.stress;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.UUID;
import java.util.function.Predicate;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.base.Preconditions;
import com.google.common.io.ByteStreams;
import io.grpc.Context;

import ru.yandex.misc.ExceptionUtils;
import ru.yandex.travel.commons.proto.ECurrency;
import ru.yandex.travel.commons.proto.TJson;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.credentials.UserCredentialsBuilder;
import ru.yandex.travel.hotels.proto.EHotelConfirmationOutcome;
import ru.yandex.travel.hotels.proto.EHotelReservationOutcome;
import ru.yandex.travel.hotels.proto.THotelTestContext;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.commons.proto.EPaymentOutcome;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.commons.proto.TDelayInterval;
import ru.yandex.travel.orders.commons.proto.TPaymentTestContext;
import ru.yandex.travel.orders.proto.EHotelOrderAggregateState;
import ru.yandex.travel.orders.proto.EInvoiceType;
import ru.yandex.travel.orders.proto.OrderInterfaceV1Grpc;
import ru.yandex.travel.orders.proto.TCreateOrderReq;
import ru.yandex.travel.orders.proto.TCreateOrderRsp;
import ru.yandex.travel.orders.proto.TCreateServiceReq;
import ru.yandex.travel.orders.proto.TGetOrderAggregateStateReq;
import ru.yandex.travel.orders.proto.TGetOrderAggregateStateRsp;
import ru.yandex.travel.orders.proto.TGetOrderInfoReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoRsp;
import ru.yandex.travel.orders.proto.TReserveReq;
import ru.yandex.travel.orders.proto.TStartPaymentReq;
import ru.yandex.travel.orders.proto.TUserInfo;
import ru.yandex.travel.orders.stress.config.OrchestratorClientFactory;

@SuppressWarnings("ResultOfMethodCallIgnored")
public class ScenarioService {

    private static final String ROOM_ID = "2741001";

    private final ObjectMapper objectMapper;

    private final OrchestratorClientFactory orchestratorClientFactory;

    private final Duration reservationWait;

    private final Duration startPaymentWait;

    private final Duration confirmationWait;

    private final Duration pollingInterval;

    private final long minYandexUID;

    private final long maxYandexUID;

    private final UserCredentialsBuilder userCredentialsBuilder;

    public ScenarioService(OrchestratorClientFactory orchestratorClientFactory, Duration reservationWait,
                           Duration startPaymentWait, Duration confirmationWait, Duration pollingInterval,
                           long minYandexUID, long maxYandexUID) {
        this.orchestratorClientFactory = orchestratorClientFactory;
        this.reservationWait = reservationWait;
        this.startPaymentWait = startPaymentWait;
        this.confirmationWait = confirmationWait;
        this.pollingInterval = pollingInterval;
        this.minYandexUID = minYandexUID;
        this.maxYandexUID = maxYandexUID;
        this.objectMapper = new ObjectMapper();
        this.userCredentialsBuilder = new UserCredentialsBuilder();
    }

    public void execute(Outcome outcome) {
        long id = minYandexUID + (long) (Math.random() * (maxYandexUID - minYandexUID));
        UserCredentials credentials = userCredentialsBuilder.build(UUID.randomUUID().toString(), Long.toString(id),
                null, null, null, "127.0.0.1", false, false);
        Context.current().withValue(UserCredentials.KEY, credentials).run(
                () -> {
                    try {
                        executeInContext(outcome, id);
                    } catch (InterruptedException e) {
                        throw ExceptionUtils.throwException(e);
                    }
                }
        );
    }

    private void executeInContext(Outcome outcome, long id) throws InterruptedException {
        OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub client = orchestratorClientFactory.createBlockingStub(false);

        TCreateOrderReq createOrderReq = createOrderRequest(id, outcome);

        TCreateOrderRsp resp = client.createOrder(createOrderReq);

        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build(); // we'll

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());

        if (outcome == Outcome.FAILURE_RESERVE) {
            waitForPredicateOrTimeout(client, orderId,
                    rsp1 -> rsp1.getOrderAggregateState().getHotelOrderAggregateState() == EHotelOrderAggregateState.HOAG_CANCELLED,
                    reservationWait, "Order " + orderId + " must be in OS_CANCELLED state");
            return;
        } else {
            waitForPredicateOrTimeout(client, orderId,
                    rsp3 -> rsp3.getOrderAggregateState().getHotelOrderAggregateState() == EHotelOrderAggregateState.HOAG_RESERVED,
                    reservationWait, "Order " + orderId + " must be in OS_WAITING_PAYMENT state");
        }

        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setUseNewInvoiceModel(true)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_return_url").build()); // safely ignoring response, as we don't need payment url

        if (outcome == Outcome.SUCCESS_RESERVE_FAILURE_PAYMENT) {
            waitForPredicateOrTimeout(client, orderId,
                    rsp1 -> rsp1.getOrderAggregateState().getHotelOrderAggregateState() == EHotelOrderAggregateState.HOAG_PAYMENT_FAILED,
                    confirmationWait, "Order " + orderId + " must be in OS_CANCELLED state");
            return;
        } else {
            waitForPredicateOrTimeout(client, orderId,
                    rsp2 -> rsp2.getOrderAggregateState().getHotelOrderAggregateState() == EHotelOrderAggregateState.HOAG_AWAITS_PAYMENT,
                    startPaymentWait, "Invoice " + orderId + " must be in IS_WAIT_FOR_PAYMENT state");
        }

        if (outcome == Outcome.FAILURE_CONFIRM) {
            TGetOrderInfoRsp confirmedOrderResponse = waitForPredicateOrTimeout(client, orderId,
                    rsp1 -> rsp1.getOrderAggregateState().getHotelOrderAggregateState() == EHotelOrderAggregateState.HOAG_CANCELLED_WITH_REFUND,
                    confirmationWait, "Order " + orderId + " must be in OS_CANCELLED state");
        } else {
            TGetOrderInfoRsp confirmedOrderResponse = waitForPredicateOrTimeout(client, orderId,
                    rsp1 -> rsp1.getOrderAggregateState().getHotelOrderAggregateState() == EHotelOrderAggregateState.HOAG_CONFIRMED,
                    confirmationWait, "Order " + orderId + " must be in OS_CONFIRMED state");
        }
    }

    private TCreateOrderReq createOrderRequest(long yandexUID, Outcome outcome) {
        ObjectNode reservationJson = getOrderItemJsonTemplate();
        String json;
        try {
            json = objectMapper.writeValueAsString(reservationJson);
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }

        THotelTestContext.Builder htc = THotelTestContext.newBuilder();
        htc.setReservationCallDelayMin(2500);
        htc.setReservationCallDelayMax(15000);
        htc.setConfirmationCallDelayMin(2500);
        htc.setConfirmationCallDelayMax(15000);

        TPaymentTestContext.Builder ptc = TPaymentTestContext.newBuilder();
        ptc.setUserActionDelay(TDelayInterval.newBuilder().setDelayMin(20000).setDelayMax(60000).build());
        ptc.setGetBasketStatusDelay(TDelayInterval.newBuilder().setDelayMin(500).setDelayMax(1500).build());

        switch (outcome) {
            case SUCCESS:
                htc.setReservationOutcome(EHotelReservationOutcome.RO_SUCCESS);
                htc.setConfirmationOutcome(EHotelConfirmationOutcome.CO_SUCCESS);
                ptc.setPaymentOutcome(EPaymentOutcome.PO_SUCCESS);
                break;
            case SUCCESS_RESERVE_FAILURE_PAYMENT:
                htc.setReservationOutcome(EHotelReservationOutcome.RO_SUCCESS);
                ptc.setPaymentOutcome(EPaymentOutcome.PO_FAILURE);
                ptc.setPaymentFailureResponseCode("user_cancelled");
                break;
            case FAILURE_CONFIRM:
                htc.setReservationOutcome(EHotelReservationOutcome.RO_SUCCESS);
                htc.setConfirmationOutcome(EHotelConfirmationOutcome.CO_NOT_FOUND);
                break;
            case FAILURE_RESERVE:
                htc.setReservationOutcome(EHotelReservationOutcome.RO_SOLD_OUT);
                break;
        }

        return TCreateOrderReq.newBuilder()
                .setOrderType(EOrderType.OT_HOTEL_EXPEDIA)
                .setDeduplicationKey(UUID.randomUUID().toString())
                .addCreateServices(
                        TCreateServiceReq.newBuilder()
                                .setHotelTestContext(htc.setPriceAmount(
                                        outcome.getPrice().getNumberStripped().intValue()).build()
                                )
                                .setServiceType(EServiceType.PT_TRAVELLINE_HOTEL)
                                .setSourcePayload(TJson.newBuilder().setValue(json))
                )
                .setCurrency(ECurrency.C_RUB)
                .setPaymentTestContext(ptc)
                .setOwner(TUserInfo.newBuilder()
                        .setEmail("test" + yandexUID + "@test.com")
                        .setPhone("+79111111111")
                        .setYandexUid(String.valueOf(yandexUID)))
                .build();
    }

    private ObjectNode getOrderItemJsonTemplate() {
        try {
            return (ObjectNode) objectMapper.readTree(
                    readResource("hotel_reservation_template.json"));
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private TGetOrderInfoRsp waitForPredicateOrTimeout(OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub client,
                                                       String orderId,
                                                       Predicate<TGetOrderAggregateStateRsp> predicate,
                                                       Duration timeout,
                                                       String description) throws InterruptedException {
        TGetOrderAggregateStateRsp rsp =
                client.getOrderAggregateState(TGetOrderAggregateStateReq.newBuilder().setOrderId(orderId).build());
        long startWaitingTime = System.currentTimeMillis();
        while ((System.currentTimeMillis() - startWaitingTime) < timeout.toMillis() && !predicate.test(rsp)) {
            Thread.sleep(pollingInterval.toMillis());
            rsp = client.getOrderAggregateState(TGetOrderAggregateStateReq.newBuilder().setOrderId(orderId).build());
        }
        if (!predicate.test(rsp)) {
            throw new RuntimeException(description + " current state " + rsp.getOrderAggregateState().getHotelOrderAggregateState());
        } else {
            return client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).build());
        }
    }

    private String readResource(String relativePath) {
        InputStream is = ScenarioService.class.getClassLoader().getResourceAsStream(relativePath);
        Preconditions.checkNotNull(is, "No such resource: %s", relativePath);
        try {
            byte[] bytes = ByteStreams.toByteArray(is);
            return new String(bytes, StandardCharsets.UTF_8);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }


}
