package ru.yandex.travel.orders.stress.config;

import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import io.grpc.Channel;
import io.grpc.ClientInterceptor;
import io.grpc.ManagedChannelBuilder;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.grpc.interceptors.DefaultTimeoutClientInterceptor;
import ru.yandex.travel.grpc.interceptors.LoggingClientInterceptor;
import ru.yandex.travel.grpc.interceptors.UserCredentialsClientInterceptor;
import ru.yandex.travel.orders.client.ChannelState;
import ru.yandex.travel.orders.client.ChannelSupplier;
import ru.yandex.travel.orders.client.GrpcChannelSupplierFactory;
import ru.yandex.travel.orders.client.HAGrpcChannelFactory;
import ru.yandex.travel.orders.client.LabeledChannel;
import ru.yandex.travel.orders.proto.EServerState;
import ru.yandex.travel.orders.proto.HADiscoveryInterfaceV1Grpc;
import ru.yandex.travel.orders.proto.OrderInterfaceV1Grpc;
import ru.yandex.travel.orders.proto.TPingRpcReq;

@Configuration
@RequiredArgsConstructor
@EnableConfigurationProperties(OrchestratorConfigurationProperties.class)
@Slf4j
public class OrchestratorClientConfiguration {
    private final OrchestratorConfigurationProperties properties;

    @Bean
    public OrchestratorClientFactory orchestratorClientFactory(HAGrpcChannelFactory haGrpcChannelFactory) {
        return requireMasterChannel -> {
            Channel channel = requireMasterChannel ? haGrpcChannelFactory.getMasterChannel() :
                    haGrpcChannelFactory.getRandomChannel();
            return OrderInterfaceV1Grpc.newBlockingStub(channel);
        };
    }

    @Bean
    public ChannelSupplier getChannelSupplier() {
        GrpcChannelSupplierFactory grpcChannelSupplierFactory = new GrpcChannelSupplierFactory(properties);

        return grpcChannelSupplierFactory.getChannelSupplier();
    }

    @Bean
    public HAGrpcChannelFactory haGrpcChannelFactory(ChannelSupplier channelSupplier) {

        HAGrpcChannelFactory.Builder builder = HAGrpcChannelFactory.Builder.newBuilder();
        return builder.withPingProducer(
                channel -> FutureUtils.buildCompletableFuture(HADiscoveryInterfaceV1Grpc
                        .newFutureStub(channel)
                        .ping(TPingRpcReq.newBuilder().build())
                ).thenApply((rsp) -> rsp.getState() == EServerState.SS_MASTER ? ChannelState.READY_MASTER :
                        ChannelState.READY))
                .withFailureDetectorProperties(properties.getFailureDetection())
                .withChannelSupplier(channelSupplier)
                .withChannelBuilder(this::createChannel)
                .build();
    }

    @SneakyThrows
    private LabeledChannel createChannel(String target) {
        String clientFqdn = Objects.requireNonNull(InetAddress.getLocalHost().getCanonicalHostName());
        LoggingClientInterceptor loggingClientInterceptor = new LoggingClientInterceptor(
                clientFqdn, target, Set.of(HADiscoveryInterfaceV1Grpc.getPingMethod().getFullMethodName())
        );
        DefaultTimeoutClientInterceptor defaultTimeoutClientInterceptor = new DefaultTimeoutClientInterceptor(
                properties.getTimeout()
        );
        List<ClientInterceptor> interceptors = new ArrayList<>(4);
        interceptors.addAll(Arrays.asList(loggingClientInterceptor, defaultTimeoutClientInterceptor));
        if (properties.isForwardCredentials()) {
            interceptors.add(new UserCredentialsClientInterceptor());
        }
        return new LabeledChannel(target,
                ManagedChannelBuilder
                        .forTarget(target)
                        .intercept(interceptors)
                        .usePlaintext()
                        .build());
    }
}
