from __future__ import print_function

import argparse
import datetime
import logging
import pprint
import socket
import uuid
import sys

import grpc

import travel.orders.proto.commons_pb2 as orders_commons_pb2
import travel.orders.proto.services.orders.admin.orders_admin_pb2 as orders_admin_pb2
import travel.orders.proto.services.orders.admin.orders_admin_pb2_grpc as orders_admin_pb2_grpc

# import travel.orders.proto.services.orders.orders_admin_pb2_grpc as orders_admin_pb2_grpc


def create_grpc_channel(args):
    endpoint = "%s:%s" % (args.host, args.port)
    logging.info("creating a channel to '%s'", endpoint)
    return grpc.insecure_channel(endpoint)


def create_grpc_call_data(timeout=1000):
    metadata = (
        ("ya-grpc-call-id", str(uuid.uuid1())),
        ("ya-grpc-started-at", datetime.datetime.utcnow().isoformat() + "Z"),
        ("ya-grpc-fqdn", socket.getfqdn())
    )
    return {"metadata": metadata, "timeout": timeout}


def handle_getOrder(args):
    stub = orders_admin_pb2_grpc.OrdersAdminInterfaceV1Stub(create_grpc_channel(args))
    req = orders_admin_pb2.TGetOrderReq(
        OrderId=args.id,
        PrettyId=args.pretty
    )
    # if args.private:
    #   snippet = req.Snippet.append(orders_commons_pb2.S_PRIVATE_INFO)
    grpc_call_data = create_grpc_call_data()
    logging.info("-> REQ %s\n%r\n%s\n", req.__class__, req, pprint.pformat(grpc_call_data))
    rsp = stub.GetOrder(req, **grpc_call_data)
    logging.info("<- RSP %s\n%r", rsp.__class__, rsp)


def handle_listOrders(args):
    stub = orders_admin_pb2_grpc.OrdersAdminInterfaceV1Stub(create_grpc_channel(args))
    req = orders_admin_pb2.TListOrdersReq(
        Page=orders_commons_pb2.TOffsetPage(
            Offset=args.offset,
            Limit=args.limit
        ),
        OrderIdFilter=args.id,
        PrettyIdFilter=args.pretty,
        EmailFilter=args.email,
        PhoneFilter=args.phone
    )
    grpc_call_data = create_grpc_call_data()
    logging.info("-> REQ %s\n%r\n%s\n", req.__class__, req, pprint.pformat(grpc_call_data))
    rsp = stub.ListOrders(req, **grpc_call_data)
    logging.info("<- RSP %s\n%r", rsp.__class__, rsp)


def handle_getWorkflow(args):
    stub = orders_admin_pb2_grpc.OrdersAdminInterfaceV1Stub(create_grpc_channel(args))
    req = orders_admin_pb2.TGetWorkflowReq(
        WorkflowId=args.wf,
        EntityId=args.entity
    )
    grpc_call_data = create_grpc_call_data()
    logging.info("-> REQ %s\n%r\n%s\n", req.__class__, req, pprint.pformat(grpc_call_data))
    rsp = stub.GetWorkflow(req, **grpc_call_data)
    logging.info("<- RSP %s\n%r", rsp.__class__, rsp)


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("--verbose", default=False, type=bool)
    parser.add_argument("--host", default="localhost")
    parser.add_argument("--port", default=30858, type=int)
    subparsers = parser.add_subparsers()

    getOrder_parser = subparsers.add_parser("getOrder")
    getOrder_parser.add_argument("--id")
    getOrder_parser.add_argument("--pretty")
    getOrder_parser.add_argument("--private", type=bool, default=False)
    getOrder_parser.set_defaults(func=handle_getOrder)

    listOrders_parser = subparsers.add_parser("listOrders")
    listOrders_parser.add_argument("--offset", type=int, default=0)
    listOrders_parser.add_argument("--limit", type=int, default=100)
    listOrders_parser.add_argument("--private", type=bool, default=False)
    listOrders_parser.add_argument("--id")
    listOrders_parser.add_argument("--pretty")
    listOrders_parser.add_argument("--email")
    listOrders_parser.add_argument("--phone")
    listOrders_parser.set_defaults(func=handle_listOrders)

    getWorkflow_parser = subparsers.add_parser("getWorkflow")
    getWorkflow_parser.add_argument("--wf")
    getWorkflow_parser.add_argument("--entity")
    getWorkflow_parser.set_defaults(func=handle_getWorkflow)

    args = parser.parse_args()
    logging.basicConfig(level=(logging.DEBUG if args.verbose else logging.INFO),
                        format="%(asctime)-15s | %(levelname)s | %(message)s",
                        stream=sys.stdout)

    args.func(args)


if __name__ == "__main__":
    main()
