from __future__ import print_function

import argparse
import datetime
import logging
import socket
import sys
import uuid
import json
import collections

import grpc

import travel.proto.commons_pb2 as commons_pb2
import travel.orders.proto.commons_pb2 as orders_commons_pb2
import travel.orders.proto.services.orders.orders_pb2 as orders_pb2
import travel.orders.proto.services.orders.orders_pb2_grpc as orders_pb2_grpc
from travel.orders.tools.library.tvm import get_tvm_service_ticket


class _GenericClientInterceptor(grpc.UnaryUnaryClientInterceptor, grpc.UnaryStreamClientInterceptor, grpc.StreamUnaryClientInterceptor, grpc.StreamStreamClientInterceptor):
    def __init__(self, interceptor_function):
        self._fn = interceptor_function

    def intercept_unary_unary(self, continuation, client_call_details, request):
        new_details, new_request_iterator, postprocess = self._fn(
            client_call_details, iter((request,)), False, False)
        response = continuation(new_details, next(new_request_iterator))
        return postprocess(response) if postprocess else response

    def intercept_unary_stream(self, continuation, client_call_details, request):
        new_details, new_request_iterator, postprocess = self._fn(
            client_call_details, iter((request,)), False, True)
        response_it = continuation(new_details, next(new_request_iterator))
        return postprocess(response_it) if postprocess else response_it

    def intercept_stream_unary(self, continuation, client_call_details, request_iterator):
        new_details, new_request_iterator, postprocess = self._fn(
            client_call_details, request_iterator, True, False)
        response = continuation(new_details, new_request_iterator)
        return postprocess(response) if postprocess else response

    def intercept_stream_stream(self, continuation, client_call_details, request_iterator):
        new_details, new_request_iterator, postprocess = self._fn(
            client_call_details, request_iterator, True, True)
        response_it = continuation(new_details, new_request_iterator)
        return postprocess(response_it) if postprocess else response_it


class _ClientCallDetails(collections.namedtuple('_ClientCallDetails', ('method', 'timeout', 'metadata', 'credentials')), grpc.ClientCallDetails):
    pass


def create_grpc_channel(args):
    ticket = get_tvm_service_ticket(
        tvm_client_id=args.tvm_client_id,
        tvm_service_id=args.tvm_service_id,
        tvm_client_secret=args.tvm_client_secret,
        skip_authentication=args.skip_authentication
    )

    def ya_grpc_interceptor(client_call_details, request_iterator,
                            request_streaming, response_streaming):
        ya_call_id = str(uuid.uuid1())
        ya_started_at = datetime.datetime.utcnow().isoformat() + "Z"
        ya_fqdn = socket.getfqdn()

        metadata = {}
        if client_call_details.metadata is not None:
            metadata = dict(client_call_details.metadata)
        metadata["ya-grpc-call-id"] = ya_call_id
        metadata["ya-grpc-started-at"] = ya_started_at
        metadata["ya-grpc-fqdn"] = ya_fqdn
        metadata["x-ya-service-ticket"] = ticket
        if args.yandex_uid:
            metadata["x-ya-yandexuid"] = args.yandex_uid
        if args.session_key:
            metadata["x-ya-session-key"] = args.session_key
        if args.passport_id:
            metadata["x-ya-passportid"] = args.passport_id
        if args.user_ticket:
            metadata["x-ya-user-ticket"] = args.user_ticket
        if args.login:
            metadata["x-ya-login"] = args.login

        timeout = args.timeout
        if client_call_details.timeout is not None:
            timeout = client_call_details.timeout

        client_call_details = _ClientCallDetails(
            client_call_details.method,
            timeout,
            list(metadata.items()),
            client_call_details.credentials)

        postprocess = None

        assert not request_streaming
        assert not response_streaming

        request = next(request_iterator)
        logging.info("<- REQ %s (CallId: %s)", client_call_details.method, ya_call_id)
        logging.debug("%s\n%r", request.__class__, request)

        def complete(rendezvous):
            logging.info("-> RSP %s (CallId: %s, Code: %s, Cancelled: %s)",
                         client_call_details.method, ya_call_id,
                         rendezvous.code(), rendezvous.cancelled())

            metadata = {}
            for item in rendezvous.initial_metadata():
                metadata[item.key] = item.value
            for item in rendezvous.trailing_metadata():
                metadata[item.key] = item.value
            for key, value in sorted(metadata.items()):
                logging.debug("Meta: [%s] => %r", key, value)

            if not rendezvous.cancelled():
                response = rendezvous.result()
                logging.debug("%s\n%r", response.__class__, response)

        def postprocess(rendezvous):
            cb = lambda: complete(rendezvous)
            if not rendezvous.add_callback(cb):
                cb()
            return rendezvous

        return client_call_details, iter((request,)), postprocess

    endpoint = "%s:%s" % (args.host, args.port)
    logging.info("Creating a channel to '%s'", endpoint)
    channel = grpc.insecure_channel(endpoint)
    channel = grpc.intercept_channel(channel, _GenericClientInterceptor(ya_grpc_interceptor))
    return channel


def handle_createOrder(args):
    provider_mapping = {
        "expedia": orders_pb2.PT_EXPEDIA_HOTEL,
        "travelline": orders_pb2.PT_TRAVELLINE_HOTEL,
        "train": orders_pb2.PT_TRAIN,
        "flight": orders_pb2.PT_FLIGHT
    }

    owner = orders_pb2.TUserInfo(
        Login=args.owner_login,
        YandexUid=args.owner_yandex_uid,
        PassportId=args.owner_passport_id,
        Email=args.owner_email,
        Phone=args.owner_phone,
        Ip=args.owner_ip)
    with open(args.payload) as handle:
        payload = commons_pb2.TJson(Value=handle.read())
    create_service_req = orders_pb2.TCreateOrderReq.TCreateServiceReq(
        SourcePayload=payload,
        ServiceType=provider_mapping.get(args.provider))
    fx_rate = orders_pb2.TCreateOrderReq.TFxRate(
        currency=commons_pb2.C_USD,
        precision=2,
        value=6600)
    create_order_req = orders_pb2.TCreateOrderReq(
        OrderType=orders_commons_pb2.OT_HOTEL_EXPEDIA,
        Owner=owner,
        DeduplicationKey=str(uuid.uuid4()),
        Currency=commons_pb2.C_RUB,
        FxRate=[fx_rate],
        CreateServices=[create_service_req]
    )

    with create_grpc_channel(args) as channel:
        stub = orders_pb2_grpc.OrderInterfaceV1Stub(channel)
        rsp = stub.CreateOrder(create_order_req)
        service_ids = []
        for service in rsp.NewOrder.Service:
            service_ids.append(service.ServiceId)
        logging.info("Created order [%s] with order items [%s]", rsp.NewOrder.OrderId, ", ".join(service_ids))


def handle_getOrderInfo(args):
    req = orders_pb2.TGetOrderInfoReq(OrderId=args.order_id)
    with create_grpc_channel(args) as channel:
        stub = orders_pb2_grpc.OrderInterfaceV1Stub(channel)
        rsp = stub.GetOrderInfo(req)
        logging.info("Received %s\n%r", rsp.__class__, rsp)

        if args.payload:
            for srv in rsp.Result.Service:
                payload = srv.ServiceInfo.Payload.Value
                logging.info("Payload")
                logging.info(json.dumps(json.loads(payload), indent=4))


def handle_startPayment(args):
    req = orders_pb2.TStartPaymentReq(OrderId=args.order_id, ReturnUrl=args.return_url, InvoiceType=orders_pb2.IT_TRUST)
    with create_grpc_channel(args):
        stub = orders_pb2_grpc.OrderInterfaceV1Stub(create_grpc_channel(args))
        rsp = stub.StartPayment(req)
        logging.info("Received %s\n%r", rsp.__class__, rsp)


def handle_startReservation(args):
    req = orders_pb2.TReserveReq(
        OrderId=args.order_id
    )
    with create_grpc_channel(args) as channel:
        stub = orders_pb2_grpc.OrderInterfaceV1Stub(channel)
        rsp = stub.Reserve(req)
        logging.info("Received %s\n%r", rsp.__class__, rsp)


def handle_getExchangeRate(args):
    if args.revrate == "true":
        req = orders_pb2.TGetExchangeRateReq(FromCurrency=commons_pb2.C_RUB, ToCurrency=commons_pb2.C_USD)
    else:
        req = orders_pb2.TGetExchangeRateReq(FromCurrency=commons_pb2.C_USD, ToCurrency=commons_pb2.C_RUB)
    with create_grpc_channel(args):
        stub = orders_pb2_grpc.ExchangeRateInterfaceV1Stub(create_grpc_channel(args))
        logging.info("-> REQ %s\n%r\n", req.__class__, req)
        rsp = stub.GetExchangeRate(req)
        logging.info("<- RSP %s\n%r", rsp.__class__, rsp)


def handle_itemid_request(args, request_class, request_method):
    req = request_class(OrderItemId=args.id)
    request_method(req)


def handle_registerItem(args):
    with create_grpc_channel(args) as channel:
        stub = orders_pb2_grpc.OrderItemDevInterfaceDoNotUseStub(channel)
        handle_itemid_request(args, orders_pb2.TRegisterOrderItemReq, stub.RegisterOrderItem)


def handle_reserveItem(args):
    with create_grpc_channel(args) as channel:
        stub = orders_pb2_grpc.OrderItemDevInterfaceDoNotUseStub(channel)
        handle_itemid_request(args, orders_pb2.TReserveOrderItemReq, stub.ReserveOrderItem)


def handle_cancelItem(args):
    with create_grpc_channel(args) as channel:
        stub = orders_pb2_grpc.OrderItemDevInterfaceDoNotUseStub(channel)
        handle_itemid_request(args, orders_pb2.TCancelOrderItemReq, stub.CancelOrderItem)


def handle_confirmItem(args):
    with create_grpc_channel(args) as channel:
        stub = orders_pb2_grpc.OrderItemDevInterfaceDoNotUseStub(channel)
        handle_itemid_request(args, orders_pb2.TConfirmOrderItemReq, stub.ConfirmOrderItem)


def handle_refundItem(args):
    with create_grpc_channel(args) as channel:
        stub = orders_pb2_grpc.OrderItemDevInterfaceDoNotUseStub(channel)
        handle_itemid_request(args, orders_pb2.TRefundOrderItemReq, stub.RefundOrderItem)


def handle_getItemState(args):
    with create_grpc_channel(args) as channel:
        stub = orders_pb2_grpc.OrderItemDevInterfaceDoNotUseStub(channel)
        rsp = stub.GetOrderItemWorkflowState(orders_pb2.TGetOrderItemWorkflowStateReq(OrderItemId=args.id))
        logging.info("Item state is " + rsp.State)


def handle_createInvoice(args):
    req = orders_pb2.TDevCreateInvoiceReq(
        ReturnUrl=args.return_url,
        Phone=args.phone,
        Email=args.email,
        PassportId=args.passport_id,
        UserIp=args.user_ip,
        Price=commons_pb2.TPrice(
            Currency=commons_pb2.C_RUB,
            Precision=2,
            Amount=int(args.price)
        )
    )
    with create_grpc_channel(args) as channel:
        stub = orders_pb2_grpc.InvoiceDevInterfaceDoNotUseStub(channel)
        rsp = stub.CreateInvoice(req)
        logging.info("Invoice id: {}".format(rsp.InvoiceId))


def handle_startInvoicePayment(args):
    req = orders_pb2.TDevStartPaymentReq(
        InvoiceId=args.id
    )
    with create_grpc_channel(args) as channel:
        stub = orders_pb2_grpc.InvoiceDevInterfaceDoNotUseStub(channel)
        stub.StartPayment(req)
        logging.info("Payment started")


def handle_getInvoiceInfo(args):
    req = orders_pb2.TDevGetInvoiceInfoReq(
        InvoiceId=args.id
    )
    with create_grpc_channel(args) as channel:
        stub = orders_pb2_grpc.InvoiceDevInterfaceDoNotUseStub(channel)
        rsp = stub.GetInvoiceInfo(req)
        logging.info("Invoice info: {}".format(rsp))


def handle_refreshInvoiceInfo(args):
    req = orders_pb2.TDevRefreshInvoiceInfoReq(
        InvoiceId=args.id
    )
    with create_grpc_channel(args) as channel:
        stub = orders_pb2_grpc.InvoiceDevInterfaceDoNotUseStub(channel)
        stub.RefreshInvoiceInfo(req)
        logging.info("Invoice info refreshed")


def handle_createTicket(args):
    req = orders_pb2.TDevCreateTicketReq(
        Title=args.title,
        Description=args.descr,
        OrderId=args.order_id
    )
    with create_grpc_channel(args) as channel:
        stub = orders_pb2_grpc.TicketDevInterfaceDoNotUseStub(channel)
        rsp = stub.CreateTicket(req)
        logging.info("Ticket Created: {}".format(rsp))


def handle_getTicketInfo(args):
    req = orders_pb2.TDevGetTicketInfoReq(
        TicketId=args.id
    )
    with create_grpc_channel(args) as channel:
        stub = orders_pb2_grpc.TicketDevInterfaceDoNotUseStub(channel)
        rsp = stub.GetTicketInfo(req)
        logging.info("Ticket info: {}".format(rsp))


def handle_listOrders(args):
    req = orders_pb2.TListOrdersReq(
        Page=orders_commons_pb2.TPage(
            Num=args.page,
            Size=args.size
        )
    )
    with create_grpc_channel(args) as channel:
        stub = orders_pb2_grpc.OrderInterfaceV1Stub(channel)
        rsp = stub.ListOrders(req)
        if args.payloads:
            for order in rsp.OrderInfo:
                logging.info("Order " + order.OrderId)
                for srv in order.Service:
                    payload = srv.ServiceInfo.Payload.Value
                    logging.info("Payload")
                    logging.info(json.dumps(json.loads(payload), indent=4))
        else:
            rsp_test = str(rsp)
            lines = rsp_test.split('\n')
            res = []
            i = 0
            while i < len(lines):
                if lines[i].endswith("Payload {"):
                    i += 2
                    continue
                else:
                    res.append(lines[i])
                i += 1
            filtered_text = '\n'.join(res)
            logging.info("Orders List: {}".format(filtered_text))


def handle_authorizeForOrder(args):
    req = orders_pb2.TAuthorizeForOrderReq(
        OrderId=args.order_id,
        Secret=args.secret
    )
    with create_grpc_channel(args) as channel:
        stub = orders_pb2_grpc.OrderInterfaceV1Stub(channel)
        rsp = stub.AuthorizeForOrder(req)
        logging.info("Received %s\n%r", rsp.__class__, rsp)


def handle_crashWorkflow(args):
    req = orders_pb2.TDevCrashWorkflowReq(
        OrderId=args.id,
        WorkflowId=args.wfId
    )
    with create_grpc_channel(args) as channel:
        stub = orders_pb2_grpc.TicketDevInterfaceDoNotUseStub(channel)
        rsp = stub.CrashWorkflow(req)
        logging.info("Workflow crashed: {}".format(rsp))


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("-v", "--verbose", action="store_true", default=False)
    parser.add_argument("--host", default="localhost")
    parser.add_argument("--port", default=30858)
    parser.add_argument("--timeout", default=3, type=int)
    parser.add_argument("--tvm-client-id", default=2010758, type=int)
    parser.add_argument("--tvm-service-id", default=2002740, type=int)
    parser.add_argument("--tvm-client-secret")
    parser.add_argument("--skip-authentication", action='store_true', default=False)
    parser.add_argument("--yandex-uid")
    parser.add_argument("--session-key")
    parser.add_argument("--user-ticket")
    parser.add_argument("--passport-id")
    parser.add_argument("--login")
    subparsers = parser.add_subparsers()

    createOrder_parser = subparsers.add_parser("createOrder")
    createOrder_parser.set_defaults(func=handle_createOrder)
    createOrder_parser.add_argument("--owner_login", default="john_smith")
    createOrder_parser.add_argument("--owner_yandex_uid", default="100500")
    createOrder_parser.add_argument("--owner_passport_id", default="")
    createOrder_parser.add_argument("--owner_email", default="john.smith@mail.ru")
    createOrder_parser.add_argument("--owner_phone", default="+79261111111")
    createOrder_parser.add_argument("--owner_ip", default="127.0.0.1")
    createOrder_parser.add_argument("--payload", default="payload.json")
    createOrder_parser.add_argument("--provider", default="expedia")

    getOrderInfo_parser = subparsers.add_parser("getOrderInfo")
    getOrderInfo_parser.set_defaults(func=handle_getOrderInfo)
    getOrderInfo_parser.add_argument("order_id")
    getOrderInfo_parser.add_argument("--payload", action="store_true", default=False)

    startPayment_parser = subparsers.add_parser("startPayment")
    startPayment_parser.set_defaults(func=handle_startPayment)
    startPayment_parser.add_argument("order_id")
    startPayment_parser.add_argument("--return_url", default="http://127.0.0.1/payment")

    reserve_parser = subparsers.add_parser("startReservation")
    reserve_parser.set_defaults(func=handle_startReservation)
    reserve_parser.add_argument("order_id")

    exchangeRate_parser = subparsers.add_parser("getExchangeRate")
    exchangeRate_parser.set_defaults(func=handle_getExchangeRate)
    exchangeRate_parser.add_argument("--revrate", default="false")

    register_item_parser = subparsers.add_parser("registerItem")
    register_item_parser.add_argument("id")
    register_item_parser.set_defaults(func=handle_registerItem)

    reserve_item_parser = subparsers.add_parser("reserveItem")
    reserve_item_parser.add_argument("id")
    reserve_item_parser.set_defaults(func=handle_reserveItem)

    cancel_item_parser = subparsers.add_parser("cancelItem")
    cancel_item_parser.add_argument("id")
    cancel_item_parser.set_defaults(func=handle_cancelItem)

    confirm_item_parser = subparsers.add_parser("confirmItem")
    confirm_item_parser.add_argument("id")
    confirm_item_parser.set_defaults(func=handle_confirmItem)

    refund_item_parser = subparsers.add_parser("refundItem")
    refund_item_parser.add_argument("id")
    refund_item_parser.set_defaults(func=handle_refundItem)

    get_item_parser = subparsers.add_parser("getItemState")
    get_item_parser.add_argument("id")
    get_item_parser.set_defaults(func=handle_getItemState)

    create_invcoie_parser = subparsers.add_parser("createInvoice")
    create_invcoie_parser.add_argument("--return_url", default="http://travel.yandex.ru")
    create_invcoie_parser.add_argument("--phone", default="+79261111111")
    create_invcoie_parser.add_argument("--email", default="test@yandex-team.ru")
    create_invcoie_parser.add_argument("--passport_id", default="42")
    create_invcoie_parser.add_argument("--user_ip", default="127.0.0.1")
    create_invcoie_parser.add_argument("--price", default="1000")
    create_invcoie_parser.set_defaults(func=handle_createInvoice)

    start_invoice_payment_parser = subparsers.add_parser("startInvoicePayment")
    start_invoice_payment_parser.add_argument("id")
    start_invoice_payment_parser.set_defaults(func=handle_startInvoicePayment)

    get_invoice_info_parser = subparsers.add_parser("getInvoiceInfo")
    get_invoice_info_parser.add_argument("id")
    get_invoice_info_parser.set_defaults(func=handle_getInvoiceInfo)

    refresh_invoice_info_parser = subparsers.add_parser("refreshInvoiceInfo")
    refresh_invoice_info_parser.add_argument("id")
    refresh_invoice_info_parser.set_defaults(func=handle_refreshInvoiceInfo)

    create_ticket_parser = subparsers.add_parser("createTicket")
    create_ticket_parser.add_argument("--title", default="issue happened")
    create_ticket_parser.add_argument("--descr", default="test issue description")
    create_ticket_parser.add_argument("--order_id")
    create_ticket_parser.set_defaults(func=handle_createTicket)

    get_ticket_info_parser = subparsers.add_parser("getTicketInfo")
    get_ticket_info_parser.add_argument("--id")
    get_ticket_info_parser.set_defaults(func=handle_getTicketInfo)

    list_orders_parser = subparsers.add_parser("listOrders")
    list_orders_parser.add_argument("--page", type=int, default=0)
    list_orders_parser.add_argument("--size", type=int, default=10)
    list_orders_parser.add_argument("--payloads", action="store_true", default=False)
    list_orders_parser.set_defaults(func=handle_listOrders)

    authorize_order_parser = subparsers.add_parser("authorizeForOrder")
    authorize_order_parser.add_argument("order_id")
    authorize_order_parser.add_argument("--secret")
    authorize_order_parser.set_defaults(func=handle_authorizeForOrder)

    crash_workflow_parser = subparsers.add_parser("crashWorkflow")
    crash_workflow_parser.add_argument("--id")
    crash_workflow_parser.add_argument("--wfId")
    crash_workflow_parser.set_defaults(func=handle_crashWorkflow)

    args = parser.parse_args()
    logging.basicConfig(level=(logging.DEBUG if args.verbose else logging.INFO),
                        format="%(asctime)-15s | %(levelname)s | %(message)s",
                        stream=sys.stdout)

    args.func(args)


if __name__ == "__main__":
    main()
