# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import json
import logging
import uuid

import requests

from travel.orders.tools.library.tvm import get_tvm_service_ticket


class LastDataStorage(object):
    def __init__(self, file_path="last.json"):
        self.file_path = file_path

    def save_last_data(self, response):
        try:
            order_uid = response['id']
            with open(self.file_path, 'w') as f:
                json.dump({"order_uid": order_uid}, f)
        except Exception:
            logging.exception("Can't save last data %s to %s", response, self.file_path)

    def load_last_data(self):
        with open(self.file_path) as f:
            return json.load(f)


class OrchAPI(object):
    def __init__(self, args):
        self.lastResponse = None
        self.args = args
        self.service_ticket = args.service_ticket or get_tvm_service_ticket(
            tvm_client_id=args.tvm_self_service_id,
            tvm_service_id=args.tvm_api_service_id,
            tvm_client_secret=args.tvm_client_secret,
            skip_authentication=args.skip_authentication,
            logger=logging,
            self_app_name='orchestrator_cli'
        )

        self.headers = {
            "Content-Type": "application/json",
            "X-Ya-YandexUid": args.yandex_uid,
            "X-Ya-Session-Key": args.sessionid,
            "X-Ya-Service-Ticket": self.service_ticket,
        }
        if args.passport_id:
            self.headers["X-Ya-PassportId"] = args.passport_id

        if args.user_ticket:
            self.headers["X-Ya-User-Ticket"] = args.user_ticket

        if args.icookie:
            self.headers["X-Ya-ICookie"] = args.icookie

        self.last_data_storage = LastDataStorage()

        self.label_params = {
            "from": "moscow",
            "icookie": "U2wdAwZFcCPAtz0/wqElv22Ht2pHmDSIDqUYTqxEDb8sJYDVeA1t6XvUYZ0nFIFRsd/ZGGsRkbo1o5177Zoi4ge38aE=",
            "testBuckets": "505939,0,-1",
            "device": "desktop",
            "terminal": "travel",
            "ip": "2a02:6b8:b081:b71c::1:d",
            "regionId": 54,
            "yandexUid": "3535336521618583201",
            "metrikaClientId": "1618583202747698087"
        }

    def create_order(self, request):
        response = requests.post(
            "http://{host}:{port}/api/generic_booking_flow/v1/create_order".format(host=self.args.host,
                                                                                   port=self.args.port),
            headers=self.headers,
            json=request,
            timeout=self.args.timeout
        )
        self.log_response(response)
        response.raise_for_status()

        return response.json()["id"]

    def trains_search(self, params):
        response = requests.get(
            "http://{host}:{port}/api/trains/v1/search".format(host=self.args.host, port=self.args.port),
            headers=self.headers,
            timeout=self.args.timeout,
            params=params
        )
        self.log_response(response)
        response.raise_for_status()
        return response.json()

    def trains_direction(self, params):
        response = requests.get(
            "http://{host}:{port}/api/trains/v1/direction".format(host=self.args.host, port=self.args.port),
            headers=self.headers,
            timeout=self.args.timeout,
            params=params
        )
        self.log_response(response)
        response.raise_for_status()
        return response.json()

    def add_service(self, request):
        response = requests.post(
            "http://{host}:{port}/api/generic_booking_flow/v1/add_service".format(host=self.args.host,
                                                                                  port=self.args.port),
            headers=self.headers,
            json=request,
            timeout=self.args.timeout
        )
        self.log_response(response)
        response.raise_for_status()

        return response.json()

    def start_payment(self, order_uid):
        response = requests.post(
            "http://{host}:{port}/api/generic_booking_flow/v1/start_payment".format(host=self.args.host,
                                                                                    port=self.args.port),
            headers=self.headers,
            json={
                "order_id": order_uid,
                "return_url": "https://example.org/",
                "source": "desktop"
            },
            timeout=self.args.timeout
        )
        self.log_response(response)
        response.raise_for_status()

    def cancel_order(self, order_uid):
        response = requests.post(
            "http://{host}:{port}/api/generic_booking_flow/v1/cancel_order".format(host=self.args.host,
                                                                                   port=self.args.port),
            headers=self.headers,
            json={
                "order_id": order_uid
            },
            timeout=self.args.timeout
        )
        self.log_response(response)
        response.raise_for_status()

    def get_status(self, order_uid):
        response = requests.get(
            "http://{host}:{port}/api/generic_booking_flow/v1/get_order_state".format(
                host=self.args.host,
                port=self.args.port,
            ),
            headers=self.headers,
            timeout=self.args.timeout,
            params={
                'orderId': order_uid,
            },
        )
        self.log_response(response)
        response.raise_for_status()
        return response.json()

    def get_order_info(self, order_uid, source='OTHER'):
        response = requests.get(
            "http://{host}:{port}/api/generic_booking_flow/v1/get_order".format(
                host=self.args.host,
                port=self.args.port,
            ),
            headers=self.headers,
            timeout=self.args.timeout,
            params={
                'orderId': order_uid,
                'source': source,
            },
        )
        self.log_response(response)
        response.raise_for_status()

        response_json = response.json()
        self.last_data_storage.save_last_data(response_json)

        return response_json

    def get_last_order_uid(self):
        last_data = self.last_data_storage.load_last_data()
        return last_data['order_uid']

    def get_order_info_last(self):
        return self.get_order_info(self.get_last_order_uid())

    def calculate_refund_amount(self, request):
        response = requests.post(
            "http://{host}:{port}/api/generic_booking_flow/v1/calculate_refund_amount".format(
                host=self.args.host,
                port=self.args.port
            ),
            headers=self.headers,
            json=request,
            timeout=self.args.timeout
        )
        self.log_response(response)
        response.raise_for_status()

        return response.json()

    def start_refund(self, request):
        response = requests.post(
            "http://{host}:{port}/api/generic_booking_flow/v1/start_refund".format(
                host=self.args.host,
                port=self.args.port
            ),
            headers=self.headers,
            json=request,
            timeout=self.args.timeout
        )
        self.log_response(response)
        response.raise_for_status()

    def trains_download_blank(self, order_uid):
        return self.get_content("/api/trains_booking_flow/v1/download_blank", {'id': order_uid})

    def download_blank(self, token):
        return self.get_content("/api/generic_booking_flow/v1/download_blank", {'token': token})

    def get_content(self, query, params=None):
        response = requests.get(
            "http://{host}:{port}{query}".format(
                host=self.args.host,
                port=self.args.port,
                query=query,
            ),
            headers=self.headers,
            timeout=self.args.timeout,
            params=params,
        )
        self.log_response(response)
        response.raise_for_status()
        return response.content

    def add_insurance(self, order_uid):
        response = requests.post(
            "http://{host}:{port}/api/trains_booking_flow/v1/add_insurance".format(
                host=self.args.host,
                port=self.args.port,
            ),
            json={'id': order_uid},
            headers=self.headers,
            timeout=self.args.timeout
        )
        self.log_response(response)
        response.raise_for_status()

    def get_train_test_context_token(self, params):
        response = requests.get(
            "http://{host}:{port}/api/test_context/v1/train_token".format(
                host=self.args.host,
                port=self.args.port,
            ),
            headers=self.headers,
            timeout=self.args.timeout,
            params=params,
        )
        self.log_response(response)
        response.raise_for_status()
        return response.json()["test_context_token"]

    def get_bus_test_context_token(self, params):
        response = requests.get(
            "http://{host}:{port}/api/test_context/v1/bus_token".format(
                host=self.args.host,
                port=self.args.port,
            ),
            headers=self.headers,
            timeout=self.args.timeout,
            params=params,
        )
        self.log_response(response)
        response.raise_for_status()
        return response.json()["test_context_token"]

    def get_suburban_test_context_token(self, params):
        response = requests.get(
            "http://{host}:{port}/api/test_context/v1/suburban_token".format(
                host=self.args.host,
                port=self.args.port,
            ),
            headers=self.headers,
            timeout=self.args.timeout,
            params=params,
        )
        self.log_response(response)
        response.raise_for_status()
        return response.json()["test_context_token"]

    def get_test_context_payment_token(self, payment_code=None, fail_desc=None):
        if not payment_code:
            query = {"paymentOutcome": "PO_SUCCESS"}
        else:
            query = {
                "paymentOutcome": "PO_FAILURE",
                "paymentFailureResponseCode": payment_code,  # https://wiki.yandex-team.ru/trust/payments/rc/
                "paymentFailureResponseDescription": fail_desc,
            }

        response = requests.get(
            "http://{host}:{port}/api/test_context/v1/payment_token".format(
                host=self.args.host,
                port=self.args.port,
            ),
            params=query,
            headers=self.headers,
            timeout=self.args.timeout
        )
        self.log_response(response)
        response.raise_for_status()
        return response.json()["token"]

    def happy_page(self, order_id):
        response = requests.get(
            "http://{host}:{port}/api/orders/v1/get_order_happy_page".format(
                host=self.args.host,
                port=self.args.port,
            ),
            headers=self.headers,
            timeout=self.args.timeout,
            params={'orderId': order_id, 'geoId': '213'},
        )
        self.log_response(response)
        response.raise_for_status()
        return response.json()

    def start_takeout_get(self):
        response = requests.post(
            "http://{host}:{port}/1/takeout/start".format(
                host=self.args.host,
                port=self.args.port,
            ),
            headers=self.headers,
            timeout=self.args.timeout,
            params={'uid': self.args.passport_id},
        )
        self.log_response(response)
        response.raise_for_status()
        return response.json()

    def get_takeout_result(self, start_result):
        if start_result['status'] != 'ok':
            raise Exception('Bad takeout', start_result)

        response = requests.post(
            "http://{host}:{port}/1/takeout/result".format(
                host=self.args.host,
                port=self.args.port,
            ),
            headers=self.headers,
            timeout=self.args.timeout,
            params={'job_id': start_result['job_id']},
        )
        self.log_response(response)
        response.raise_for_status()
        return response.json()

    def start_takeout_delete(self):
        response = requests.post(
            "http://{host}:{port}/1/takeout/delete".format(
                host=self.args.host,
                port=self.args.port,
            ),
            headers=self.headers,
            timeout=self.args.timeout,
            params={'request_id': uuid.uuid4(), "id": ["orders"]},
        )
        self.log_response(response)
        response.raise_for_status()
        return response.json()

    def takeout_status(self):
        response = requests.get(
            "http://{host}:{port}/1/takeout/status".format(
                host=self.args.host,
                port=self.args.port,
            ),
            headers=self.headers,
            timeout=self.args.timeout,
            params={'request_id': uuid.uuid4()},
        )
        self.log_response(response)
        response.raise_for_status()
        return response.json()

    def create_ride_offer(self, ride_id):
        response = requests.post(
            "http://{host}:{port}/api/buses/v1/create_ride_offer".format(
                host=self.args.host,
                port=self.args.port,
            ),
            headers=self.headers,
            timeout=self.args.timeout,
            data=json.dumps({'rideId': ride_id, 'labelParams': self.label_params}),
        )
        self.log_response(response)
        response.raise_for_status()
        return response.json()

    def log_response(self, response):
        if (self.lastResponse and self.lastResponse.request.url == response.request.url
                and self.lastResponse.text == response.text and self.lastResponse.status_code == response.status_code):

            logging.log(logging.INFO, "^^^ Same response ^^^")
        else:
            logging.log(logging.INFO, "{} => {}".format(response.request.url, response.status_code))
            if 'application/pdf' in response.headers.get('Content-Type', {}) or response.content.startswith(b'%PDF'):
                logging.log(logging.INFO, '<some pdfdata>')
            else:
                logging.log(logging.INFO, response.text)
        self.lastResponse = response

    def get_admin_order_info(self, order_id):
        logging.log(logging.INFO, self.headers)
        response = requests.get(
            "http://{host}:{port}/api/admin/v1/get_order_no_auth".format(
                host=self.args.host,
                port=self.args.port,
            ),
            headers=self.headers,
            timeout=self.args.timeout,
            params={'id': order_id, 'snippets': 'S_PRIVATE_INFO'},
        )
        self.log_response(response)
        response.raise_for_status()
        return response.json()
