# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import time as c_time
import uuid
from datetime import datetime, timedelta
from typing import Any

from travel.orders.tools.generic_client.common import last_order_info_scenario
from travel.orders.tools.generic_client.orch_api import OrchAPI
from travel.orders.tools.generic_client.suburban.export_api import ExportClient
from travel.orders.tools.generic_client.suburban.suburban_selling_api import SuburbanSellingClient

ST_BELORUSSKIY = {'id': 2000006, 'express_id': 2000006, 'name': 'Москва (Белорусский вокзал)', 'esr': '198230'}
ST_SAVELOVSKIY = {'id': 2000009, 'express_id': 2000009, 'name': 'Москва (Савёловский вокзал)', 'esr': '196004'}
ST_KURSKIY = {'id': 2000001, 'express_id': 2000001, 'name': 'Москва (Курский вокзал)', 'esr': '191602'}

ST_ODINCOVO = {'id': 9600721, 'express_id': 2000055, 'name': 'Одинцово', 'esr': '182209'}
ST_LOBNYA = {'id': 9600781, 'express_id': 2000115, 'name': 'Лобня', 'esr': '237908'}
ST_SVO = {'id': 9600213, 'express_id': 2001100, 'name': 'а/п Шереметьево', 'esr': '321'}

ST_MIITOVSKAYA = {'id': 9600675, 'express_id': 2000004, 'name': 'Миитовская', 'esr': '9600675'}

ST_FINLANDSKIY_VOKZAL = {'id': 9602497, 'express_id': 2004004, 'name': 'Финляндский вокзал', 'esr': '038205'}
ST_BALTIYSKIY_VOKZAL = {'id': 9602498, 'express_id': 2004005, 'name': 'Балтийский вокзал', 'esr': '036002'}
ST_MOSKOVSKIY_VOKZAL_SPB = {'id': 9602494, 'express_id': 2004001, 'name': 'Московский вокзал (Спб)', 'esr': '031812'}
ST_UDELNAYA = {'id': 9603463, 'express_id': 2005065, 'name': 'Удельная', 'esr': '038525'}
ST_NOVIY_PETERGOF = {'id': 9603887, 'express_id': 2006300, 'name': 'Новый Петергоф', 'esr': '075512'}

ST_MGA = {'id': 9603127, 'express_id': 2004634, 'name': 'Мга', 'esr': '030203'}
ST_TVER ={'id': 9603093, 'express_id': 2004600, 'name': 'Тверь', 'esr': '061502'}
ST_BOLOGOE ={'id': 9603153, 'express_id': 2004660, 'name': 'Бологое', 'esr': '050009'}
ST_PROLETARSKAYA = {'id': 9603880, 'express_id': 2006103, 'name': 'Пролетарская', 'esr': '061517'}

ST_UFA = {'id': 9606364, 'express_id': 2024600, 'name': 'Уфа', 'esr': '654504'}
ST_URMAN = {'id': 9606509, 'express_id': 2024746, 'name': 'Урман', 'esr': '655704'}

ST_KRASNOUFIMSK = {'id': 9612412, 'express_id': 2060610, 'name': 'Красноуфимск', 'esr': '258906'}
ST_DRUGININO = {'id': 9607445, 'express_id': 2030042, 'name': 'Дружинино', 'esr': '782505'}


class OrderApi(object):
    def create_order(self, args, date, station_from, station_to, book_data, price, label, mock_payment_token='', suburban_test_context_token=''):
        raise NotImplementedError

    def get_status(self, order_uid):
        raise NotImplementedError

    def start_payment(self, order_uid):
        raise NotImplementedError


def get_provider_and_carrier(args):
    provider = args.suburban_provider if args.suburban_provider else 'movista'
    carrier = args.suburban_carrier
    if not carrier:
        if provider == 'aeroexpress':
            carrier = 'aeroexpress'
        elif provider == 'movista':
            carrier = 'cppk'
        elif provider == 'im':
            carrier = 'szppk'
    return provider, carrier


class OrcOrderApi(OrderApi):
    def __init__(self, orc_api, args):
        # type: (OrchAPI, Any) -> None
        self.orc_api = orc_api

    def _create_service(self, args, station_from, station_to, book_data, price):
        provider, carrier = get_provider_and_carrier(args)
        return {
            'provider': provider,
            'carrier_partner': carrier,
            'station_from_id': station_from['id'],
            'station_to_id': station_to['id'],
            'price': price,
            'provider_book_data': book_data,
        }

    def create_order(
        self, args, date, station_from, station_to, book_data, price, label,
        mock_payment_token='', suburban_test_context_token=''
    ):
        request = {
            'deduplication_key': str(uuid.uuid4()),
            'contact_info': {
                'phone': args.phone,
                'email': args.email,
            },
            'user_info': {
                'ip': '1.2.3.4',
                'geo_id': 213,
            },
            'label': label,
            'order_history': [],
            'suburban_services': [
                self._create_service(args, station_from, station_to, book_data, price)
            ],
            'payment_test_context_token': mock_payment_token,
        }

        if suburban_test_context_token:
            for service in request['suburban_services']:
                service['test_context_token'] = suburban_test_context_token

        return self.orc_api.create_order(request)

    def get_status(self, order_uid):
        return self.orc_api.get_status(order_uid)['state']

    def start_payment(self, order_uid):
        return self.orc_api.start_payment(order_uid)


class SuburbanSellingOrderApi(OrderApi):
    def __init__(self, ss_api, args):
        # type: (SuburbanSellingClient, Any) -> None
        self.ss_api = ss_api

    def _create_data(self, args, date, station_from, station_to, price):
        date_str = date.strftime('%Y-%m-%d')
        provider, carrier = get_provider_and_carrier(args)
        return {
            'provider': provider,
            'partner': carrier,
            'version': 3,
            'station_from': station_from['id'],
            'station_to': station_to['id'],
            'departure_date': date_str,
            'price': price,
        }

    def create_order(
        self, args, date, station_from, station_to, book_data, price, label,
        mock_payment_token='', suburban_test_context_token=''
    ):
        user_info = {
            'email': args.email,
            'phone': args.phone,
            'region_id': 213,
            'ip': '1.2.3.4',
        }

        data = self._create_data(args, date, station_from, station_to, price)
        resp = self.ss_api.create_order(data, book_data, user_info)
        return resp['uid']

    def get_status(self, order_uid):
        return self.ss_api.order_info(order_uid)['status']

    def start_payment(self, order_uid):
        return self.ss_api.order_info(order_uid, cmd='start_payment')


def main(args):
    if args.suburban_selling_host:
        args.sessionid = args.yandex_uid  # в мобилках используются одинаковые значения всегда

    orc_api = OrchAPI(args)

    if args.last:
        last_order_info_scenario(orc_api)
        return

    suburban_test_context_token = ''
    if args.mock_suburban_token:
        suburban_test_context_token = args.mock_suburban_token
    elif args.mock_suburban:
        suburban_test_context_token = orc_api.get_suburban_test_context_token({
            'ticketBody': '333',
            'ticketNumber': '222',
            'actualPrice': '78',
            'validForSeconds': '1800',

            'bookHandlerErrorType': 'ET_RETRYABLE',
            'bookHandlerErrorCount': 1,
            'confirmHandlerErrorType': 'ET_RETRYABLE',
            'confirmHandlerErrorCount': 1,
            'orderInfoHandlerErrorType': 'ET_RETRYABLE',
            'orderInfoHandlerErrorCount': 0,
            'ticketBarcodeHandlerErrorType': 'ET_RETRYABLE',
            'ticketBarcodeHandlerErrorCount': 0,
            'blankPdfHandlerErrorType': 'ET_RETRYABLE',
            'blankPdfHandlerErrorCount': 1,
        })

    mock_payment_token = ''
    if args.mock_payment:
        mock_payment_token = orc_api.get_test_context_payment_token(args.mock_payment_code, args.mock_payment_desc)

    date = (datetime.now() + timedelta(days=1)).date()

    provider, carrier = get_provider_and_carrier(args)

    if provider == 'im':
        if carrier == 'szppk':
            station_from, station_to = ST_FINLANDSKIY_VOKZAL, ST_UDELNAYA
            # station_from, station_to = ST_BALTIYSKIY_VOKZAL, ST_NOVIY_PETERGOF
            # station_from, station_to = ST_MOSKOVSKIY_VOKZAL_SPB, ST_MGA
        elif carrier == 'mtppk':
            # station_from, station_to = ST_TVER, ST_BOLOGOE
            station_from, station_to = ST_TVER, ST_PROLETARSKAYA
        elif carrier == 'bashppk':
            station_from, station_to = ST_UFA, ST_URMAN
        elif carrier == 'sodruzhestvo':
            station_from, station_to = ST_KRASNOUFIMSK, ST_DRUGININO
        else:
            raise Exception('Unknown carrier', carrier)
    elif provider == 'aeroexpress':
        station_from, station_to = ST_BELORUSSKIY, ST_SVO
    elif provider == 'movista':
        station_from, station_to = ST_ODINCOVO, ST_BELORUSSKIY
        # station_from, station_to = ST_MIITOVSKAYA, ST_KURSKIY
        # station_from, station_to = ST_SAVELOVSKIY, ST_LOBNYA
    else:
        raise Exception('Unknown provider', args.suburban_provider)

    ss_api = SuburbanSellingClient(args, suburban_test_context_token, mock_payment_token)

    if suburban_test_context_token:
        if provider == 'aeroexpress':
            book_data = {
                'menu_id': 14,
                'order_type': 39,
                'date': date.strftime('%Y-%m-%d')
            }
            price = 450
        elif provider == 'movista':
            book_data = {
                'date': date.strftime('%Y-%m-%d'),
                'station_to_express_id': station_from['express_id'],
                'station_from_express_id': station_to['express_id'],
                'fare_id': 1200675,
            }
            price = 78
        elif provider == 'im':
            book_data = {
                'date': date.strftime('%Y-%m-%d'),
                'station_to_express_id': station_from['express_id'],
                'station_from_express_id': station_to['express_id'],
                'train_number': '6666',
            }
            price = 78
        else:
            raise Exception('Test context is not implemented for {}'.format(args.suburban_provider))
    else:
        export_api = ExportClient(args.suburban_export_host)
        # book_data, price = ss_api.get_fare_and_price(date, station_from['id'], station_to['id'], args.suburban_provider)
        if provider == 'aeroexpress':
            book_data, price = export_api.get_book_data(date, station_from['esr'], station_to['esr'], 'aeroexpress')
            book_data['date'] = date.strftime('%Y-%m-%d')
        else:
            book_data, price = export_api.get_book_data(date, station_from['esr'], station_to['esr'])

    if args.suburban_selling_order:
        order_api = SuburbanSellingOrderApi(ss_api, args)
    else:
        order_api = OrcOrderApi(orc_api, args)

    logging.info('Creating order from {} to {} on {}'.format(station_from['name'], station_to['name'], date))
    order_uid = order_api.create_order(
        args=args,
        date=date,
        station_from=station_from,
        station_to=station_to,
        book_data=book_data,
        price=price,
        label='random_label42',
        mock_payment_token=mock_payment_token,
        suburban_test_context_token=suburban_test_context_token,
    )

    logging.info('Создали заказ %s', order_uid)
    orc_api.get_order_info(order_uid)

    while True:
        status = order_api.get_status(order_uid)
        if status == 'RESERVED':
            break
        elif status == 'CANCELLED':
            logging.info('Ошибка бронирования заказа %s', order_uid)
            orc_api.get_order_info(order_uid)
            return
        c_time.sleep(1)

    orc_api.get_order_info(order_uid)
    order_api.start_payment(order_uid)
    logging.info('Запустили оплату')

    while True:
        status = order_api.get_status(order_uid)
        if status == 'WAITING_PAYMENT':
            break
        c_time.sleep(1)
    logging.info('Заказ ожидает оплаты %s', order_uid)

    orc_api.get_order_info(order_uid)
    cnt = 0
    while True:
        cnt += 1
        status = order_api.get_status(order_uid)
        if status == 'CONFIRMED':
            logging.info('Билеты выкуплены')
            break
        elif status == 'CANCELLED':
            logging.info('Заказ отменен')
            break

        c_time.sleep(2)
        if not (cnt % 5):
            logging.info(status)

    orc_api.get_order_info(order_uid)
