# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import json
import logging

import requests


class SuburbanSellingClient(object):
    def __init__(self, args, suburban_test_context_token, mock_payment_token):
        self.suburban_test_context_token = suburban_test_context_token
        self.mock_payment_token = mock_payment_token
        self.host = args.suburban_selling_host
        self.yuid = args.yandex_uid

    def call(self, method, handler, params=None, data=None):
        headers = {'X-YA-UUID': self.yuid}
        if self.suburban_test_context_token:
            headers['X-TEST-CONTEXT-TOKEN'] = self.suburban_test_context_token
        if self.mock_payment_token:
            headers['X-PAYMENT-TEST-CONTEXT-TOKEN'] = self.mock_payment_token

        url = '{}/{}'.format(self.host, handler)

        logging.info('{}: {} {}'.format(url, params, data))
        res = requests.request(
            method=method,
            url=url,
            headers=headers,
            params=params,
            data=data,
        )
        logging.info('{} {}'.format(res.status_code, res.text))

        try:
            res.raise_for_status()
        except Exception:
            print(res.status_code, res.text)
            raise

        return res.json()

    def create_order(self, data, book_data, user_info):
        data['book_data'] = json.dumps(book_data)
        data['user_info'] = json.dumps(user_info)
        print(json.dumps(data, indent=4))
        return self.call('POST', 'create_order/', data=data)

    def order_info(self, order_id, cmd=''):
        res = self.call('GET', 'order_info/', params={
            'version': 2,
            'uid': order_id,
            'command': cmd,
        })

        return res

    def get_tariffs(self, date_str, station_from, station_to, provider, tariff_type='etrain'):
        provider_to_code = {
            'movista': 153,
            'im': 1332,
            'aeroexpress': 162
        }

        return self.call('POST', 'get_tariffs/', data=json.dumps(
            {
                'keys': [
                    {
                        'company': provider_to_code[provider],
                        'date': date_str,
                        'station_from': station_from,
                        'station_to': station_to,
                        'tariff_type': tariff_type,
                    }
                ],
                'selling_flows': ['validator', 'simple', 'aeroexpress'],
                'tariff_types': {
                    '6153': 1,
                    '6171': 1,
                }
        }))

    def get_fare_and_price(self, date, station_from_id, station_to_id, provider):
        date_str = date.strftime('%Y-%m-%d')
        tariffs = self.get_tariffs(date_str, station_from_id, station_to_id, provider)

        for tariff in tariffs['result']['selling_tariffs']:
            if tariff['provider'] == provider:
                book_data = tariff['tariffs'][0]['book_data']
                return json.loads(book_data), tariff['tariffs'][0]['price']
        else:
            raise Exception("Can't find fare if for {} {} {}: {}".format(
                date_str, station_from_id, station_to_id,
                tariffs
            ))
