# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
import time as c_time
import uuid
from datetime import datetime, date, time, timedelta

import requests
from six.moves.urllib.parse import parse_qs, urlparse

from travel.orders.tools.generic_client.common import last_order_info_scenario
from travel.orders.tools.generic_client.orch_api import OrchAPI
from travel.orders.tools.library.tvm import get_tvm_service_ticket

LAST_NAMES = ["Сугробин", "Иванов", "Петров", "Сидоров"]
PASSPORT_NUMBERS = ["6500112233", "6500112244", "6500112255", "6500112266"]
BIRTH_CERT_NUMBERS = ["XXАБ111111", "XXАБ222222", "XXАБ333333", "XXАБ444444"]


def create_train_order_request(args, offer_id, label, offer_id2=None,
                               train_test_context_token=None, train_test_context_token2=None,
                               payment_test_context_token=None):
    def create_train_service(o):
        result_service = {
            "passengers": [],
            "offer_id": o,
        }
        index = 0
        for i in range(args.full):
            result_service["passengers"].append(
                {
                    "index": index,
                    "first_name": "Илья",
                    "patronymic": "Викторович",
                    "last_name": LAST_NAMES[index],
                    "birth_date": "1990-01-01",
                    "doc_id": PASSPORT_NUMBERS[index],
                },
            )
            index += 1
        for i in range(args.child):
            result_service["passengers"].append(
                {
                    "index": index,
                    "first_name": "Сергей",
                    "patronymic": "Ильич",
                    "last_name": LAST_NAMES[index],
                    "birth_date": "2015-02-02",
                    "doc_id": BIRTH_CERT_NUMBERS[index],
                },
            )
            index += 1
        for i in range(args.baby):
            result_service["passengers"].append(
                {
                    "index": index,
                    "first_name": "Игорь",
                    "patronymic": "Сергеевич",
                    "last_name": LAST_NAMES[index],
                    "birth_date": "2019-03-03",
                    "doc_id": BIRTH_CERT_NUMBERS[index],
                },
            )
            index += 1
        return result_service

    result = {
        "deduplication_key": str(uuid.uuid4()),
        "label": label,
        "contact_info": {
            "phone": args.phone,
            "email": args.email,
        },
        "user_info": {
            "ip": "1.2.3.4",
            "region_id": 213,
            "yandex_uid": args.yandex_uid,
            "geo_id": 213,
        },
        "order_history": [],
        "train_services": [create_train_service(offer_id)]
    }
    if offer_id2:
        result['train_services'].append(create_train_service(offer_id2))
    if payment_test_context_token:
        result['payment_test_context_token'] = payment_test_context_token
    if train_test_context_token:
        result['train_services'][0]['train_test_context_token'] = train_test_context_token
        if len(result['train_services']) > 1:
            result['train_services'][1]['train_test_context_token'] = train_test_context_token2 or train_test_context_token
    return result


def create_train_offer_request(args):
    departure = datetime.combine(date.today(), time(5, 45)) + timedelta(days=19, hours=-3)
    arrival = departure + timedelta(hours=3, minutes=30)
    offer = {
        "partner": "im",
        "arrival": arrival.strftime("%Y-%m-%dT%H:%M:00+00:00"),
        "departure": departure.strftime("%Y-%m-%dT%H:%M:00+00:00"),
        "passengers": create_train_offer_passengers(args),
        "label_params": {
            "req_id": "reqId",
            "device": "desktop",
            "utm_source": "utmSource",
            "utm_medium": "utmMedium",
            "utm_campaign": "utmCampaign",
            "utm_term": "utmTerm",
            "utm_content": "utmContent",
            "from": "from",
            "gclid": "gugloid",
            "icookie": "test-icookie",
        },
        "train_number": "752А",
        "train_ticket_number": "752А",
        "train_title": "Москва — Санкт-Петербург",
        "station_from_id": 2006004,
        "station_to_id": 9602494,
        "car_number": "08",
        "car_type": "sitting",
        "service_class": "2С",
        "company_title": "ФБК",
        "electronic_registration": True,
        "bedding": True,
        "brand_title": "САПСАН",
        "route_policy": "internal",
        "icookie": "<test-train-client>",
        "amount": 3198.58,
        "direction": "FORWARD",
        "segment_index": 0,
    }
    if args.places:
        offer['places'] = args.places
    return offer


def create_train_offer_transfer_request(args):
    departure = datetime.combine(date.today(), time(19, 25)) + timedelta(days=19, hours=-3)
    arrival = departure + timedelta(hours=1, minutes=20)
    return {
        "partner": "im",
        "arrival": arrival.strftime("%Y-%m-%dT%H:%M:00+00:00"),
        "departure": departure.strftime("%Y-%m-%dT%H:%M:00+00:00"),
        "passengers": create_train_offer_passengers(args),
        "label_params": {
            "req_id": "reqId",
            "device": "desktop",
            "utm_source": "utmSource",
            "utm_medium": "utmMedium",
            "utm_campaign": "utmCampaign",
            "utm_term": "utmTerm",
            "utm_content": "utmContent",
            "from": "from",
            "gclid": "gugloid",
            "icookie": "test-icookie",
        },
        "train_number": "823В",
        "train_ticket_number": "823В",
        "train_title": "Санкт-Петербург — Сортавала",
        "im_final_station_name": "СОРТАВАЛА",
        "im_initial_station_name": "С-ПЕТ-ФИН",
        "station_from_id": 9602497,
        "station_to_id": 9603175,
        "car_number": "02",
        "car_type": "sitting",
        "service_class": "2П",
        "electronic_registration": True,
        "bedding": True,
        "can_choose_bedding": False,
        "brand_title": "«Ласточка»",
        "route_policy": "internal",
        "icookie": "<test-train-client>",
        "amount": 0,
        "direction": "FORWARD",
        "segment_index": 1,
    }


def create_train_offer_back_request(args):
    departure = datetime.combine(date.today(), time(5, 30)) + timedelta(days=47, hours=-3)
    arrival = departure + timedelta(hours=3, minutes=30)
    return {
        "partner": "im",
        "arrival": arrival.strftime("%Y-%m-%dT%H:%M:00+00:00"),
        "departure": departure.strftime("%Y-%m-%dT%H:%M:00+00:00"),
        "passengers": create_train_offer_passengers(args),
        "label_params": {
            "req_id": "reqId",
            "device": "desktop",
            "utm_source": "utmSource",
            "utm_medium": "utmMedium",
            "utm_campaign": "utmCampaign",
            "utm_term": "utmTerm",
            "utm_content": "utmContent",
            "from": "from",
            "gclid": "gugloid",
            "icookie": "test-icookie",
        },
        "train_number": "751А",
        "train_ticket_number": "751А",
        "train_title": "Санкт-Петербург — Москва",
        "station_from_id": 9602494,
        "station_to_id": 2006004,
        "car_number": "08",
        "car_type": "sitting",
        "service_class": "2Р",
        "electronic_registration": True,
        "bedding": True,
        "brand_title": "САПСАН",
        "route_policy": "internal",
        "icookie": "<test-train-client>",
        "amount": 3198.58,
        "direction": "BACKWARD",
        "segment_index": 0,
    }


def create_train_offer_passengers(args):
    passengers = []
    index = 0
    def set_place(passenger):
        if args.places:
            passenger['places'] = [args.places[passenger['index']]]

    for i in range(args.full):
        passengers.append(
            {
                "index": index,
                "doc_type": "ПН",
                "sex": "M",
                "citizenship_geo_id": 225,
                "age_group": "adults",
                "tariff": "full"
            },
        )
        set_place(passengers[index])
        index += 1
    for i in range(args.child):
        passengers.append(
            {
                "index": index,
                "sex": "M",
                "doc_type": "СР",
                "citizenship_geo_id": 225,
                "loyalty_cards": [],
                "tariff": "child",
                "age_group": "children"
            }
        )
        set_place(passengers[index])
        index += 1
    for i in range(args.baby):
        passengers.append(
            {
                "index": index,
                "sex": "M",
                "doc_type": "СР",
                "citizenship_geo_id": 225,
                "loyalty_cards": [],
                "tariff": "baby",
                "age_group": "babies"
            }
        )
        index += 1
    return passengers


class TrainOfferStorageClient(object):
    def __init__(self, args):
        self.args = args
        self.service_ticket = args.service_ticket or get_tvm_service_ticket(
            tvm_client_id=args.tvm_self_service_id,
            tvm_service_id=args.tvm_train_offers_service_id,
            tvm_client_secret=args.tvm_client_secret,
            skip_authentication=args.skip_authentication,
            logger=logging,
            self_app_name='orchestrator_cli'
        )

        self.headers = {
            "Content-Type": "application/json",
            "X-Ya-YandexUid": args.yandex_uid,
            "X-Ya-Session-Key": args.sessionid
        }

    def create_offer(self, request):
        response = requests.post(
            "https://{host}:{port}/store".format(host=self.args.train_offer_host, port=self.args.train_offer_port),
            headers=self.headers,
            json=request,
            timeout=self.args.timeout
        )
        self.log_response(response)
        response.raise_for_status()
        return response.json()

    def log_response(self, response):
        logging.log(logging.INFO, "{} => {}".format(response.request.url, response.status_code))
        logging.log(logging.INFO, response.text)


def get_check_url(payment_url):
    parsed_url = urlparse(payment_url)
    purchase_token = parse_qs(parsed_url.query)["purchase_token"][0]
    return "https://{0}/pchecks/{1}/receipts/{1}/?mode=pdf".format(parsed_url.hostname, purchase_token)


def get_train_test_context_default_query_params():
    return {
        "insurancePricingOutcome": "IPO_SUCCESS",
        "insuranceCheckoutOutcome": "ICO_SUCCESS",
        "insuranceCheckoutConfirmOutcome": "ICCO_SUCCESS",
        "refundPricingOutcome": "RPO_SUCCESS",
        "refundCheckoutOutcome": "RCO_SUCCESS",
        "createReservationOutcome": "RCRO_SUCCESS",
        "confirmReservationOutcome": "RCOO_SUCCESS",
        "officeReturnDelayInSeconds": 0,
        "officeAcquireDelayInSeconds": 0,
        "alwaysTimeoutAfterConfirmingInSeconds": 0,
    }


def get_train_test_context_office_refund_query_params():
    res = get_train_test_context_default_query_params()
    res["officeReturnDelayInSeconds"] = 5
    return res


def get_train_test_context_insurance_confirm_fail_query_params():
    res = get_train_test_context_default_query_params()
    res["insuranceCheckoutConfirmOutcome"] = 'ICCO_FAILURE'
    return res


def main(args):
    orc_api = OrchAPI(args)

    if args.last:
        last_order_info_scenario(orc_api)
        return

    offer_client = TrainOfferStorageClient(args)
    create_offer_rsp = offer_client.create_offer(create_train_offer_request(args))
    offer_id = create_offer_rsp["offer_id"]
    label_hash = create_offer_rsp["label_hash"]
    offer_id2 = None
    test_context_token = None
    test_context_token2 = None
    payment_test_context_token = None

    if args.office_refund:
        test_context_token = orc_api.get_train_test_context_token(get_train_test_context_office_refund_query_params())
        test_context_token2 = orc_api.get_train_test_context_token(get_train_test_context_default_query_params())
    elif args.insurance_confirm_fail:
        test_context_token = orc_api.get_train_test_context_token(get_train_test_context_insurance_confirm_fail_query_params())
        args.insurance = True
    elif args.mock_im:
        test_context_token = orc_api.get_train_test_context_token(get_train_test_context_default_query_params())

    if args.mock_payment:
        payment_test_context_token = orc_api.get_test_context_payment_token()

    if args.round:
        create_offer_rsp = offer_client.create_offer(create_train_offer_back_request(args))
        offer_id2 = create_offer_rsp["offer_id"]
    elif args.transfer:
        create_offer_rsp = offer_client.create_offer(create_train_offer_transfer_request(args))
        offer_id2 = create_offer_rsp["offer_id"]

    order_uid = orc_api.create_order(create_train_order_request(args, offer_id, label_hash, offer_id2,
                                                                train_test_context_token=test_context_token,
                                                                train_test_context_token2=test_context_token2,
                                                                payment_test_context_token=payment_test_context_token))
    logging.info("Создали заказ %s", order_uid)

    status_data = None
    while True:
        status_data = orc_api.get_status(order_uid)
        if status_data["state"] == "RESERVED":
            break
        elif status_data["state"] == "CANCELLED":
            logging.info("Ошибка бронирования заказа %s", order_uid)
            orc_api.get_order_info(order_uid)
            return
        c_time.sleep(1)

    order_info = orc_api.get_order_info(order_uid)
    if args.insurance:
        insurance_status = order_info['services'][0]['train_info']['insurance_status']
        version_hash = status_data['version_hash']
        if insurance_status == 'PRICED':
            orc_api.add_insurance(order_uid)
            while True:
                status_data = orc_api.get_status(order_uid)
                if status_data["state"] == "RESERVED" and version_hash != status_data['version_hash']:
                    break
                c_time.sleep(1)
            logging.info("Добавили страховку")
            order_info = orc_api.get_order_info(order_uid)
        else:
            logging.info("Не добавили страховку. insurance_status != PRICED")

    if args.cancel:
        orc_api.cancel_order(order_uid)

        while True:
            status_data = orc_api.get_status(order_uid)
            if status_data["state"] == "CANCELLED":
                logging.info("Заказ отменен %s", order_uid)
                return
            c_time.sleep(1)

    orc_api.start_payment(order_uid)
    logging.info("Запустили оплату")

    while True:
        status_data = orc_api.get_status(order_uid)
        if status_data["state"] == "WAITING_PAYMENT":
            break
        c_time.sleep(1)
    logging.info("Заказ ожидает оплаты %s", order_uid)

    order_info = orc_api.get_order_info(order_uid)
    payment_url = order_info['payment']['payment_url']
    logging.info("Ссылка на форму оплаты: %s", payment_url)

    while True:
        status_data = orc_api.get_status(order_uid)
        if status_data["state"] == "CONFIRMED":
            break
        c_time.sleep(2)
    logging.info("Билеты выкуплены")

    order_info = orc_api.get_order_info(order_uid, source='ORDER_PAGE')

    if not test_context_token:
        body = orc_api.download_blank(order_info['services'][0]['train_info']['download_blank_token'])
        with open('blank.pdf', 'w') as f:
            f.write(body)
        logging.info("Билеты сохранены в blank.pdf")

    if args.office_refund:
        version_hash = status_data['version_hash']
        while True:
            status_data = orc_api.get_status(order_uid)
            if status_data["state"] in {"CONFIRMED", "REFUNDED"} and version_hash != status_data['version_hash']:
                break
            c_time.sleep(2)
        logging.info("Билеты возвращены (кассовый возврат)")
    elif args.refund:
        status_data = orc_api.get_status(order_uid)
        order_info = orc_api.get_order_info(order_uid)
        refund_part_info = [
            p['refund_part_info'] for s in order_info['services']
            for p in s['train_info']['passengers']
            if p['refund_part_info']['state'] == 'ENABLED'
        ][0]
        refund_amount_rsp = orc_api.calculate_refund_amount({'order_id': order_uid, 'refund_part_contexts': [refund_part_info['context']]})
        logging.info("Запросили сумму к возврату")

        version_hash = status_data['version_hash']
        orc_api.start_refund({'order_id': order_uid, 'refund_token': refund_amount_rsp['refund_token']})
        logging.info("Запустили возврат")
        while True:
            status_data = orc_api.get_status(order_uid)
            if status_data["state"] in {"CONFIRMED", "REFUNDED"} and version_hash != status_data['version_hash']:
                break
            c_time.sleep(2)
        logging.info("Билеты возвращены")

        order_info = orc_api.get_order_info(order_uid)
        if not test_context_token:
            refunded_passenger = [
                p for s in order_info['services']
                for p in s['train_info']['passengers']
                if p['refund_part_info']['context'] == refund_part_info['context']
            ][0]
            refund_blank_token = refunded_passenger['refund_part_info']['refund']['refund_blank_token']
            body = orc_api.download_blank(refund_blank_token)
            with open('refund_blank.pdf', 'w') as f:
                f.write(body)
            logging.info("КРС сохранена в refund_blank.pdf")

    logging.info("Счастливого пути!")

    while True:
        orc_api.get_order_info(order_uid)
        c_time.sleep(10)
