import logging
import uuid
from datetime import datetime

import ydb

log = logging.getLogger(__name__)

SELECT_NEXT_OWNER_LOGS = """
PRAGMA TablePathPrefix("{0}");

DECLARE $owner_uuid as Utf8;
DECLARE $loggers as List<Utf8>;
DECLARE $limit AS Uint32;
DECLARE $offset AS Uint32;

SELECT owner_uuid, `timestamp`, message_id,
    (logger in $loggers and String::Contains(message, "application/pdf")) as is_pdf
FROM {1}
WHERE owner_uuid >= $owner_uuid
ORDER BY owner_uuid, `timestamp`
LIMIT $limit
OFFSET $offset
;
"""

DELETE_LOG_RECORD = """
PRAGMA TablePathPrefix("{0}");

DECLARE $owner_uuid as "Utf8";
DECLARE $timestamp as "Uint64";
DECLARE $message_id as "Utf8";

DELETE FROM {1}
WHERE
    owner_uuid = $owner_uuid AND
    timestamp = $timestamp AND
    message_id = $message_id
"""

PROGRESS_REPORT_AFTER_ROWS = 10000

PDF_BODY_LOGGERS = {
    'ru.yandex.travel.orders.management.mail.HttpLogger',
    'ru.yandex.travel.orders.partners.im.HttpLogger',
    'ru.yandex.travel.orders.management.voucher.HttpLogger',
    'ru.yandex.travel.orders.partners.movista.HttpLogger',
    'ru.yandex.travel.orders.partners.imSuburban.HttpLogger'
}

MAX_UID_INT = uuid.UUID('ffffffff-ffff-ffff-ffff-ffffffffffff').int


class Patcher(object):
    def __init__(self, args, ydb_provider, executor):
        self.args = args
        self.ydb_provider = ydb_provider
        self.executor = executor
        self.table_name = args.log_table_name
        self.ydb_database = args.ydb_database
        self.rows_limit = args.rows_limit
        self.skipped_uids_limit = args.skipped_uids_limit
        self.max_uid_offset = args.max_uid_offset

        self.current_uid = self.args.start_uid
        self.current_uid_offset = 0
        self.select_next_query = None
        self.update_query = None
        self.processed_count = 0
        self.processed_uids_count = 0
        self.deleted_count = 0
        self.progress_report_at = 0
        self.skipped_uids = []

    def _prepare(self):
        self.select_next_query = ydb.ScanQuery(
            yql_text=SELECT_NEXT_OWNER_LOGS.format(self.ydb_database, self.table_name),
            parameters_types={
                '$owner_uuid': ydb.PrimitiveType.Utf8,
                '$loggers': ydb.ListType(ydb.PrimitiveType.Utf8),
                '$limit': ydb.PrimitiveType.Uint32,
                '$offset': ydb.PrimitiveType.Uint32,
            }
        )
        self.delete_query_string = DELETE_LOG_RECORD.format(self.ydb_database, self.table_name)
        self.processed_count = 0
        self.processed_uids_count = 0
        self.deleted_count = 0
        self.progress_report_at = 0
        self.skipped_uids = []

    @property
    def table_client(self):
        return self.ydb_provider.get_driver().table_client

    def start(self):
        self._prepare()
        try:
            while True:
                log_rows = self.select_log_rows()
                if not log_rows:
                    break
                for row in log_rows:
                    if self.current_uid == row.owner_uuid:
                        self.current_uid_offset += 1
                    else:
                        self.processed_uids_count += 1
                        self.current_uid = row.owner_uuid
                        self.current_uid_offset = 1

                    if row.is_pdf:
                        self.do_delete_row(row)
                        self.deleted_count += 1
                    self.processed_count += 1
                if self.processed_count - self.progress_report_at >= PROGRESS_REPORT_AFTER_ROWS:
                    self.progress_report_at = self.processed_count
                    self.progress_report()
        except Exception:
            log.exception("Patcher processing error")
            self.progress_report()
            return
        self.progress_report()
        log.info("Done")

    def select_log_rows(self):
        while True:
            if self.max_uid_offset and self.current_uid_offset >= self.max_uid_offset:
                self.jump_to_next_uid()
            query_params = {'$owner_uuid': self.current_uid, '$loggers': list(PDF_BODY_LOGGERS),
                            '$limit': self.rows_limit, '$offset': self.current_uid_offset}
            log.debug(query_params)
            rows = []
            try:
                responce_iterator = self.table_client.scan_query(self.select_next_query, query_params)
                while True:
                    try:
                        result = next(responce_iterator)
                        rows += list(result.result_set.rows)
                    except StopIteration:
                        break
            except ydb.Error:
                log.exception("select rows error with params: %s. Uid skipped", query_params)
                self.skipped_uids.append(self.current_uid)
                if len(self.skipped_uids) >= self.skipped_uids_limit:
                    raise Exception("Skipped uids limit exceded")
                self.jump_to_next_uid()
                continue
            return rows

    def jump_to_next_uid(self):
        log.info("Uid %s skipped", self.current_uid)
        self.current_uid = str(uuid.UUID(int=uuid.UUID(self.current_uid).int+1))
        self.current_uid_offset = 0

    def do_delete_row(self, log_row):
        self.executor.map(self.delete_row_sync, (log_row,))

    def delete_row_sync(self, log_row):
        query_params = {
            '$owner_uuid': log_row.owner_uuid,
            '$timestamp': log_row.timestamp,
            '$message_id': log_row.message_id,
        }
        def callee(session):
            delete_query = session.prepare(self.delete_query_string)
            session.transaction().execute(
                delete_query,
                query_params,
                commit_tx=True,
            )
        try:
            self.ydb_provider.get_session_pool().retry_operation_sync(callee)
            log.debug("deleted %s", query_params)
        except Exception:
            log.exception("delete row error with params: %s", query_params)

    def progress_report(self):
        percent = 100.0 * uuid.UUID(self.current_uid).int / MAX_UID_INT
        report = '{0:.3f}% - processed rows {1}, deleted rows {2}, processed uids {3}, current uid = {4}, offset = {5}'.format(
            percent, self.processed_count, self.deleted_count,
            self.processed_uids_count, self.current_uid, self.current_uid_offset
        )
        log.info(report)
        print(datetime.now().strftime('%H:%M:%S') + ' - ' + report)
