package main

import (
	"a.yandex-team.ru/travel/orders/proto/services/finances"
	"a.yandex-team.ru/travel/orders/proto/services/orders/admin"
	travel_commons_proto "a.yandex-team.ru/travel/proto"
	"bufio"
	"fmt"
	"github.com/gofrs/uuid"
	"github.com/golang/protobuf/ptypes"
	"github.com/golang/protobuf/ptypes/timestamp"
	"github.com/shopspring/decimal"
	"github.com/spf13/cobra"
	"google.golang.org/grpc"
	"log"
	"math"
	"os"
	"strconv"
	"strings"
	"text/tabwriter"
	"time"
)

const (
	timeFormat = "2006-01-02T15:04:05"
)

func ping(cmd *cobra.Command, args []string) error {
	channel, err := getAliveConnection()
	if err != nil {
		return err
	}
	log.Printf("Found alive channel at %s", channel.Target())
	return nil
}

func restoreDolphinOrder(cmd *cobra.Command, arg []string) error {
	client, err := getAdmin()
	if err != nil {
		return err
	}
	req := &admin.TRestoreDolphinOrderReq{}
	id := getOrderID(arg)
	req.OrderId = id
	var action string
	if cancel, _ := cmd.Flags().GetBool("cancel"); cancel {
		req.Cancel = true
		action = "отменен"
	} else {
		action = "восстановлен"
	}
	ctx, cancel := timeoutCtx()
	rsp, err := client.RestoreDolphinOrder(ctx, req, grpc.EmptyCallOption{})
	cancel()
	if err != nil {
		return err
	}

	if adminURL, exists := envToAdminMap[env]; exists {
		fmt.Printf("Заказ %s %s. Проверьте его статус по адресу `%s/%s`\n", rsp.OrderPrettyId, action, adminURL, rsp.OrderId)
	} else {
		fmt.Printf("Заказ %s %s\n", rsp.OrderPrettyId, action)
	}
	if rsp.TicketKey != "" {
		fmt.Printf("Откомментирован тикет `https://st.yandex-team.ru/%s`\n", rsp.TicketKey)
	}
	if req.Cancel && !rsp.HadPartnerOrder {
		fmt.Println("Внимание! Отменен заказ, для которого не известен партнерский ID, поэтому никаких действий " +
			"на стороне партнера не выполнялось. Убедитесь, что на стороне партнера действительно не осталось заказа!")
	}
	return nil
}

func getOrderID(arg []string) *admin.TOrderId {
	id := &admin.TOrderId{}
	if uuid.FromStringOrNil(arg[0]) != uuid.Nil {
		id.OneOfOrderIds = &admin.TOrderId_OrderId{OrderId: arg[0]}
	} else {
		id.OneOfOrderIds = &admin.TOrderId_PrettyId{PrettyId: arg[0]}
	}
	return id
}

func refundHotelOrder(cmd *cobra.Command, arg []string) error {
	if cmd.Flag("full").Changed && cmd.Flag("amount").Changed {
		return fmt.Errorf("'full' and 'amount' cannot be combined")
	}

	client, err := getAdmin()
	if err != nil {
		return err
	}
	getReq := &admin.TCalculateHotelOrderRefundReq{OrderId: getOrderID(arg)}
	ctx, cancel := timeoutCtx()
	rsp, err := client.CalculateHotelOrderRefund(ctx, getReq, grpc.EmptyCallOption{})
	cancel()
	if err != nil {
		return err
	}
	expected := float64(rsp.RefundAmountByRules.Amount) / math.Pow(10, float64(rsp.RefundAmountByRules.Precision))
	paid := float64(rsp.PaidAmount.Amount) / math.Pow(10, float64(rsp.PaidAmount.Precision))
	total := float64(rsp.TotalAmount.Amount) / math.Pow(10, float64(rsp.TotalAmount.Precision))
	skip, _ := cmd.Flags().GetBool("skip-fin-events")
	returnUserMoneyFirst, _ := cmd.Flags().GetBool("return-user-money-first")
	proportionalFinancialEvents, _ := cmd.Flags().GetBool("proportional-financial-events")
	var toRefund float64
	if full, _ := cmd.Flags().GetBool("full"); full {
		toRefund = paid
	} else if cmd.Flag("amount").Changed {
		if !rsp.Flags.CanAmountRefund && !proportionalFinancialEvents {
			return fmt.Errorf("amount refund cannot be applyied")
		}
		toRefund, _ = cmd.Flags().GetFloat64("amount")
	} else {
		toRefund = expected
	}
	fineventAction := "будут"
	returnMoneyOrder := ""
	moneyRefundMode := finances.EMoneyRefundMode_MRM_PROMO_MONEY_FIRST
	if skip {
		fineventAction = "НЕ будут"
	} else {
		if returnUserMoneyFirst {
			moneyRefundMode = finances.EMoneyRefundMode_MRM_USER_MONEY_FIRST
			returnMoneyOrder = ", сперва будут возвращены ПОЛЬЗОВАТЕЛЬСКИЕ деньги"
		} else if proportionalFinancialEvents {
			moneyRefundMode = finances.EMoneyRefundMode_MRM_PROPORTIONAL
			returnMoneyOrder = ", пользовательские и промокодные деньги вернутся пропорционально"
		} else {
			returnMoneyOrder = ", сперва будут возвращены промокодные деньги"
		}
	}
	orderURL := ""
	if adminURL, exists := envToAdminMap[env]; exists {
		orderURL = fmt.Sprintf(", %s/%s", adminURL, rsp.OrderId)
	}
	message := fmt.Sprintf("Будет оформлен возврат заказа %s (id=%s%s).\r\n"+
		"Пользователю будет возвращено %.2f руб из %.2f руб уплаченных (по правилам возврата должно быть %.2f), при общей сумме %.2f руб \r\n"+
		"Финансовые события %s сгенерированы%s.\r\n", rsp.OrderPrettyId, rsp.OrderId, orderURL, toRefund, paid, expected, total, fineventAction, returnMoneyOrder)

	var noPrompt bool
	if noPrompt, _ = cmd.Flags().GetBool("no-prompt"); !noPrompt {
		fmt.Printf("%sПродолжить (y/n) ", message)
		reader := bufio.NewReader(os.Stdin)
		readText, _, _ := reader.ReadLine()
		if strings.ToLower(string(readText)) != "y" {
			fmt.Println("Отменено")
			return nil
		}
	}
	refReq := &admin.TManualRefundHotelOrderReq{
		OrderId: getOrderID(arg),
		RefundAmount: &travel_commons_proto.TPrice{
			Currency:  travel_commons_proto.ECurrency_C_RUB,
			Amount:    int64(math.Round(toRefund * 100)),
			Precision: 2,
		},
		GenerateFinEvents: !skip,
		MoneyRefundMode:   moneyRefundMode,
	}
	ctx, cancel = timeoutCtx()
	_, err = client.ManualRefundHotelOrder(ctx, refReq, grpc.EmptyCallOption{})
	cancel()
	if err != nil {
		return err
	}
	if noPrompt {
		fmt.Print(message)
	} else {
		fmt.Println("Возврат инициирован")
	}
	return nil
}

func refundMoneyOnly(cmd *cobra.Command, arg []string) error {
	if cmd.Flag("full").Changed && cmd.Flag("amount").Changed {
		return fmt.Errorf("'full' and 'amount' cannot be combined")
	}
	if cmd.Flag("skip-fin-events").Changed && cmd.Flag("skip-user-money").Changed {
		return fmt.Errorf("skip-fin-events and skip-user-money cannot be combined")
	}

	client, err := getAdmin()
	if err != nil {
		return err
	}
	getReq := &admin.TCalculateMoneyOnlyRefundReq{OrderId: getOrderID(arg)}
	ctx, cancel := timeoutCtx()
	rsp, err := client.CalculateMoneyOnlyRefund(ctx, getReq, grpc.EmptyCallOption{})
	cancel()
	if err != nil {
		return err
	}
	remaining := float64(rsp.RemainingAmount.Amount) / math.Pow(10, float64(rsp.RemainingAmount.Precision))
	total := float64(rsp.TotalAmount.Amount) / math.Pow(10, float64(rsp.TotalAmount.Precision))
	var toRefund float64
	if full, _ := cmd.Flags().GetBool("full"); full {
		toRefund = remaining
	} else if cmd.Flag("amount").Changed {
		toRefund, _ = cmd.Flags().GetFloat64("amount")
	} else {
		return fmt.Errorf("either 'full' or 'amount' is required")
	}
	if toRefund <= 0 {
		return fmt.Errorf("refund amount should be positive")
	}
	remainingAfterRefund := remaining - toRefund

	skipFinEvents, _ := cmd.Flags().GetBool("skip-fin-events")
	skipUserMoney, _ := cmd.Flags().GetBool("skip-user-money")
	proportionalFinancialEvents, _ := cmd.Flags().GetBool("proportional-financial-events")
	fineventAction := "будут"
	if skipFinEvents {
		fineventAction = "НЕ будут"
	}
	userMoneyAction := "будут"
	if skipUserMoney {
		userMoneyAction = "НЕ будут"
	}
	moneyRefundMode := finances.EMoneyRefundMode_MRM_PROMO_MONEY_FIRST
	fineventMode := "сначала промо-деньги"
	if proportionalFinancialEvents {
		moneyRefundMode = finances.EMoneyRefundMode_MRM_PROPORTIONAL
		fineventMode = "пропорционально"
	}
	orderURL := ""
	if adminURL, exists := envToAdminMap[env]; exists {
		orderURL = fmt.Sprintf(", %s/%s", adminURL, rsp.OrderId)
	}
	message := fmt.Sprintf("Будут возвращены деньги за заказ %s (id=%s%s).\r\n"+
		"Заказ НЕ будет отменён\r\n"+
		"Будет возвращено %.2f руб из %.2f руб оставшихся, при общей сумме %.2f руб. После возврата останется %.2f руб. \r\n"+
		"Пользователю %s возвращены деньги.\r\n"+
		"Финансовые события %s сгенерированы (%s).\r\n", rsp.OrderPrettyId, rsp.OrderId, orderURL, toRefund, remaining, total, remainingAfterRefund, userMoneyAction, fineventAction, fineventMode)

	var noPrompt bool
	if noPrompt, _ = cmd.Flags().GetBool("no-prompt"); !noPrompt {
		fmt.Printf("%sПродолжить (y/n) ", message)
		reader := bufio.NewReader(os.Stdin)
		readText, _, _ := reader.ReadLine()
		if strings.ToLower(string(readText)) != "y" {
			fmt.Println("Отменено")
			return nil
		}
	}
	refReq := &admin.TManualRefundMoneyOnlyReq{
		OrderId: getOrderID(arg),
		NewInvoiceAmount: &travel_commons_proto.TPrice{
			Currency:  travel_commons_proto.ECurrency_C_RUB,
			Amount:    int64(math.Round(remainingAfterRefund * 100)),
			Precision: 2,
		},
		RefundAmount: &travel_commons_proto.TPrice{
			Currency:  travel_commons_proto.ECurrency_C_RUB,
			Amount:    int64(math.Round(toRefund * 100)),
			Precision: 2,
		},
		RefundUserMoney:   !skipUserMoney,
		GenerateFinEvents: !skipFinEvents,
		MoneyRefundMode:   moneyRefundMode,
	}
	ctx, cancel = timeoutCtx()
	_, err = client.ManualRefundMoneyOnly(ctx, refReq, grpc.EmptyCallOption{})
	cancel()
	if err != nil {
		return err
	}
	if noPrompt {
		fmt.Print(message)
	} else {
		fmt.Println("Возврат инициирован")
	}
	return nil
}

func regenerateVoucher(cmd *cobra.Command, arg []string) error {
	client, err := getAdmin()
	if err != nil {
		return err
	}
	regReq := &admin.TRegenerateVoucherReq{OrderId: getOrderID(arg)}
	ctx, cancel := timeoutCtx()
	rsp, err := client.RegenerateVoucher(ctx, regReq, grpc.EmptyCallOption{})
	cancel()
	if err != nil {
		return err
	}
	var orderURL string
	if adminURL, exists := envToAdminMap[env]; exists {
		orderURL = fmt.Sprintf(", %s/%s", adminURL, rsp.OrderId)
	}
	fmt.Printf("Для заказа %s (id=%s%s) запрошена перегенерация ваучера\r\n", rsp.GetOrderPrettyId(), rsp.GetOrderId(), orderURL)
	return nil
}

func addExtra(cmd *cobra.Command, arg []string) error {
	client, err := getAdmin()
	if err != nil {
		return err
	}
	_ = client
	amount, err := cmd.Flags().GetFloat64("amount")
	if err != nil {
		return err
	}
	amountPrice := &travel_commons_proto.TPrice{
		Currency:  travel_commons_proto.ECurrency_C_RUB,
		Amount:    int64(math.Trunc(amount * 100)),
		Precision: 2,
	}
	description, err := cmd.Flags().GetString("description")
	if err != nil {
		return err
	}

	extraReq := &admin.TAddExtraChargeReq{
		OrderId:          getOrderID(arg),
		ExtraAmount:      amountPrice,
		ExtraDescription: description,
	}
	ctx, cancel := timeoutCtx()
	rsp, err := client.AddExtraCharge(ctx, extraReq, grpc.EmptyCallOption{})
	cancel()
	if err != nil {
		return err
	}
	fmt.Printf("Для заказа %s (id=%s) запрошена доплата\r\n", rsp.GetOrderPrettyId(), rsp.GetOrderId())
	url, exists := envToPaymentURLMap[env]
	if exists {
		fmt.Printf("Отправьте пользователю ссылку на оплату: %s%s\r\n", url, rsp.GetOrderId())
	}

	return nil
}

func pauseExportTransactions(cmd *cobra.Command, arg []string) error {
	client, err := getAdmin()
	if err != nil {
		return err
	}
	pauseReq := &admin.TPauseExportBillingTransactionsToYTReq{OrderId: getOrderID(arg)}
	ctx, cancel := timeoutCtx()
	rsp, err := client.PauseExportBillingTransactionsToYt(ctx, pauseReq, grpc.EmptyCallOption{})
	cancel()
	if err != nil {
		return err
	}
	switch rsp.Status {
	case admin.EPauseExportTransactionStatus_ER_NOTHING_TO_PAUSE:
		fmt.Printf("Заказ был найден, но для него не было сформировано ни одной транзакции - останавливать нечего")
	case admin.EPauseExportTransactionStatus_EP_ALREADY_EXPORTED:
		fmt.Printf("Остановка откруток невозможна - транзакции уже были экспортированы")
	case admin.EPauseExportTransactionStatus_EP_ABOUT_TO_EXPORT:
		fmt.Printf("Остановка откруток невозможна - транзакции будут экспортированы меньше чем через 5 минут")
	case admin.EPauseExportTransactionStatus_EP_SUCCESS:
		fmt.Printf("Остановка откруток произошла успешно. Транзакций поставленных на паузу: %d", rsp.PausedTransactions)
	default:
		fmt.Printf("Неожиданный статус: %s", rsp.Status)
	}
	return nil
}

func resumeExportTransaction(cmd *cobra.Command, arg []string) error {
	transactionID, err := strconv.ParseInt(arg[0], 10, 64)
	if err != nil {
		return err
	}
	client, err := getAdmin()
	if err != nil {
		return err
	}
	resumeReq := &admin.TResumeExportBillingTransactionToYtReq{
		BillingTransactionId: transactionID,
	}
	ctx, cancel := timeoutCtx()
	rsp, err := client.ResumeExportBillingTransactionToYt(ctx, resumeReq, grpc.EmptyCallOption{})
	cancel()
	if err != nil {
		return err
	}
	fmt.Printf(rsp.ResumeResult)
	return nil
}

func listTransactions(cmd *cobra.Command, arg []string) error {
	client, err := getAdmin()
	if err != nil {
		return err
	}
	listReq := &admin.TListTransactionsAndFinEventsReq{OrderId: getOrderID(arg)}
	ctx, cancel := timeoutCtx()
	rsp, err := client.ListTransactionsAndFinEventsReq(ctx, listReq, grpc.EmptyCallOption{})
	cancel()
	if err != nil {
		return err
	}
	fmt.Printf("С заказом %s (id=%s) связаны следующие сущности:\r\n", rsp.GetOrderPrettyId(), rsp.GetOrderId())
	fmt.Printf("Financial events:\r\n")
	writer := tabwriter.NewWriter(os.Stdout, 0, 0, 2, ' ', tabwriter.AlignRight)
	_, _ = fmt.Fprintln(writer, "Id\tOriginalEventId\tOrderId\tOrderPrettyId\tOrderItemId\tBillingClientId\t"+
		"BillingContractId\tType\tAccrualAt\tPayoutAt\tPartnerAmount\tPartnerRefundAmount\tFeeAmount\t"+
		"FeeRefundAmount\tProcessed\tAccountingActAt\t")
	for _, event := range rsp.GetFinancialEvent() {
		_, _ = fmt.Fprintf(writer, "%d\t%d\t%s\t%s\t%s\t%d\t%d\t%s\t%s\t%s\t%s\t%s\t%s\t%s\t%t\t%s\t\r\n",
			event.Id, event.OriginalEventId, event.OrderId, event.OrderPrettyId, event.OrderItemId,
			event.BillingClientId, event.BillingContractId, event.Type, event.AccrualAt, event.PayoutAt,
			event.PartnerAmount, event.PartnerRefundAmount, event.FeeAmount, event.FeeRefundAmount,
			event.Processed, event.AccountingActAt)
	}
	_ = writer.Flush()
	fmt.Printf("\r\n")
	fmt.Printf("BillingTransactions:\r\n")
	_, _ = fmt.Fprintln(writer, "Id\tOriginalEventTransactionId\tSourceFinancialEventId\tTransactionType\tPaymentType\t"+
		"PartnerId\tClientId\tTrustPaymentId\tServiceOrderId\tValue\tCreatedAt\tPayoutAt\tAccountingActAt\t"+
		"ExportedToYt\tActCommitted\tYtId\tExportedToYtAt\tActCommittedAt\tPaused\tPaymentSystemType\t")
	for _, transaction := range rsp.GetBillingTransaction() {
		_, _ = fmt.Fprintf(writer, "%d\t%d\t%d\t%s\t%s\t%d\t%d\t%s\t%s\t%s\t%s\t%s\t%s\t%t\t%t\t%d\t%s\t%s\t%t\t%s\t\r\n",
			transaction.Id, transaction.OriginalTransactionId, transaction.SourceFinancialEventId,
			transaction.TransactionType, transaction.PaymentType, transaction.PartnerId, transaction.ClientId,
			transaction.TrustPaymentId, transaction.ServiceOrderId, transaction.Value, transaction.CreatedAt,
			transaction.PayoutAt, transaction.AccountingActAt, transaction.ExportedToYt, transaction.ActCommitted,
			transaction.YtId, transaction.ExportedToYtAt, transaction.ActCommittedAt, transaction.Paused,
			transaction.PaymentSystemType)
	}
	_ = writer.Flush()
	return nil
}

func startAutoPayment(cmd *cobra.Command, arg []string) error {
	client, err := getAdmin()
	if err != nil {
		return err
	}
	req := &admin.TStartAutoPaymentReq{OrderId: getOrderID(arg)}
	ctx, cancel := timeoutCtx()
	_, err = client.StartAutoPayment(ctx, req, grpc.EmptyCallOption{})
	cancel()
	if err != nil {
		return err
	}
	fmt.Printf("Автооплата инициирована, но ее результат не известен. Проверьте заказ в админке!\r\n")
	return nil
}

func stopPayments(cmd *cobra.Command, arg []string) error {
	if !cmd.Flag("client-id").Changed && !cmd.Flag("contract-id").Changed {
		return fmt.Errorf("client-id or contract-id should be specified")
	}
	if cmd.Flag("client-id").Changed && cmd.Flag("contract-id").Changed {
		return fmt.Errorf("client-id and contract-id cannot be combined")
	}
	billingClientID, err := cmd.Flags().GetInt64("client-id")
	if err != nil {
		return err
	}
	billingContractID, err := cmd.Flags().GetInt64("contract-id")
	if err != nil {
		return err
	}

	client, err := getAdmin()
	if err != nil {
		return err
	}
	stopReq := &admin.TStopPaymentsReq{
		BillingClientId:   billingClientID,
		BillingContractId: billingContractID,
	}
	ctx, cancel := timeoutCtx()
	rsp, err := client.StopPayments(ctx, stopReq, grpc.EmptyCallOption{})
	cancel()
	if err != nil {
		return err
	}
	fmt.Printf(rsp.StopResult)
	return nil
}

func resumePayments(cmd *cobra.Command, arg []string) error {
	if !cmd.Flag("client-id").Changed && !cmd.Flag("contract-id").Changed {
		return fmt.Errorf("client-id or contract-id should be specified")
	}
	if cmd.Flag("client-id").Changed && cmd.Flag("contract-id").Changed {
		return fmt.Errorf("client-id and contract-id cannot be combined")
	}
	billingClientID, err := cmd.Flags().GetInt64("client-id")
	if err != nil {
		return err
	}
	billingContractID, err := cmd.Flags().GetInt64("contract-id")
	if err != nil {
		return err
	}

	client, err := getAdmin()
	if err != nil {
		return err
	}
	resumeReq := &admin.TResumePaymentsReq{
		BillingClientId:   billingClientID,
		BillingContractId: billingContractID,
	}
	ctx, cancel := timeoutCtx()
	rsp, err := client.ResumePayments(ctx, resumeReq, grpc.EmptyCallOption{})
	cancel()
	if err != nil {
		return err
	}
	fmt.Printf(rsp.ResumeResult)
	return nil
}

func changeTransactionDates(cmd *cobra.Command, arg []string) error {
	transactionID, err := strconv.ParseInt(arg[0], 10, 64)
	if err != nil {
		return err
	}
	if !cmd.Flag("payout-at").Changed && !cmd.Flag("accounting-act-at").Changed {
		return fmt.Errorf("at least payout-at or accounting-act-at should be specified")
	}
	changeDatesReq := &admin.TChangeBillingTransactionDatesReq{
		BillingTransactionId: transactionID,
	}
	if cmd.Flag("payout-at").Changed {
		payoutAtStr, err := cmd.Flags().GetString("payout-at")
		if err != nil {
			return err
		}
		payoutAt, err := stringToProtoDate(payoutAtStr)
		if err != nil {
			return err
		}
		changeDatesReq.PayoutAt = payoutAt
	}
	if cmd.Flag("accounting-act-at").Changed {
		accountingActAtStr, err := cmd.Flags().GetString("accounting-act-at")
		if err != nil {
			return err
		}
		accountingActAt, err := stringToProtoDate(accountingActAtStr)
		if err != nil {
			return err
		}
		changeDatesReq.AccountingActAt = accountingActAt
	}
	client, err := getAdmin()
	if err != nil {
		return err
	}
	ctx, cancel := timeoutCtx()
	rsp, err := client.ChangeBillingTransactionDates(ctx, changeDatesReq, grpc.EmptyCallOption{})
	cancel()
	if err != nil {
		return err
	}
	fmt.Printf(rsp.ChangeDatesResult)
	return nil
}

func createFinancialEvent(cmd *cobra.Command, arg []string) error {
	var orderItemID string
	if uuid.FromStringOrNil(arg[0]) != uuid.Nil {
		orderItemID = arg[0]
	} else {
		return fmt.Errorf(arg[0] + " is not a valid UUID")
	}
	billingClientID, err := cmd.Flags().GetInt64("client-id")
	if err != nil {
		return err
	}
	billingContractID, err := cmd.Flags().GetInt64("contract-id")
	if err != nil {
		return err
	}
	payoutAtStr, err := cmd.Flags().GetString("payout-at")
	if err != nil {
		return err
	}
	payoutAt, err := stringToProtoDate(payoutAtStr)
	if err != nil {
		return err
	}
	accountingActAtStr, err := cmd.Flags().GetString("accounting-act-at")
	if err != nil {
		return err
	}
	accountingActAt, err := stringToProtoDate(accountingActAtStr)
	if err != nil {
		return err
	}
	financialEventTypeStr, err := cmd.Flags().GetString("type")
	if err != nil {
		return err
	}
	financialEventType := admin.EFinancialEventType_value[financialEventTypeStr]
	if financialEventType == 0 {
		return fmt.Errorf("Unknown FinancialEventType: " + financialEventTypeStr)
	}
	financialEventPaymentSchemeStr, err := cmd.Flags().GetString("payment-scheme")
	if err != nil {
		return err
	}
	financialEventPaymentScheme := admin.EFinancialEventPaymentScheme_value[financialEventPaymentSchemeStr]
	if financialEventPaymentScheme == 0 {
		return fmt.Errorf("Unknown PaymentScheme: " + financialEventPaymentSchemeStr)
	}

	createReq := &admin.TCreateFinancialEventReq{
		OrderItemId:       orderItemID,
		BillingClientId:   billingClientID,
		BillingContractId: billingContractID,
		Type:              admin.EFinancialEventType(financialEventType),
		PayoutAt:          payoutAt,
		AccountingActAt:   accountingActAt,
		PaymentScheme:     admin.EFinancialEventPaymentScheme(financialEventPaymentScheme),
	}
	if cmd.Flags().Changed("original-event-id") {
		originalEventID, err := cmd.Flags().GetInt64("original-event-id")
		if err != nil {
			return err
		}
		createReq.OriginalEventId = originalEventID
	}
	partnerAmountStr, err := cmd.Flags().GetString("partner-amount")
	if err != nil {
		return err
	}
	partnerAmount, err := decimal.NewFromString(partnerAmountStr)
	if err != nil {
		return err
	}
	feeAmountStr, err := cmd.Flags().GetString("fee-amount")
	if err != nil {
		return err
	}
	feeAmount, err := decimal.NewFromString(feeAmountStr)
	if err != nil {
		return err
	}
	switch admin.EFinancialEventType(financialEventType) {
	case admin.EFinancialEventType_EET_PAYMENT, admin.EFinancialEventType_EET_INSURANCE_PAYMENT, admin.EFinancialEventType_EET_MANUAL_PAYMENT:
		//payment
		createReq.PartnerAmount = &travel_commons_proto.TPrice{
			Currency:  travel_commons_proto.ECurrency_C_RUB,
			Amount:    partnerAmount.Coefficient().Int64(),
			Precision: -partnerAmount.Exponent(),
		}
		createReq.FeeAmount = &travel_commons_proto.TPrice{
			Currency:  travel_commons_proto.ECurrency_C_RUB,
			Amount:    feeAmount.Coefficient().Int64(),
			Precision: -feeAmount.Exponent(),
		}
	case admin.EFinancialEventType_EET_REFUND, admin.EFinancialEventType_EET_INSURANCE_REFUND, admin.EFinancialEventType_EET_MANUAL_REFUND:
		//refund
		createReq.PartnerRefundAmount = &travel_commons_proto.TPrice{
			Currency:  travel_commons_proto.ECurrency_C_RUB,
			Amount:    partnerAmount.Coefficient().Int64(),
			Precision: -partnerAmount.Exponent(),
		}
		createReq.FeeRefundAmount = &travel_commons_proto.TPrice{
			Currency:  travel_commons_proto.ECurrency_C_RUB,
			Amount:    feeAmount.Coefficient().Int64(),
			Precision: -feeAmount.Exponent(),
		}
	default:
		return fmt.Errorf("Unexpected FinancialEventType: " + financialEventTypeStr)
	}
	client, err := getAdmin()
	if err != nil {
		return err
	}
	ctx, cancel := timeoutCtx()
	rsp, err := client.CreateFinancialEvent(ctx, createReq, grpc.EmptyCallOption{})
	cancel()
	if err != nil {
		return err
	}
	fmt.Printf(rsp.CreateResult)
	return nil
}

func stringToProtoDate(dateString string) (*timestamp.Timestamp, error) {
	payoutDate, err := time.Parse(timeFormat, dateString)
	if err != nil {
		return nil, err
	}
	dateProto, err := ptypes.TimestampProto(payoutDate)
	if err != nil {
		return nil, err
	}
	return dateProto, nil
}

func moveOrder(cmd *cobra.Command, arg []string) error {
	client, err := getAdmin()
	orderID := getOrderID(arg)
	if err != nil {
		return err
	}
	clientID, err := cmd.Flags().GetInt64("client-id")
	if err != nil {
		return err
	}
	contractID, err := cmd.Flags().GetInt64("contract-id")
	if err != nil {
		return err
	}
	regReq := &admin.TMoveHotelOrderToNewContractReq{
		OrderId:    orderID,
		ClientId:   clientID,
		ContractId: contractID,
	}
	payoutDateStr, err := cmd.Flags().GetString("payout-date")
	if err != nil {
		return err
	}
	if payoutDateStr != "" {
		payoutDate, err := time.Parse(timeFormat, payoutDateStr)
		if err != nil {
			return err
		}
		payoutDateProto, err := ptypes.TimestampProto(payoutDate)
		if err != nil {
			return err
		}
		regReq.PayoutDate = payoutDateProto
	}
	ctx, cancel := timeoutCtx()
	_, err = client.MoveHotelOrderToNewContract(ctx, regReq, grpc.EmptyCallOption{})
	cancel()
	if err != nil {
		return err
	}
	fmt.Printf("Запрошен перевод заказа %s на клиента %d, договор %d\r\n", orderID, clientID, contractID)
	return nil
}

func addPlus(cmd *cobra.Command, arg []string) error {
	client, err := getAdmin()
	if err != nil {
		return err
	}
	amount, _ := cmd.Flags().GetInt64("amount")
	passportID, _ := cmd.Flags().GetString("passport-id")
	login, _ := cmd.Flags().GetString("login")
	if passportID != "" && login != "" {
		return fmt.Errorf("exactly one of 'login' or 'passport-id' should be specified")
	}
	now, _ := cmd.Flags().GetBool("now")
	allowPassportIDAutoDetection, _ := cmd.Flags().GetBool("allow-passport-id-auto-detection")
	ignoreOrderStatus, _ := cmd.Flags().GetBool("ignore-order-status")
	overridePayloadPointsInsteadOfSum, _ := cmd.Flags().GetBool("override-payload-points-instead-of-sum")
	skipPayloadUpdate, _ := cmd.Flags().GetBool("skip-payload-update")
	req := &admin.TAddPlusPointsReq{
		OrderId:                           getOrderID(arg),
		Points:                            amount,
		PassportID:                        passportID,
		Login:                             login,
		Now:                               now,
		AllowPassportIdAutoDetection:      allowPassportIDAutoDetection,
		IgnoreOrderStatus:                 ignoreOrderStatus,
		OverridePayloadPointsInsteadOfSum: overridePayloadPointsInsteadOfSum,
		SkipPayloadUpdate:                 skipPayloadUpdate,
	}
	ctx, cancel := timeoutCtx()
	defer cancel()
	rsp, err := client.AddPlusPoints(ctx, req)
	if err != nil {
		return err
	}
	if adminURL, exists := envToAdminMap[env]; exists {
		fmt.Printf("Запланировано начисление плюсов для заказа %s. Проверьте его статус по адресу `%s/%s`\n", rsp.OrderPrettyId, adminURL, rsp.OrderId)
	} else {
		fmt.Printf("Запланировано начисление плюсов для заказа %s.\n", rsp.OrderPrettyId)
	}
	return nil
}

func modifyHotelOrderDates(cmd *cobra.Command, arg []string) error {
	client, err := getAdmin()
	if err != nil {
		return err
	}
	orderID := getOrderID(arg)
	checkIn, _ := cmd.Flags().GetString("check-in")
	checkOut, _ := cmd.Flags().GetString("check-out")
	reason, _ := cmd.Flags().GetString("reason")
	req := &admin.TModifyHotelOrderDetailsInsecureReq{
		Params: &admin.TModifyHotelOrderParams{
			OrderId: orderID,
			Reason:  reason,
			Dates: &admin.TModifyHotelOrderDatesParams{
				CheckInDate:  checkIn,
				CheckOutDate: checkOut,
			},
		},
	}
	ctx, cancel := timeoutCtx()
	defer cancel()
	_, err = client.ModifyHotelOrderDetails(ctx, req)
	if err != nil {
		return err
	}
	fmt.Printf("Смена дат в заказе произведена. Перегенерируйте ваучер, если нужно. Заказ %s\r\n", orderID)
	return nil
}

func patchPartnerCommission(cmd *cobra.Command, arg []string) error {
	client, err := getAdmin()
	if err != nil {
		return err
	}
	orderID := getOrderID(arg)
	confirmedRatePct, _ := cmd.Flags().GetUint32("confirmed-rate-pct")
	refundedRatePct, _ := cmd.Flags().GetUint32("refunded-rate-pct")
	inferBillingIdsFromEvents, _ := cmd.Flags().GetBool("infer-billing-ids-from-events")
	req := &admin.TPatchPartnerCommissionReq{
		OrderId:                   orderID,
		TargetConfirmedRatePct:    confirmedRatePct,
		TargetRefundedRatePct:     refundedRatePct,
		InferBillingIdsFromEvents: inferBillingIdsFromEvents,
	}
	ctx, cancel := timeoutCtx()
	defer cancel()
	rsp, err := client.PatchPartnerCommission(ctx, req)
	if err != nil {
		return err
	}
	prefix := ""
	if !rsp.RegisteredNewEvents {
		prefix = "НЕ "
	}
	fmt.Printf("Комиссия в заказе обновлена, новые финансовые события %sсгенерированы. Заказ %s\r\n", prefix, orderID)

	return nil
}

func cancelExtra(cmd *cobra.Command, arg []string) error {
	client, err := getAdmin()
	if err != nil {
		return err
	}
	_ = client
	cancelReq := &admin.TCancelExtraChargeReq{
		OrderId: getOrderID(arg),
	}
	ctx, cancel := timeoutCtx()
	rsp, err := client.CancelExtraCharge(ctx, cancelReq, grpc.EmptyCallOption{})
	cancel()
	if err != nil {
		return err
	}
	if adminURL, exists := envToAdminMap[env]; exists {
		fmt.Printf("Для заказа %s (id=%s) запланирована отмена доплаты\r\n"+
			"Проверьте его статус по адресу `%s/%s`\r\n", rsp.GetOrderPrettyId(), rsp.GetOrderId(), adminURL, rsp.GetOrderId())
	} else {
		fmt.Printf("Для заказа %s (id=%s) запланирована отмена доплаты\r\n", rsp.GetOrderPrettyId(), rsp.GetOrderId())
	}
	return nil
}
