package main

import (
	"a.yandex-team.ru/travel/orders/proto/services/orders/admin"
	"fmt"
	"github.com/spf13/cobra"
	"os"
	"strings"
)

var allowedEnvs = map[string]bool{
	"prod":    true,
	"testing": true,
	"local":   true,
}

var envToEndpointMap = map[string]string{
	"prod":    "travel-orders-app-prod",
	"testing": "travel-orders-app-testing",
}

var envToAdminMap = map[string]string{
	"prod":    "https://travel-orders-admin.yandex-team.ru",
	"testing": "https://travel-orders-admin-testing.common.yandex-team.ru",
}

var envToPaymentURLMap = map[string]string{
	"prod":    "https://travel.yandex.ru/hotels/book/startDeferredPayment?orderId=",
	"testing": "https://travel-test.yandex.ru/hotels/book/startDeferredPayment?orderId=",
}

var envToTvmClientIDMap = map[string]uint32{
	"prod":    2002742,
	"testing": 2002740,
}

var (
	env                    string
	host                   string
	endpointSet            string
	port                   int32
	tvmSourceClientID      uint32
	tvmDestinationClientID uint32
	noTvm                  bool
	tvmClientSecret        string
	tvmTicket              string
	timeoutMs              uint32
)

type location struct {
	host string
	port int32
}

var targets []location

func configure(cmd *cobra.Command, args []string) error {
	err := configureTargets(cmd)
	if err != nil {
		return err
	}
	err = configureTvm()
	if err != nil {
		return err
	}
	return nil
}

func configureTargets(cmd *cobra.Command) error {
	if cmd.Flag("env").Changed {
		_, exists := allowedEnvs[env]
		if !exists {
			return fmt.Errorf("unsupported environment '%s'", env)
		}
		if endpointSet != "" {
			return fmt.Errorf("endpoint-set and env cannot be combined")
		}
		if host != "" || port != 0 {
			return fmt.Errorf("host/port and env cannot be combined")
		}
		if tvmDestinationClientID != 0 {
			return fmt.Errorf("tvm-dst-client-id and env cannot be combined")
		}
	} else {
		if endpointSet != "" && (host != "" || port != 0) {
			return fmt.Errorf("endpoint-set and host/port cannot be combined")
		}
	}

	if env == "local" {
		host = "localhost"
		port = 30858
	}
	if host != "" || port != 0 {
		if host == "" {
			return fmt.Errorf("please specify host too")
		}
		if port == 0 {
			port = 30858
		}
		targets = []location{
			{
				host: host,
				port: port,
			},
		}
		return nil
	}

	if endpointSet == "" {
		endpointSet = envToEndpointMap[env]
	}
	locations := discover(endpointSet)
	if len(locations) == 0 {
		return fmt.Errorf("unable to discover target location")
	}
	targets = locations
	return nil
}

func configureTvm() error {
	if !noTvm && env != "local" {
		tvmDestinationClientID = envToTvmClientIDMap[env]
		if tvmTicket == "" {
			if tvmSourceClientID == 0 {
				return fmt.Errorf("tvm-src-client-id and tvm-dst-client-id must be passed if tvm is enabled and no env is selected")
			}
			if tvmClientSecret == "" {
				if secret, exists := os.LookupEnv("TVM_CLIENT_SECRET"); exists {
					tvmClientSecret = secret
				}
			}
			if tvmClientSecret != "" {
				ticket, err := getTvmTicketWithSecret(tvmSourceClientID, tvmDestinationClientID, tvmClientSecret)
				if err != nil {
					return err
				} else {
					tvmTicket = ticket
				}
			} else {
				ticket, err := getTvmTicketViaKnife(tvmSourceClientID, tvmDestinationClientID)
				if err != nil {
					return err
				} else {
					tvmTicket = ticket
				}
			}
		}
	}
	return nil
}

type extraRefund struct {
	rule admin.EExtraRefund
}

func (e *extraRefund) String() string {
	switch e.rule {
	case admin.EExtraRefund_ER_FULL:
		return "full"
	case admin.EExtraRefund_ER_NONREF:
		return "no"
	case admin.EExtraRefund_ER_PROPORTIONAL:
		return "proportional"
	}
	return ""
}

func (e *extraRefund) Set(s string) error {
	switch s {
	case "full":
		e.rule = admin.EExtraRefund_ER_FULL
	case "no":
		e.rule = admin.EExtraRefund_ER_NONREF
	case "proportional":
		e.rule = admin.EExtraRefund_ER_PROPORTIONAL
	default:
		return fmt.Errorf("unsupported refund rule")
	}
	return nil
}

func (e *extraRefund) Type() string {
	return "no / full / proportional"
}

func main() {
	var rootCmd = &cobra.Command{
		Use:   "orders-admin",
		Short: "CLI tool to manage travel orders",
	}
	rootCmd.PersistentFlags().StringVarP(&env, "env", "e", "prod", "(prod, testing, local) target environment. Optional, but if missing requires other ways to define target")
	rootCmd.PersistentFlags().StringVar(&endpointSet, "endpoint-set", "", "target endpoint set. Should be empty if target environment is specified")
	rootCmd.PersistentFlags().StringVar(&host, "host", "", "target host. Should be empty if target environment or target endpoint set are specified")
	rootCmd.PersistentFlags().Int32VarP(&port, "port", "p", 0, "target port. Should be empty if target environment or target endpoint-set are specified")
	rootCmd.PersistentFlags().BoolVar(&noTvm, "no-tvm", false, "skip tvm ticket authentication")
	rootCmd.PersistentFlags().StringVar(&tvmClientSecret, "tvm-client-secret", "", "tvm client secret. If not specified falls back to TVM_CLIENT_SECRET env variable. If it is also missing will attempt to use ya tool tvm-knife to obtain the ticket")
	rootCmd.PersistentFlags().Uint32Var(&tvmSourceClientID, "tvm-src-client-id", 2010758, "tvm source client id")
	rootCmd.PersistentFlags().Uint32Var(&tvmDestinationClientID, "tvm-dst-client-id", 0, "tvm destination client id. Should be empty if target environment is specified")
	rootCmd.PersistentFlags().StringVar(&tvmTicket, "tvm-ticket", "", "tvm service ticket. May be empty in favor of tvm-client secret / tvm-knife")
	rootCmd.PersistentFlags().Uint32Var(&timeoutMs, "timeoutCtx", 2000, "grpc call timeoutCtx")
	rootCmd.PersistentPreRunE = configure
	rootCmd.SilenceUsage = true

	var pingCmd = &cobra.Command{
		Use:   "ping",
		Short: "Ping orchestrator",
		RunE:  ping,
	}
	rootCmd.AddCommand(pingCmd)

	var restoreDolphinCommand = &cobra.Command{
		Use:   "restore-dolphin-order <order id or pretty id>",
		Short: "Restores (or cancels) a pending dolphin order by its id or pretty id",
		RunE:  restoreDolphinOrder,
	}
	restoreDolphinCommand.Args = cobra.ExactArgs(1)
	restoreDolphinCommand.Flags().Bool("cancel", false, "cancels the order. If the order exists on partner side it will be annulled. If not, only our side of the order will be cancelled")
	rootCmd.AddCommand(restoreDolphinCommand)

	var refundHotelOrderCommand = &cobra.Command{
		Use:   "refund-hotel-order <order id or pretty id>",
		Short: "Refunds a confirmed hotel order, returning money to the payer and generating financial events",
		RunE:  refundHotelOrder,
	}
	refundHotelOrderCommand.Args = cobra.ExactArgs(1)
	refundHotelOrderCommand.Flags().Bool("full", false, "refund 100% of the order (mutually exclusive with --amount)")
	refundHotelOrderCommand.Flags().Float64("amount", 0.0, "amount to refund in RUB (mutually exclusive with --full)")
	refundHotelOrderCommand.Flags().Bool("skip-fin-events", false, "do not generate financial events")
	refundHotelOrderCommand.Flags().Bool("return-user-money-first", false, "defines the order in which promo and user money are returned, the default app-wide behaviour is returning promo money first")
	refundHotelOrderCommand.Flags().Bool("proportional-financial-events", false, "defines the order in which promo and user money are returned, the default app-wide behaviour is returning promo money first")
	refundHotelOrderCommand.Flags().Bool("no-prompt", false, "if flag is set the command will not prompt for confirmation")
	rootCmd.AddCommand(refundHotelOrderCommand)

	var refundMoneyOnlyCommand = &cobra.Command{
		Use:   "refund-money-only <order id or pretty id>",
		Short: "Returning money to the payer and generating financial events without refunding the order",
		RunE:  refundMoneyOnly,
	}
	refundMoneyOnlyCommand.Args = cobra.ExactArgs(1)
	refundMoneyOnlyCommand.Flags().Bool("full", false, "refund 100% of the order (mutually exclusive with --amount)")
	refundMoneyOnlyCommand.Flags().Float64("amount", 0.0, "amount to refund in RUB (mutually exclusive with --full)")
	refundMoneyOnlyCommand.Flags().Bool("skip-user-money", false, "do not return user money")
	refundMoneyOnlyCommand.Flags().Bool("skip-fin-events", false, "do not generate financial events")
	refundMoneyOnlyCommand.Flags().Bool("proportional-financial-events", false, "defines the order in which promo and user money are returned, the default app-wide behaviour is returning promo money first")
	refundMoneyOnlyCommand.Flags().Bool("no-prompt", false, "if flag is set the command will not prompt for confirmation")
	rootCmd.AddCommand(refundMoneyOnlyCommand)

	var regenerateVoucherCommand = &cobra.Command{
		Use:   "regenerate-voucher <order id or pretty id>",
		Short: "Regenerates voucher(s) for a given order by its id or pretty id",
		RunE:  regenerateVoucher,
	}
	regenerateVoucherCommand.Args = cobra.ExactArgs(1)
	rootCmd.AddCommand(regenerateVoucherCommand)

	var addExtraCommand = &cobra.Command{
		Use:   "add-extra <order id or pretty id>",
		Short: "Adds an extra payment to the order by adding a new financial item to the service, moving order by to AWAITS_PAYMENT state and creating a new invoice",
		RunE:  addExtra,
	}
	addExtraCommand.Flags().Float64("amount", 0.0, "amount in RUB to charge as extra")
	addExtraCommand.Flags().String("description", "", "charge description for a receipt")
	_ = addExtraCommand.MarkFlagRequired("amount")
	rootCmd.AddCommand(addExtraCommand)

	var pauseExportTransactionsToYTCommand = &cobra.Command{
		Use:   "pause-export-transactions <order id or pretty id>",
		Short: "Pauses export of Billing Transactions to YT of given Order",
		RunE:  pauseExportTransactions,
	}
	pauseExportTransactionsToYTCommand.Args = cobra.ExactArgs(1)
	rootCmd.AddCommand(pauseExportTransactionsToYTCommand)

	var listTransactionsAndFinEventsCommand = &cobra.Command{
		Use:   "list-transactions <order id or pretty id>",
		Short: "Shows all billing transactions and financial events linked to given order",
		RunE:  listTransactions,
	}
	listTransactionsAndFinEventsCommand.Args = cobra.ExactArgs(1)
	rootCmd.AddCommand(listTransactionsAndFinEventsCommand)

	var startAutoPaymentCommand = &cobra.Command{
		Use:   "start-auto-payment  <order id or pretty id>",
		Short: "Starts automatic payment with a bound card for an order with deferred payment",
		RunE:  startAutoPayment,
	}
	rootCmd.AddCommand(startAutoPaymentCommand)

	var stopPaymentsCommand = &cobra.Command{
		Use:   "stop-payments (--client-id <client_id>|--contract-id <contract_id>)",
		Short: "Stops payments for a given partner specified by clientId or contractId",
		RunE:  stopPayments,
	}
	stopPaymentsCommand.Flags().Int64("client-id", 0, "Billing ClientId (mutually exclusive with --contract-id)")
	stopPaymentsCommand.Flags().Int64("contract-id", 0, "Billing ContractId (mutually exclusive with --client-id)")
	rootCmd.AddCommand(stopPaymentsCommand)

	var resumePaymentsCommand = &cobra.Command{
		Use:   "resume-payments (--client-id <client_id>|--contract-id <contract_id>)",
		Short: "Resumes payments for a given partner specified by clientId or contractId",
		RunE:  resumePayments,
	}
	resumePaymentsCommand.Flags().Int64("client-id", 0, "Billing ClientId (mutually exclusive with --contract-id)")
	resumePaymentsCommand.Flags().Int64("contract-id", 0, "Billing ContractId (mutually exclusive with --client-id)")
	rootCmd.AddCommand(resumePaymentsCommand)

	var resumeExportTransactionToYTCommand = &cobra.Command{
		Use:   "resume-export-transaction <billingTransaction id>",
		Short: "Resumes export of Billing Transaction to YT",
		RunE:  resumeExportTransaction,
	}
	resumeExportTransactionToYTCommand.Args = cobra.ExactArgs(1)
	rootCmd.AddCommand(resumeExportTransactionToYTCommand)

	var changeBillingTransactionDatesCommand = &cobra.Command{
		Use:   "change-transaction-dates <billingTransaction id> [--payout-at <yyyy-MM-ddThh:mm:ss>] [--accounting-act-at <yyyy-MM-ddThh:mm:ss>]",
		Short: "Change PayoutAt and/or AccountingActAt of paused Billing Transaction",
		RunE:  changeTransactionDates,
	}
	changeBillingTransactionDatesCommand.Args = cobra.ExactArgs(1)
	changeBillingTransactionDatesCommand.Flags().String("payout-at", "", "PayoutAt in yyyy-MM-ddThh:mm:ss")
	changeBillingTransactionDatesCommand.Flags().String("accounting-act-at", "", "AccountingActAt in yyyy-MM-ddThh:mm:ss")
	rootCmd.AddCommand(changeBillingTransactionDatesCommand)

	var createFinancialEventCommand = &cobra.Command{
		Use:   "create-fin-event <orderItem id> [--original-event-id <event_id>] --type <event_type> --client-id <client_id> --contract-id <contract_id> --payout-at <yyyy-MM-ddThh:mm:ss> --accounting-act-at <yyyy-MM-ddThh:mm:ss> --partner-amount <partner_amount> --fee-amount <fee_amount> --payment-scheme <FEPS_SCHEME>",
		Short: "Creates FinancialEvent for a given OrderItem",
		RunE:  createFinancialEvent,
	}
	createFinancialEventCommand.Args = cobra.ExactArgs(1)
	createFinancialEventCommand.Flags().Int64("original-event-id", 0, "Id of original FinancialEvent")
	var typesDescription strings.Builder
	for key, value := range admin.EFinancialEventType_name {
		if key > 0 {
			typesDescription.WriteString(value + "|")
		}
	}
	var paymentSchemeDescription strings.Builder
	for key, value := range admin.EFinancialEventPaymentScheme_name {
		if key > 0 {
			paymentSchemeDescription.WriteString(value + "|")
		}
	}
	createFinancialEventCommand.Flags().String("type", "", typesDescription.String())
	createFinancialEventCommand.Flags().Int64("client-id", 0, "Billing client id")
	createFinancialEventCommand.Flags().Int64("contract-id", 0, "Billing contract id")
	createFinancialEventCommand.Flags().String("payout-at", "", "PayoutAt in yyyy-MM-ddThh:mm:ss")
	createFinancialEventCommand.Flags().String("accounting-act-at", "", "AccountingActAt in yyyy-MM-ddThh:mm:ss")
	createFinancialEventCommand.Flags().String("partner-amount", "", "Partner (refund) amount in Rubles")
	createFinancialEventCommand.Flags().String("fee-amount", "", "Fee (refund) amount in Rubles")
	createFinancialEventCommand.Flags().String("payment-scheme", "", paymentSchemeDescription.String())
	_ = createFinancialEventCommand.MarkFlagRequired("type")
	_ = createFinancialEventCommand.MarkFlagRequired("client-id")
	_ = createFinancialEventCommand.MarkFlagRequired("contract-id")
	_ = createFinancialEventCommand.MarkFlagRequired("payout-at")
	_ = createFinancialEventCommand.MarkFlagRequired("accounting-act-at")
	_ = createFinancialEventCommand.MarkFlagRequired("partner-amount")
	_ = createFinancialEventCommand.MarkFlagRequired("fee-amount")
	_ = createFinancialEventCommand.MarkFlagRequired("payment-scheme")
	rootCmd.AddCommand(createFinancialEventCommand)

	var moveOrderCommand = &cobra.Command{
		Use:   "move-order <order id or pretty id>",
		Short: "Moves HotelOrder to new (client,contract) generating compensating financial events for old client and confirm financial events for new client",
		RunE:  moveOrder,
	}
	moveOrderCommand.Flags().Int64("client-id", 0, "destination ClientId for HotelOrder")
	moveOrderCommand.Flags().Int64("contract-id", 0, "destination ContractId for HotelOrder")
	moveOrderCommand.Flags().String("payout-date", "", "payout date")
	_ = moveOrderCommand.MarkFlagRequired("client-id")
	_ = moveOrderCommand.MarkFlagRequired("contract-id")
	rootCmd.AddCommand(moveOrderCommand)

	var addPlusCommand = &cobra.Command{
		Use:   "add-plus <order id or pretty-id> --amount <top-up amount> [--passport-id <receiving user passport id>] [--login <order-authorized login>] [--now]",
		Short: "Adds plus points for a given order to target user",
		RunE:  addPlus,
	}
	addPlusCommand.Flags().Int64("amount", 0, "Amount of points to add")
	addPlusCommand.Flags().String("passport-id", "", "passport identifier of a user to receive the top up")
	addPlusCommand.Flags().String("login", "", "login of a user to receive the top up. The user has to authorized in th order.")
	addPlusCommand.Flags().Bool("now", false, "flag to indicate that the top-up should happen immediately, without waiting for guest to check out")
	addPlusCommand.Flags().Bool("allow-passport-id-auto-detection", false, "flag to allow passport id detection based on data in order (for users with plus during order creation)")
	addPlusCommand.Flags().Bool("ignore-order-status", false, "flag to skip checking of order status during topup")
	addPlusCommand.Flags().Bool("override-payload-points-instead-of-sum", false, "flag to override plus points in order payload (default behaviour is to add them)")
	addPlusCommand.Flags().Bool("skip-payload-update", false, "flag to skip payload update")
	_ = addPlusCommand.MarkFlagRequired("amount")
	rootCmd.AddCommand(addPlusCommand)

	var modifyHotelOrderDatesCommand = &cobra.Command{
		Use:   "modify-hotel-order-dates <order id or pretty-id> --check-in <check-in date> --check-out <check-out date>  --reason <reason>",
		Short: "Modify hotel order dates",
		RunE:  modifyHotelOrderDates,
	}
	modifyHotelOrderDatesCommand.Flags().String("check-in", "", "check-in date")
	modifyHotelOrderDatesCommand.Flags().String("check-out", "", "check-out date")
	modifyHotelOrderDatesCommand.Flags().String("reason", "", "reason")
	_ = modifyHotelOrderDatesCommand.MarkFlagRequired("check-in")
	_ = modifyHotelOrderDatesCommand.MarkFlagRequired("check-out")
	_ = modifyHotelOrderDatesCommand.MarkFlagRequired("reason")
	rootCmd.AddCommand(modifyHotelOrderDatesCommand)

	var patchPartnerCommissionCommand = &cobra.Command{
		Use:   "patch-partner-commission <order id or pretty-id> --confirmed-rate-pct <target confirmed-rate (percent)> --refunded-rate-pct <target refunded-rate (percent)>  --infer-billing-ids-from-events <use billing ids from events instead of agreement>",
		Short: "Patch hotel order partner commission",
		RunE:  patchPartnerCommission,
	}
	patchPartnerCommissionCommand.Flags().Uint32("confirmed-rate-pct", 0, "target confirmed-rate (percent)")
	patchPartnerCommissionCommand.Flags().Uint32("refunded-rate-pct", 0, "target refunded-rate (percent)")
	patchPartnerCommissionCommand.Flags().Bool("infer-billing-ids-from-events", false, "use billing ids from events instead of agreement")
	_ = patchPartnerCommissionCommand.MarkFlagRequired("confirmed-rate-pct")
	_ = patchPartnerCommissionCommand.MarkFlagRequired("refunded-rate-pct")
	rootCmd.AddCommand(patchPartnerCommissionCommand)

	var cancelExtraCommand = &cobra.Command{
		Use:   "cancel-extra <order id or pretty id>",
		Short: "Cancels a previously added extra payment of the order",
		RunE:  cancelExtra,
	}
	rootCmd.AddCommand(cancelExtraCommand)

	_ = rootCmd.Execute()
}
