package main

import (
	"bufio"
	"context"
	"encoding/json"
	"fmt"
	"google.golang.org/protobuf/types/known/timestamppb"
	"io"
	"log"
	"math"
	"os"
	"time"

	"github.com/golang/protobuf/ptypes"
	"github.com/golang/protobuf/ptypes/wrappers"
	"github.com/spf13/cobra"

	"a.yandex-team.ru/library/go/core/xerrors"
	math2 "a.yandex-team.ru/library/go/x/math"
	ordercommons "a.yandex-team.ru/travel/orders/proto"
	pb "a.yandex-team.ru/travel/orders/proto/services/promo"
	travel_commons_proto "a.yandex-team.ru/travel/proto"
)

const requestBatchSizeMax = 1000
const retriesCountMax = 3
const loggingPeriod = time.Second * 60
const idlePeriod = time.Second * 30

var moscowLoc, _ = time.LoadLocation("Europe/Moscow")

func castToProtobufDate(dateAsString string, startOfDay bool) *timestamppb.Timestamp {
	if dateAsString != "" {
		date, err := time.Parse("2006-01-02", dateAsString)
		if err != nil {
			log.Fatalf("Error parsing valid-from: %v", dateAsString)
		}
		if startOfDay {
			date = time.Date(date.Year(), date.Month(), date.Day(),
				0, 0, 0, 0, moscowLoc)
		} else {
			date = time.Date(date.Year(), date.Month(), date.Day(),
				23, 59, 59, 0, moscowLoc)
		}
		result, err := ptypes.TimestampProto(date)
		if err != nil {
			log.Fatalf("Error casting date to protobuf format: %v", dateAsString)
		}
		return result
	}
	return nil
}

func createPromoAction(cmd *cobra.Command, args []string) error {
	conn, client, err := getPromoCodeAdmin()
	if err != nil {
		return err
	}
	defer conn.Close()
	promoActionName, _ := cmd.Flags().GetString("action-name")
	userTypeRestrictionTypeStr, _ := cmd.Flags().GetString("user-type-restriction")
	additive, _ := cmd.Flags().GetBool("additive")
	validFrom, _ := cmd.Flags().GetString("valid-from")
	validTill, _ := cmd.Flags().GetString("valid-till")
	minTotalCost, _ := cmd.Flags().GetFloat64("min-total-cost")
	nominal, _ := cmd.Flags().GetFloat64("nominal")
	nominalTypeStr, _ := cmd.Flags().GetString("nominal-type")
	maxUsageCount, _ := cmd.Flags().GetInt32("max-usage-count")
	maxActivations, _ := cmd.Flags().GetInt32("max-activations")
	fixedDays, _ := cmd.Flags().GetInt64("fixed-days")
	generationTypeStr, _ := cmd.Flags().GetString("generation-type")
	prefix, _ := cmd.Flags().GetString("prefix")
	suffix, _ := cmd.Flags().GetString("suffix")
	maxConfirmedHotelOrders, _ := cmd.Flags().GetInt32("max-confirmed-hotel-orders")
	budget, _ := cmd.Flags().GetInt64("budget")
	maxNominalDiscount, _ := cmd.Flags().GetFloat64("max-nominal-discount")

	hotelRestrictionPartnerStr, _ := cmd.Flags().GetString("hotel-restriction-partner")
	hotelRestrictionOriginalID, _ := cmd.Flags().GetString("hotel-restriction-original-id")

	hotelRestrictionPartner, ok := ordercommons.EServiceType_value[hotelRestrictionPartnerStr]
	if !ok {
		log.Fatalf("Unknown hotel restriction partner")
	}

	nominalType, ok := ordercommons.EPromoCodeNominalType_value[nominalTypeStr]
	if !ok || nominalType == 0 {
		log.Fatalf("Invalid nominalType: %v", nominalTypeStr)
	}

	if maxUsageCount < 0 {
		log.Fatalf("Max usage count must be positive. %v given", maxUsageCount)
	}

	userTypeRestriction, ok := pb.EUserTypeRestriction_value[userTypeRestrictionTypeStr]
	if !ok {
		log.Fatalf("Unknown user type restriction")
	}

	generationType, ok := pb.EPromoCodeGenerationType_value[generationTypeStr]
	if !ok || generationType == 0 {
		log.Fatalf("Invalid generationType: %v", generationTypeStr)
	}

	req := &pb.TCreatePromoActionReq{
		ActionName:             promoActionName,
		Budget:                 budget,
		UserTypeRestriction:    pb.EUserTypeRestriction(userTypeRestriction),
		AddsUpWithOtherActions: additive,
		Nominal:                nominal,
		NominalType:            ordercommons.EPromoCodeNominalType(nominalType),
		MaxUsageCount:          maxUsageCount,
		MaxActivations:         maxActivations,
		FixedDaysDuration:      fixedDays,
		GenerationType:         pb.EPromoCodeGenerationType(generationType),
		Prefix:                 prefix,
		Suffix:                 suffix,
	}

	req.ActionValidFrom = castToProtobufDate(validFrom, true)
	req.ActionValidTill = castToProtobufDate(validTill, false)

	if minTotalCost > 0 {
		req.MinTotalCost = &travel_commons_proto.TPrice{
			Amount:    int64(math.Ceil(minTotalCost * 100)),
			Currency:  travel_commons_proto.ECurrency_C_RUB,
			Precision: 2,
		}
	}
	if maxConfirmedHotelOrders >= 0 {
		req.MaxConfirmedHotelOrders = &wrappers.Int32Value{Value: maxConfirmedHotelOrders}
	}

	if hotelRestrictionOriginalID != "" {
		req.HotelRestrictions = []*pb.THotelRestriction{
			{
				ServiceType: ordercommons.EServiceType(hotelRestrictionPartner),
				OriginalId:  hotelRestrictionOriginalID,
			},
		}
	}

	if maxNominalDiscount > 0 {
		req.MaxNominalDiscount = &travel_commons_proto.TPrice{
			Amount:    int64(math.Ceil(maxNominalDiscount * 100)),
			Currency:  travel_commons_proto.ECurrency_C_RUB,
			Precision: 2,
		}
	}

	ctx, cancel := context.WithTimeout(context.Background(), time.Millisecond*time.Duration(timeoutMs))
	defer cancel()
	rsp, err := client.CreatePromoAction(ctx, req)
	if err != nil {
		log.Fatalf("Error creating promo action with gen params %v, %v", err, ctx)
		return err
	}
	fmt.Printf("Created promo action with id %s, %s\n", rsp.Id, rsp.Name)
	return nil
}

func createPromoCode(cmd *cobra.Command, args []string) error {
	conn, client, err := getPromoCodeAdmin()
	if err != nil {
		return err
	}
	defer conn.Close()
	actionID, _ := cmd.Flags().GetString("action-id")
	code, _ := cmd.Flags().GetString("code")
	nominalType, _ := cmd.Flags().GetString("nominal-type")
	nominal, _ := cmd.Flags().GetFloat64("nominal")
	maxUsageCount, _ := cmd.Flags().GetInt32("max-usage-count")
	maxActivations, _ := cmd.Flags().GetInt32("max-activations")
	validFrom, _ := cmd.Flags().GetString("valid-from")
	validTill, _ := cmd.Flags().GetString("valid-till")
	passportID, _ := cmd.Flags().GetString("passport-id")

	val, ok := ordercommons.EPromoCodeNominalType_value[nominalType]
	if !ok || val == 0 {
		log.Fatalf("Invalid nominalType: %v", nominalType)
	}

	if maxUsageCount < 0 {
		log.Fatalf("Max usage count must be positive. %v given", maxUsageCount)
	}

	req := &pb.TCreatePromoCodeReq{
		PromoActionId:  actionID,
		Code:           code,
		Nominal:        nominal,
		NominalType:    ordercommons.EPromoCodeNominalType(val),
		MaxUsageCount:  maxUsageCount,
		MaxActivations: maxActivations,
		ValidFrom:      castToProtobufDate(validFrom, true),
		ValidTill:      castToProtobufDate(validTill, false),
	}
	if passportID != "" {
		req.PassportId = []string{passportID}
	}
	ctx, cancel := context.WithTimeout(context.Background(), time.Millisecond*time.Duration(timeoutMs))
	defer cancel()
	rsp, err := client.CreatePromoCode(ctx, req)
	if err != nil {
		log.Fatalf("Error creating promo promo code %v, %v", err, ctx)
	}
	fmt.Printf("Created promo code with id %s\n", rsp.Id)
	return nil
}

func resetPromoCodeActivation(cmd *cobra.Command, args []string) error {
	conn, client, err := getPromoCodeAdmin()
	if err != nil {
		return err
	}
	defer conn.Close()
	passportID, _ := cmd.Flags().GetString("passport-id")
	code, _ := cmd.Flags().GetString("code")

	req := &pb.TResetPromoCodeActivationReq{
		PassportId: passportID,
		Code:       code,
	}
	ctx, cancel := context.WithTimeout(context.Background(), time.Millisecond*time.Duration(timeoutMs))
	defer cancel()
	rsp, err := client.ResetPromoCodeActivation(ctx, req)
	if err != nil {
		log.Fatalf("Error reseting promo code activation %v, %v", err, ctx)
	}
	fmt.Printf("Promo code activation id %s reset\n", rsp.PromoCodeActivationId)
	return nil
}

func createPromoCodesBatch(cmd *cobra.Command, args []string) error {
	conn, client, err := getPromoCodeAdmin()
	if err != nil {
		return err
	}
	defer conn.Close()
	actionID, _ := cmd.Flags().GetString("action-id")
	actionName, _ := cmd.Flags().GetString("action-name")
	batchSize, _ := cmd.Flags().GetInt32("batch-size")
	forDateStr, _ := cmd.Flags().GetString("for-date")
	outFile, _ := cmd.Flags().GetString("out-file")
	showCodesOnly, _ := cmd.Flags().GetBool("show-codes-only")

	if actionID == "" && actionName == "" {
		log.Fatalf("Either action ID or name must be provided")
	}

	if batchSize < 0 {
		log.Fatalf("Batch size must be positive. %v given", batchSize)
	}

	req := &pb.TCreatePromoCodesBatchReq{
		PromoActionId:   actionID,
		PromoActionName: actionName,
		ForDate:         castToProtobufDate(forDateStr, true),
	}

	var w io.Writer
	if outFile != "" {
		f, err := os.Create(outFile)
		if err != nil {
			log.Fatalf("Failed to create file %s: %v", outFile, err)
		}
		bw := bufio.NewWriter(f)
		defer func() {
			err := bw.Flush()
			if err != nil {
				return
			}
			err = f.Close()
			if err != nil {
				return
			}
		}()
		w = bw
	} else {
		w = os.Stdout
	}

	promoCodesToCreateTotal := int(batchSize)
	promoCodesCreated := 0
	retriesLeft := retriesCountMax
	lastReported := time.Now()
	for promoCodesCreated < promoCodesToCreateTotal {
		promoCodesToCreateLeft := promoCodesToCreateTotal - promoCodesCreated
		req.BatchSize = int32(math2.MinInt(promoCodesToCreateLeft, requestBatchSizeMax))
		count, err := createSingleBatch(client, req, w, showCodesOnly)
		if err != nil {
			log.Print(err)
			retriesLeft -= 1
			if retriesLeft == 0 {
				log.Fatalf("Error creating batch of promo codes %v", err)
			}
			time.Sleep(idlePeriod)
			continue
		}
		retriesLeft = retriesCountMax
		promoCodesCreated += count

		now := time.Now()
		if now.Sub(lastReported) > loggingPeriod {
			log.Printf("%d promo codes created\n", promoCodesCreated)
			lastReported = now
		}
	}

	log.Printf("%d promo codes created\n", promoCodesCreated)
	log.Print("All done\n")
	return nil
}

func createSingleBatch(
	client pb.PromoCodesOperatorManagementInterfaceV1Client,
	req *pb.TCreatePromoCodesBatchReq,
	w io.Writer,
	showCodesOnly bool,
) (int, error) {
	ctx, cancel := context.WithTimeout(context.Background(), time.Millisecond*time.Duration(timeoutMs))
	defer cancel()
	rsp, err := client.CreatePromoCodesBatch(ctx, req)
	if err != nil {
		return 0, xerrors.Errorf("Error creating batch of promo codes %v, %v", err, ctx)
	}

	for _, promoCode := range rsp.PromoCode {
		if showCodesOnly {
			_, err := fmt.Fprint(w, promoCode.Code, "\n")
			if err != nil {
				return 0, err
			}
		} else {
			_, err := fmt.Fprint(w, promoCode, "\n")
			if err != nil {
				return 0, err
			}
		}
	}
	return len(rsp.PromoCode), nil
}

func getPromoActionDetails(cmd *cobra.Command, args []string) error {
	conn, client, err := getPromoCodeAdmin()
	if err != nil {
		return err
	}
	defer conn.Close()
	actionID, _ := cmd.Flags().GetString("action-id")

	req := &pb.TGetPromoActionReq{
		PromoActionId: actionID,
	}
	ctx, cancel := context.WithTimeout(context.Background(), time.Millisecond*time.Duration(timeoutMs))
	defer cancel()
	rsp, err := client.GetPromoActionDetails(ctx, req)
	if err != nil {
		log.Fatalf("Error getting promo action details %v, %v", err, ctx)
	}
	rspJSON, _ := json.MarshalIndent(rsp, "  ", "    ")
	fmt.Println(string(rspJSON))
	return nil
}

func getPromoCodeActivationsCount(cmd *cobra.Command, args []string) error {
	codeID, _ := cmd.Flags().GetString("code-id")
	code, _ := cmd.Flags().GetString("code")

	if codeID == "" && code == "" {
		log.Fatalf("Either code-id or code must be provided")
	}
	if codeID != "" && code != "" {
		log.Fatalf("Only one of code-id or code must be provided")
	}

	conn, client, err := getPromoCodeAdmin()
	if err != nil {
		return err
	}
	defer conn.Close()

	req := &pb.TGetPromoCodeActivationsCountReq{
		CodeId: codeID,
		Code:   code,
	}
	ctx, cancel := context.WithTimeout(context.Background(), time.Millisecond*time.Duration(timeoutMs))
	defer cancel()
	rsp, err := client.GetPromoCodeActivationsCount(ctx, req)
	if err != nil {
		log.Fatalf("Error getting promo code activations count %v, %v", err, ctx)
	}
	fmt.Printf("Promo code activations: %d\n", rsp.TotalCount)
	return nil
}

func blacklistPromoCode(cmd *cobra.Command, args []string) error {
	codeID, _ := cmd.Flags().GetString("code-id")
	code, _ := cmd.Flags().GetString("code")

	if codeID == "" && code == "" {
		log.Fatalf("Either code-id or code must be provided")
	}
	if codeID != "" && code != "" {
		log.Fatalf("Only one of code-id or code must be provided")
	}

	conn, client, err := getPromoCodeAdmin()
	if err != nil {
		return err
	}
	defer conn.Close()

	req := &pb.TBlacklistPromoCodeReq{
		CodeId: codeID,
		Code:   code,
	}
	ctx, cancel := context.WithTimeout(context.Background(), time.Millisecond*time.Duration(timeoutMs))
	defer cancel()
	_, err = client.BlacklistPromoCode(ctx, req)
	if err != nil {
		log.Fatalf("Error blacklisting promo code %v, %v", err, ctx)
	}
	fmt.Printf("Promo code blacklisted successfully")
	return nil
}
