package main

import (
	"a.yandex-team.ru/infra/yp_service_discovery/golang/resolver"
	"a.yandex-team.ru/infra/yp_service_discovery/golang/resolver/httpresolver"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmauth"
	"a.yandex-team.ru/travel/orders/proto/services/orders"
	"a.yandex-team.ru/travel/orders/proto/services/promo"
	"context"
	"fmt"
	"google.golang.org/grpc"
	"google.golang.org/grpc/metadata"
	"log"
	"os/exec"
	"strings"
	"time"
)

func getTvmTicketViaKnife(src uint32, dst uint32) (string, error) {
	cmd := exec.Command("ya", "tool", "tvm-knife", "get_service_ticket", "sshkey", "--src", fmt.Sprint(src), "--dst", fmt.Sprint(dst))
	out, err := cmd.Output()
	if err != nil {
		e, cast := err.(*exec.ExitError)
		if cast {
			return "", fmt.Errorf("error while getting tvm ticket with tvm-knife: %s", string(e.Stderr))
		} else {
			return "", err
		}
	}
	trimmed := strings.Trim(string(out), "\n\r ")
	return trimmed, err
}

func getTvmTickerWithSecret(src uint32, dst uint32, secret string) (string, error) {
	settings := tvmauth.TvmAPISettings{
		SelfID:               tvm.ClientID(src),
		ServiceTicketOptions: tvmauth.NewIDsOptions(secret, []tvm.ClientID{tvm.ClientID(dst)}),
	}
	c, err := tvmauth.NewAPIClient(settings, &nop.Logger{})
	if err != nil {
		return "", err
	}
	return c.GetServiceTicketForID(context.Background(), tvm.ClientID(dst))
}

func tvmHeaderClientInterceptor(serviceTicket string) func(
	ctx context.Context,
	method string,
	req interface{},
	reply interface{},
	cc *grpc.ClientConn,
	invoker grpc.UnaryInvoker,
	opts ...grpc.CallOption) error {
	return func(
		ctx context.Context,
		method string,
		req interface{},
		reply interface{},
		cc *grpc.ClientConn,
		invoker grpc.UnaryInvoker,
		opts ...grpc.CallOption,
	) error {
		// Logic before invoking the invoker
		// Calls the invoker to execute RPC
		newCtx := metadata.AppendToOutgoingContext(ctx, "x-ya-service-ticket", serviceTicket)
		err := invoker(newCtx, method, req, reply, cc, opts...)
		// Logic after invoking the invoker
		return err
	}
}

func yandexUIDHeaderClientInterceptor(yandexUID string) func(
	ctx context.Context,
	method string,
	req interface{},
	reply interface{},
	cc *grpc.ClientConn,
	invoker grpc.UnaryInvoker,
	opts ...grpc.CallOption) error {
	return func(
		ctx context.Context,
		method string,
		req interface{},
		reply interface{},
		cc *grpc.ClientConn,
		invoker grpc.UnaryInvoker,
		opts ...grpc.CallOption,
	) error {
		// Logic before invoking the invoker
		// Calls the invoker to execute RPC
		newCtx := metadata.AppendToOutgoingContext(ctx, "x-ya-yandexUID", yandexUID)
		err := invoker(newCtx, method, req, reply, cc, opts...)
		// Logic after invoking the invoker
		return err
	}
}

func getAliveConnection() (*grpc.ClientConn, error) {
	for _, target := range targets {
		var opts []grpc.DialOption

		opts = append(opts, grpc.WithInsecure())
		opts = append(opts, grpc.WithUnaryInterceptor(yandexUIDHeaderClientInterceptor("123")))

		if tvmTicket != "" {
			opts = append(opts, grpc.WithUnaryInterceptor(tvmHeaderClientInterceptor(tvmTicket)))
		}

		conn, err := grpc.Dial(fmt.Sprintf("%s:%d", target.host, target.port), opts...)
		if err != nil {
			continue
		}
		client := orders.NewHADiscoveryInterfaceV1Client(conn)
		pingErr := ping(client)
		if pingErr == nil {
			return conn, nil
		}
	}
	return nil, fmt.Errorf("no alive channels present")
}

func ping(client orders.HADiscoveryInterfaceV1Client) error {
	ctx, cancel := context.WithTimeout(context.Background(), time.Millisecond*time.Duration(timeoutMs))
	defer cancel()
	_, pingErr := client.Ping(ctx, &orders.TPingRpcReq{}, grpc.WaitForReady(true))
	return pingErr
}

func discover(endpointSet string) []location {
	var results []location
	r, err := httpresolver.New()
	if err != nil {
		panic(err)
	}

	ctx := context.Background()
	for _, cluster := range resolver.AvailableClusters {
		resp, err := r.ResolveEndpoints(ctx, cluster, endpointSet)
		if err != nil {
			log.Println(fmt.Errorf("unable to resolve endpoints for cluster %s: %w", cluster, err))
			continue
		}
		if resp.ResolveStatus == resolver.StatusEndpointOK {
			for _, ep := range resp.EndpointSet.Endpoints {
				results = append(results, location{
					host: ep.FQDN,
					port: ep.Port,
				})
			}
		}
	}
	return results
}

func getPromoCodeAdmin() (*grpc.ClientConn, promo.PromoCodesOperatorManagementInterfaceV1Client, error) {
	conn, err := getAliveConnection()
	if err != nil {
		return nil, nil, err
	}
	return conn, promo.NewPromoCodesOperatorManagementInterfaceV1Client(conn), nil
}
