package main

import (
	"fmt"
	"log"
	"os"

	ordercommons "a.yandex-team.ru/travel/orders/proto"
	"a.yandex-team.ru/travel/orders/proto/services/promo"
	"github.com/spf13/cobra"
)

var (
	env                    string
	host                   string
	endpointSet            string
	port                   int32
	tvmSourceClientID      uint32
	tvmDestinationClientID uint32
	noTvm                  bool
	tvmClientSecret        string
	tvmTicket              string
	timeoutMs              uint32
)

var allowedEnvs = map[string]bool{
	"prod":    true,
	"testing": true,
	"local":   true,
}

var envToEndpointMap = map[string]string{
	"prod":    "travel-orders-app-prod",
	"testing": "travel-orders-app-testing",
}

var envToTvmClientIDMap = map[string]uint32{
	"prod":    2002742,
	"testing": 2002740,
}

type location struct {
	host string
	port int32
}

var targets []location

func configure(cmd *cobra.Command, args []string) error {
	if cmd.Flag("env").Changed {
		_, exists := allowedEnvs[env]
		if !exists {
			return fmt.Errorf("unsupported environment '%s'", env)
		}
		if endpointSet != "" {
			return fmt.Errorf("endpoint-set and env cannot be combined")
		}
		if host != "" || port != 0 {
			return fmt.Errorf("host/port and env cannot be combined")
		}
		if tvmDestinationClientID != 0 {
			return fmt.Errorf("tvm-dst-client-id and env cannot be combined")
		}
	}

	if env != "dev" {
		endpointSet = envToEndpointMap[env]
		tvmDestinationClientID = envToTvmClientIDMap[env]
	}

	if cmd.Flag("endpoint-set").Changed {
		if host != "" || port != 0 {
			return fmt.Errorf("host/port and endpoint-set cannot be combined")
		}
	}
	if endpointSet != "" {
		locations := discover(endpointSet)
		if len(locations) == 0 {
			return fmt.Errorf("unable to discover target location")
		}
		targets = locations
	} else {
		if env == "local" {
			host = "localhost"
			port = 30858
		}
		if host == "" || port == 0 {
			return fmt.Errorf("host and port must be defined if no env or endpoint-set are provided")
		}
		targets = []location{
			{
				host: host,
				port: port,
			},
		}
	}
	if !noTvm && env != "local" {
		if tvmTicket == "" {
			if tvmSourceClientID == 0 || tvmDestinationClientID == 0 {
				return fmt.Errorf("tvm-src-client-id and tvm-dst-client-id must be passed if tvm is enabled and no env is selected")
			}
			if tvmClientSecret == "" {
				if secret, exists := os.LookupEnv("TVM_CLIENT_SECRET"); exists {
					tvmClientSecret = secret
				}
			}
			if tvmClientSecret != "" {
				ticket, err := getTvmTickerWithSecret(tvmSourceClientID, tvmDestinationClientID, tvmClientSecret)
				if err != nil {
					return err
				} else {
					tvmTicket = ticket
				}
			} else {
				ticket, err := getTvmTicketViaKnife(tvmSourceClientID, tvmDestinationClientID)
				if err != nil {
					return err
				} else {
					tvmTicket = ticket
				}
			}
		}
	}
	return nil
}

func main() {

	var rootCmd = &cobra.Command{
		Use:   "Promo codes admin",
		Short: "CLI tool to manage promo codes",
	}
	rootCmd.PersistentFlags().StringVarP(&env, "env", "e", "prod", "(prod, testing, dev) target environment. Optional, but if missing requires other ways to define target")
	rootCmd.PersistentFlags().StringVar(&endpointSet, "endpoint-set", "", "target endpoint set. Should be empty if target environment is specified")
	rootCmd.PersistentFlags().StringVar(&host, "host", "", "target host. Should be empty if target environment or target endpoint set are specified")
	rootCmd.PersistentFlags().Int32VarP(&port, "port", "p", 0, "target port. Should be empty if target environment or target endpoint-set are specified")
	rootCmd.PersistentFlags().BoolVar(&noTvm, "no-tvm", false, "skip tvm ticket authentication")
	rootCmd.PersistentFlags().StringVar(&tvmClientSecret, "tvm-client-secret", "", "tvm client secret. If not specified falls back to TVM_CLIENT_SECRET env variable. If it is also missing will attempt to use ya tool tvm-knife to obtain the ticket")
	rootCmd.PersistentFlags().Uint32Var(&tvmSourceClientID, "tvm-src-client-id", 2010758, "tvm source client id")
	rootCmd.PersistentFlags().Uint32Var(&tvmDestinationClientID, "tvm-dst-client-id", 0, "tvm destination client id. Should be empty if target environment is specified")
	rootCmd.PersistentFlags().StringVar(&tvmTicket, "tvm-ticket", "", "tvm service ticket. May be empty in favor of tvm-client secret / tvm-knife")
	rootCmd.PersistentFlags().Uint32Var(&timeoutMs, "timeout", 20000, "grpc call timeoutCtx")
	rootCmd.PersistentPreRunE = configure
	rootCmd.SilenceUsage = true

	var createPromoAction = &cobra.Command{
		Use:   "createPromoAction",
		Short: "Create promo action with params for generation",
		Long:  "Create promo action with params for promo codes generation",
		RunE:  createPromoAction,
	}
	createPromoAction.Flags().String("action-name", "", "Promo action name")
	createPromoAction.Flags().Int64("budget", 0, "Budget of action in RUB")
	createPromoAction.Flags().Float64("min-total-cost", 0, "Minimum total cost")
	createPromoAction.Flags().String("valid-from", "", "Valid from (yyyy-MM-dd)")
	createPromoAction.Flags().String("valid-till", "", "Valid till (yyyy-MM-dd)")
	createPromoAction.Flags().Bool("additive", false, "Adds up with other actions")
	createPromoAction.Flags().String("user-type-restriction", promo.EUserTypeRestriction_UTR_NONE.String(), "User type restriction")
	createPromoAction.Flags().Float64("nominal", 0.0, "Promo code nominal")
	createPromoAction.Flags().String("nominal-type", ordercommons.EPromoCodeNominalType_NT_UNKNOWN.String(), "Type of nominal")
	createPromoAction.Flags().Int32("max-usage-count", 1, "Max usage count")
	createPromoAction.Flags().Int32("max-activations", 1, "Max activations")
	createPromoAction.Flags().Int64("fixed-days", 0, "Fixed days duration of code")
	createPromoAction.Flags().String("generation-type", promo.EPromoCodeGenerationType_PGT_SIMPLE.String(), "Code generation type")
	createPromoAction.Flags().String("prefix", "YA", "Prefix of promo code")
	createPromoAction.Flags().String("suffix", "AY", "Suffix of promo code")
	createPromoAction.Flags().String("hotel-restriction-partner", ordercommons.EServiceType_PT_TRAVELLINE_HOTEL.String(), "Service type restriction")
	createPromoAction.Flags().String("hotel-restriction-original-id", "", "Hotel restriction original id")
	createPromoAction.Flags().Int32("max-confirmed-hotel-orders", -1, "Number of orders already done by user should be less then the value "+
		"to let the promo code work. When set to 1, means that the promo can be used only for the 1st order. Negative value means no restriction.")
	createPromoAction.Flags().Float64("max-nominal-discount", 0, "The maximum promo code discount for percent nominal type")
	err := createPromoAction.MarkFlagRequired("action-name")
	if err != nil {
		log.Fatalf("Error setting required name flag. %v\n", err)
	}

	var createPromoCode = &cobra.Command{
		Use:   "createPromoCode",
		Short: "Create promo code",
		Long:  "Create promo code",
		RunE:  createPromoCode,
	}
	createPromoCode.Flags().String("action-id", "", "Action Id")
	createPromoCode.Flags().String("code", "", "Promo code")
	createPromoCode.Flags().Float64("nominal", 0.0, "Promo code nominal")
	createPromoCode.Flags().String("nominal-type", ordercommons.EPromoCodeNominalType_NT_UNKNOWN.String(), "Promo code nominal type (NT_VALUE|NT_PERCENT)")
	createPromoCode.Flags().Int32("max-usage-count", 1, "Max usage count")
	createPromoCode.Flags().Int32("max-activations", -1, "Max activations")
	createPromoCode.Flags().String("valid-from", "", "Valid from (yyyy-MM-dd)")
	createPromoCode.Flags().String("valid-till", "", "Valid till (yyyy-MM-dd)")
	createPromoCode.Flags().String("passport-id", "", "Passport Id to tie to")
	err = createPromoCode.MarkFlagRequired("action-id")
	if err != nil {
		log.Fatalf("Error setting required action-id flag. %v\n", err)
	}
	err = createPromoCode.MarkFlagRequired("code")
	if err != nil {
		log.Fatalf("Error setting required code flag. %v\n", err)
	}
	err = createPromoCode.MarkFlagRequired("nominal")
	if err != nil {
		log.Fatalf("Error setting required nominal flag. %v\n", err)
	}
	err = createPromoCode.MarkFlagRequired("nominal-type")
	if err != nil {
		log.Fatalf("Error setting required nominal-type flag. %v\n", err)
	}

	var resetPromoCodeActivation = &cobra.Command{
		Use:   "resetPromoCodeActivation",
		Short: "Reset promo code activation",
		Long:  "Reset promo code activation",
		RunE:  resetPromoCodeActivation,
	}
	resetPromoCodeActivation.Flags().String("passport-id", "", "Passport id")
	resetPromoCodeActivation.Flags().String("code", "", "Promo code")
	err = resetPromoCodeActivation.MarkFlagRequired("passport-id")
	if err != nil {
		log.Fatalf("Error setting required passport-id flag. %v\n", err)
	}
	err = resetPromoCodeActivation.MarkFlagRequired("code")
	if err != nil {
		log.Fatalf("Error setting required code flag. %v\n", err)
	}

	var createPromoCodesBatch = &cobra.Command{
		Use:   "createPromoCodesBatch",
		Short: "Create batch of promo codes",
		Long:  "Create batch of promo codes",
		RunE:  createPromoCodesBatch,
	}
	createPromoCodesBatch.Flags().String("action-id", "", "Action Id. Either this or name must be set.")
	createPromoCodesBatch.Flags().String("action-name", "", "Action Name. Either this or id must be set.")
	createPromoCodesBatch.Flags().Int32("batch-size", 1, "Batch Size")
	createPromoCodesBatch.Flags().String("for-date", "", "Date for batch generation (yyyy-MM-dd)")
	createPromoCodesBatch.Flags().String("out-file", "", "File to write promo codes to")
	createPromoCodesBatch.Flags().Bool("show-codes-only", false, "Output only codes. Do not show ids and other fields")

	var getPromoActionDetails = &cobra.Command{
		Use:   "getPromoActionDetails",
		Short: "Get promo action details",
		Long:  "Get promo action details",
		RunE:  getPromoActionDetails,
	}
	getPromoActionDetails.Flags().String("action-id", "", "Action Id")
	err = getPromoActionDetails.MarkFlagRequired("action-id")
	if err != nil {
		log.Fatalf("Error setting required action-id flag. %v\n", err)
	}

	var getPromoCodeActivationsCount = &cobra.Command{
		Use:   "getPromoCodeActivationsCount",
		Short: "Get number of promo code activations",
		Long:  "Get number of promo code activations",
		RunE:  getPromoCodeActivationsCount,
	}
	getPromoCodeActivationsCount.Flags().String("code-id", "", "Promocode id. Either this or code must be set.")
	getPromoCodeActivationsCount.Flags().String("code", "", "Promocode. Either this or id must be set.")

	var blacklistPromoCode = &cobra.Command{
		Use:   "blacklistPromoCode",
		Short: "Blacklist promo code",
		Long:  "Blacklist promo code",
		RunE:  blacklistPromoCode,
	}
	blacklistPromoCode.Flags().String("code-id", "", "Promocode id. Either this or code must be set.")
	blacklistPromoCode.Flags().String("code", "", "Promocode. Either this or id must be set.")

	rootCmd.AddCommand(
		createPromoAction,
		createPromoCode,
		resetPromoCodeActivation,
		createPromoCodesBatch,
		getPromoActionDetails,
		getPromoCodeActivationsCount,
		blacklistPromoCode,
	)
	err = rootCmd.Execute()
	if err != nil {
		log.Fatalf("Error executing command %+v. %v\n", rootCmd, err)
	}
}
