# coding: utf-8
from __future__ import unicode_literals, absolute_import, division, print_function

import argparse
import logging
import sys
import time as c_time
import uuid
from datetime import datetime, date, time, timedelta

import requests
from six.moves.urllib.parse import parse_qs, urlparse

from travel.orders.tools.library.tvm import get_tvm_service_ticket


def create_order_data_v2(args, offer_id, label):
    return {
        "passengers": [
            {
                "index": 0,
                "first_name": "Илья",
                "patronymic": "Викторович",
                "last_name": "Сугробин",
                "birth_date": "1990-01-01",
                "doc_id": "6500112233",
            },
            {
                "index": 1,
                "first_name": "Тест",
                "last_name": "РебенокА",
                "patronymic": "Тест",
                "birth_date": "2018-10-10",
                "doc_id": "XXАБ111111",
            }
        ],
        "customer_phone": args.phone,
        "customer_email": args.email,
        "user_info": {
            "ip": "1.2.3.4",
            "region_id": 213,
            "yandex_uid": args.yandex_uid,
            "geo_id": 213,
        },
        "order_history": [],
        "deduplication_key": str(uuid.uuid4()),
        "offer_id": offer_id,
        "label": label,
    }


def create_offer_data(args):
    departure = datetime.combine(date.today(), time(23, 00)) + timedelta(days=7, hours=-3)
    arrival = departure + timedelta(hours=8)
    return {
        "partner": "im",
        "arrival": arrival.strftime("%Y-%m-%dT%H:%M:00+00:00"),
        "departure": departure.strftime("%Y-%m-%dT%H:%M:00+00:00"),
        "passengers": [
            {
                "index": 0,
                "doc_type": "ПН",
                "sex": "M",
                "citizenship_geo_id": 225,
                "age_group": "adults",
                "tariff": "full"
            },
            {
                "index": 1,
                "sex": "M",
                "doc_type": "СР",
                "citizenship_geo_id": 225,
                "loyalty_cards": [],
                "tariff": "baby",
                "age_group": "babies"
            }
        ],
        "label_params": {
            "req_id": "reqId",
            "device": "desktop",
            "utm_source": "utmSource",
            "utm_medium": "utmMedium",
            "utm_campaign": "utmCampaign",
            "utm_term": "utmTerm",
            "utm_content": "utmContent",
            "from": "from",
            "gclid": "gugloid",
            "icookie": "test-icookie",
        },
        "train_number": "006А",
        "train_ticket_number": "006А",
        "train_title": "Москва — Санкт-Петербург",
        "station_from_id": 2006004,
        "station_to_id": 9602494,
        "car_number": "02",
        "car_type": "compartment",
        "service_class": "2Б",
        "electronic_registration": True,
        "bedding": True,
        "brand_title": "фирменный «Двухэтажный состав»",
        "route_policy": "internal",
        "icookie": "<test-train-client>",
        "amount": 1042.1,
    }


class OrchAPI(object):
    def __init__(self, args):
        self.lastResponse = None
        self.args = args
        self.service_ticket = args.service_ticket or get_tvm_service_ticket(
            tvm_client_id=args.tvm_client_id,
            tvm_service_id=args.tvm_service_id,
            tvm_client_secret=args.tvm_client_secret,
            skip_authentication=args.skip_authentication,
            logger=logging
        )

        self.headers = {
            "Content-Type": "application/json",
            "X-Ya-YandexUid": args.yandex_uid,
            "X-Ya-Session-Key": args.sessionid,
            "X-Ya-Service-Ticket": self.service_ticket
        }
        if args.user_ticket:
            self.headers["X-Ya-User-Ticket"] = args.user_ticket

    def create_order_v2(self, offer_id, label):
        response = requests.post(
            "http://{host}:{port}/api/trains_booking_flow/v2/create_order".format(host=self.args.host,
                                                                                  port=self.args.port),
            headers=self.headers,
            json=create_order_data_v2(self.args, offer_id, label),
            timeout=self.args.timeout
        )
        self.log_response(response)
        response.raise_for_status()

        return response.json()["id"]

    def add_insurance(self, order_uid):
        response = requests.post(
            "http://{host}:{port}/api/trains_booking_flow/v1/add_insurance".format(
                host=self.args.host,
                port=self.args.port,
            ),
            json={'id': order_uid},
            headers=self.headers,
            timeout=self.args.timeout
        )
        self.log_response(response)
        response.raise_for_status()

    def start_payment(self, order_uid):
        response = requests.post(
            "http://{host}:{port}/api/trains_booking_flow/v1/start_payment".format(host=self.args.host,
                                                                                   port=self.args.port),
            headers=self.headers,
            json={
                "order_id": order_uid,
                "return_url": "https://example.org/"
            },
            timeout=self.args.timeout
        )
        self.log_response(response)
        response.raise_for_status()

    def get_status(self, order_uid):
        response = requests.get(
            "http://{host}:{port}/api/trains_booking_flow/v1/get_order_status?id={uid}".format(
                host=self.args.host,
                port=self.args.port,
                uid=order_uid,
            ),
            headers=self.headers,
            timeout=self.args.timeout
        )
        self.log_response(response)
        response.raise_for_status()
        return response.json()

    def calculate_refund_amount(self, order_uid, blanks):
        response = requests.post(
            "http://{host}:{port}/api/trains_booking_flow/v1/calculate_refund_amount".format(
                host=self.args.host,
                port=self.args.port
            ),
            json={
                "order_id": order_uid,
                "blank_ids": blanks
            },
            headers=self.headers,
            timeout=self.args.timeout
        )
        self.log_response(response)
        response.raise_for_status()
        return response.json()

    def start_refund(self, order_uid, refund_token):
        response = requests.post(
            "http://{host}:{port}/api/trains_booking_flow/v1/start_refund".format(
                host=self.args.host,
                port=self.args.port
            ),
            json={
                "order_id": order_uid,
                "refund_token": refund_token
            },
            headers=self.headers,
            timeout=self.args.timeout
        )
        self.log_response(response)
        response.raise_for_status()

    def get_order_info(self, order_uid):
        response = requests.get(
            "http://{host}:{port}/api/trains_booking_flow/v1/get_order_info?id={uid}".format(
                host=self.args.host,
                port=self.args.port,
                uid=order_uid,
            ),
            headers=self.headers,
            timeout=self.args.timeout
        )
        self.log_response(response)
        response.raise_for_status()
        return response.json()

    def get_actualized_order_info(self, order_uid):
        response = requests.get(
            "http://{host}:{port}/api/trains_booking_flow/v1/get_actualized_order_info?id={uid}".format(
                host=self.args.host,
                port=self.args.port,
                uid=order_uid,
            ),
            headers=self.headers,
            timeout=self.args.timeout
        )
        self.log_response(response)
        response.raise_for_status()
        return response.json()

    def get_blank(self, order_uid):
        response = requests.get(
            "http://{host}:{port}/api/trains_booking_flow/v1/download_blank?id={uid}".format(
                host=self.args.host,
                port=self.args.port,
                uid=order_uid,
            ),
            headers=self.headers,
            timeout=self.args.timeout
        )
        self.log_response(response)
        response.raise_for_status()
        return response.content

    def get_refund_blank(self, order_uid, blank_id):
        response = requests.get(
            "http://{host}:{port}/api/trains_booking_flow/v1/download_blank?id={uid}&blankId={blank_id}".format(
                host=self.args.host,
                port=self.args.port,
                uid=order_uid,
                blank_id=blank_id
            ),
            headers=self.headers,
            timeout=self.args.timeout
        )
        self.log_response(response)
        response.raise_for_status()
        return response.content

    def log_response(self, response):
        if (self.lastResponse and self.lastResponse.request.url == response.request.url
                and self.lastResponse.text == response.text and self.lastResponse.status_code == response.status_code):
            logging.log(logging.INFO, "^^^ Same response ^^^")
        else:
            logging.log(logging.INFO, "{} => {}".format(response.request.url, response.status_code))
            if 'application/pdf' in response.headers.get('Content-Type', {}) or response.content.startswith(b'%PDF'):
                logging.log(logging.INFO, '<some pdfdata>')
            else:
                logging.log(logging.INFO, response.text)
        self.lastResponse = response


class OfferStorageClient(object):
    def __init__(self, args):
        self.args = args
        self.service_ticket = args.service_ticket or get_tvm_service_ticket(
            tvm_client_id=args.tvm_client_id,
            tvm_service_id=args.tvm_service_id,
            tvm_client_secret=args.tvm_client_secret,
            skip_authentication=args.skip_authentication,
            logger=logging
        )

        self.headers = {
            "Content-Type": "application/json",
            "X-Ya-YandexUid": args.yandex_uid,
            "X-Ya-Session-Key": args.sessionid,
            "X-Ya-Service-Ticket": self.service_ticket
        }
        if args.user_ticket:
            self.headers["X-Ya-User-Ticket"] = args.user_ticket

    def create_offer(self):
        response = requests.post(
            "http://{host}:{port}/store".format(host=self.args.offer_host, port=self.args.offer_port),
            headers=self.headers,
            json=create_offer_data(self.args),
            timeout=self.args.timeout
        )
        self.log_response(response)
        response.raise_for_status()
        return response.json()

    def log_response(self, response):
        logging.log(logging.INFO, "{} => {}".format(response.request.url, response.status_code))
        logging.log(logging.INFO, response.text)


def get_check_url(payment_url):
    parsed_url = urlparse(payment_url)
    purchase_token = parse_qs(parsed_url.query)["purchase_token"][0]
    return "https://{0}/pchecks/{1}/receipts/{1}/?mode=pdf".format(parsed_url.hostname, purchase_token)


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("-v", "--verbose", action="store_true", default=False)
    parser.add_argument("--host", default="localhost")
    parser.add_argument("--port", default=8090)
    parser.add_argument("--offer-host", default="train-offer-storage-test.stable.qloud-b.yandex.net")
    parser.add_argument("--offer-port", default=80)
    parser.add_argument("--timeout", default=10, type=int)

    parser.add_argument("--phone", default="+71234567890")
    parser.add_argument("--email", default="user@example.org")
    parser.add_argument("--yandex-uid", default="32431324")
    parser.add_argument("--sessionid", default="2412412341")

    parser.add_argument("--skip-authentication", action='store_true', default=False)
    parser.add_argument("--tvm-client-id", default=2010758, type=int)
    parser.add_argument("--tvm-service-id", default=2002740, type=int)
    parser.add_argument("--tvm-client-secret")

    parser.add_argument("--service-ticket")
    parser.add_argument("--user-ticket")

    parser.add_argument("--skip-insurance", action='store_true', default=False)
    parser.add_argument("--skip-refund", action='store_true', default=False)
    parser.add_argument("--is-cppk", action='store_true', default=False)

    args = parser.parse_args()
    logging.basicConfig(level=(logging.DEBUG if args.verbose else logging.INFO),
                        format="%(asctime)-15s | %(levelname)s | %(message)s",
                        stream=sys.stdout)

    offer_client = OfferStorageClient(args)
    api = OrchAPI(args)
    create_offer_rsp = offer_client.create_offer()
    offer_id = create_offer_rsp["offer_id"]
    label_hash = create_offer_rsp["label_hash"]
    order_uid = api.create_order_v2(offer_id, label_hash)

    logging.info("Создали заказ %s", order_uid)

    while True:
        status_data = api.get_status(order_uid)
        c_time.sleep(1)
        if status_data["status"] == "WAITING_PAYMENT":
            break
        elif status_data["status"] == "CANCELLED":
            logging.info("Ошибка бронирования заказа %s", order_uid)
            return

    api.get_order_info(order_uid)

    if not args.skip_insurance:
        api.add_insurance(order_uid)
        while True:
            status_data = api.get_status(order_uid)
            c_time.sleep(1)
            if status_data["status"] == "WAITING_PAYMENT" and status_data["insurance_status"] == "CHECKED_OUT":
                break
            elif status_data["insurance_status"] == "CHECKOUT_FAILED":
                logging.info("Ошибка при добавлении страховки в заказ %s", order_uid)
                return
        logging.info("Добавили страховку")

    api.start_payment(order_uid)
    logging.info("Запустили оплату")

    while True:
        status_data = api.get_status(order_uid)
        c_time.sleep(2)
        if status_data["status"] == "CONFIRMED":
            logging.info("Чек %s", get_check_url(status_data["payment_url"]))
            break
    logging.info("Билеты выкуплены")

    body = api.get_blank(order_uid)
    with open('blank.pdf', 'w') as f:
        f.write(body)
    logging.info("Билеты сохранены в blank.pdf")

    if not args.skip_refund:
        order_info = api.get_actualized_order_info(order_uid)
        blanks = list({p['tickets'][0]['blank_id'] for p in order_info['passengers']})
        refund_amount_rsp = api.calculate_refund_amount(order_uid, blanks)
        logging.info("Запросили сумму к возврату")

        api.start_refund(order_uid, refund_amount_rsp['refund_token'])
        logging.info("Запустили возврат")
        while True:
            c_time.sleep(2)
            status_data = api.get_status(order_uid)
            if status_data["status"] == "CONFIRMED":
                break
        logging.info("Билеты возвращены")

        order_info = api.get_order_info(order_uid)
        refunded_blank = [r['tickets'][0]['blank_id'] for r in order_info['refunds'] if r['tickets']][0]
        body = api.get_refund_blank(order_uid, refunded_blank)
        with open('refund_blank.pdf', 'w') as f:
            f.write(body)
        logging.info("КРС сохранена в refund_blank.pdf")

    api.get_actualized_order_info(order_uid)

    logging.info("Счастливого пути!")

    while True:
        api.get_order_info(order_uid)
        c_time.sleep(10)


if __name__ == "__main__":
    main()
