# coding: utf-8

from __future__ import unicode_literals

import logging
import os
import pickle
import subprocess
import sys
import threading

from datetime import datetime
from contextlib import contextmanager

from django.conf import settings
from django.core.management import call_command

from common.settings import WorkInstance, ServiceInstance
from travel.rasp.library.python.common23.date import environment
from common.utils.date import MSK_TZ
from common.utils.exceptions import SimpleUnicodeException
from travel.rasp.admin.lib.logs import add_stream_handler, remove_stream_handler, stdstream_file_capturing


log = logging.getLogger(__name__)


MAX_LOGS_TO_SHOW = 5


class TaskLog(object):
    class BadLogPath(SimpleUnicodeException):
        pass

    class BadLogDateFormat(SimpleUnicodeException):
        pass

    name = None
    created_at = None
    path = None
    instance_code = None
    DATE_FORMAT = '%Y-%m-%d_%H%M%S'

    def __init__(self, name, created_at=None, instance_code=None, filename_suffix=None):
        self.name = name
        self.instance_code = instance_code or settings.INSTANCE_ROLE.code
        self.created_at = created_at or environment.now_aware().astimezone(MSK_TZ)
        self.filename_suffix = filename_suffix

    @property
    def instance_role(self):
        for ir in [WorkInstance, ServiceInstance]:
            if ir.code == self.instance_code:
                return ir

    @property
    def filename(self):
        filename = self.created_at.strftime(self.DATE_FORMAT)
        if self.filename_suffix:
            filename += '---{}'.format(self.filename_suffix)
        return '{}.log'.format(filename)

    @classmethod
    def get_object_task_name(cls, obj, action_name):
        opts = obj.__class__._meta
        parts = [opts.app_label, opts.model_name, str(obj.id)]
        parts.append(action_name)
        return os.path.join(*parts)

    @classmethod
    def get_instance_log_path(cls, instance_role=None):
        instance_role = instance_role or settings.INSTANCE_ROLE
        return settings.LOG_PATHS[instance_role.code]

    @classmethod
    def get_all_tasks_dir_relative_path(cls):
        return os.path.join('special', 'tasks')

    @classmethod
    def get_all_tasks_dir_path(cls, instance_role=None):
        return os.path.join(cls.get_instance_log_path(instance_role), cls.get_all_tasks_dir_relative_path())

    @classmethod
    def get_task_dir_relative_path(cls, task_name):
        return os.path.join(cls.get_all_tasks_dir_relative_path(), task_name)

    @classmethod
    def get_task_dir_path(cls, task_name, instance_role=None):
        return os.path.join(cls.get_instance_log_path(instance_role), cls.get_task_dir_relative_path(task_name))

    def get_filename_endpoint(self):
        return os.path.join(self.get_task_dir_relative_path(self.name), self.filename)

    @property
    def path(self):
        return os.path.join(self.get_instance_log_path(self.instance_role), self.get_filename_endpoint())

    @classmethod
    def from_object_action(cls, obj, action_name):
        return cls(name=cls.get_object_task_name(obj, action_name))

    @classmethod
    def from_log_path(cls, task_log_path, instance_role=None):
        instance_role = instance_role or settings.INSTANCE_ROLE

        base_log_path = cls.get_all_tasks_dir_path(instance_role)
        if not task_log_path.startswith(base_log_path):
            raise cls.BadLogPath('Не смогли построить TaskLog, ошибка разбора времени пути {}'.format(
                task_log_path, instance_role.code
            ))

        name_with_date = task_log_path.replace(base_log_path, '', 1).strip(os.sep)
        name, filename = os.path.split(name_with_date)
        name.strip(os.sep)
        date_str_and_suffix = os.path.splitext(filename)[0].split('---')
        date_str = date_str_and_suffix[0]
        suffix = None
        if len(date_str_and_suffix) > 1:
            suffix = date_str_and_suffix[1]
        try:
            created_at = datetime.strptime(date_str, cls.DATE_FORMAT)
        except ValueError:
            raise cls.BadLogDateFormat('Не смогли построить TaskLog, ошибка разбора времени {}: путь {}'.format(
                date_str, task_log_path
            ))

        created_at = MSK_TZ.localize(created_at)

        return cls(name=name, created_at=created_at, instance_code=instance_role.code, filename_suffix=suffix)

    def ensure_dir_created(self):
        if not os.path.exists(os.path.dirname(self.path)):
            os.makedirs(os.path.dirname(self.path))

    @contextmanager
    def capture_log(self, log_name=None, format=None, capture_stdstreams=False):
        self.ensure_dir_created()

        with open(self.path, 'at') as stream:
            try:
                add_stream_handler(log_name, stream, format)

                if capture_stdstreams:
                    with stdstream_file_capturing(stream):
                        yield stream
                else:
                    yield stream

            finally:

                remove_stream_handler(log_name, stream)

    @property
    def url(self):
        domain_name = settings.DOMAIN_NAMES[self.instance_code]
        path = settings.LOG_URL_PATHS[self.instance_code]
        end_point = self.get_filename_endpoint()
        return 'https://{}{}{}'.format(domain_name, path, end_point)


def run_task(task_log, func, args=None, kwargs=None, auto_collect_zombie_process=True):
    args = args or ()
    kwargs = kwargs or {}

    task = {
        'task_log': task_log,
        'callable': func,
        'args': args,
        'kwargs': kwargs
    }

    with task_log.capture_log() as stream:
        log.info('Запускаем таск %s %s %s %s', task_log.name, task['callable'],
                 repr(task['args'])[:200], repr(task['kwargs'])[:200])

    task_data = pickle.dumps(task, protocol=pickle.HIGHEST_PROTOCOL)

    if settings.ADMIN_TASK_RUN_IN_SEPARATE_PROCESS:
        with task_log.capture_log() as stream:
            def spawn():
                process_env = os.environ.copy()
                process_env.update({'Y_PYTHON_ENTRY_POINT': 'travel.rasp.admin.app:manage'})
                call_args = [sys.executable, 'runtask']
                proc = subprocess.Popen(call_args, env=process_env,
                                        stdin=subprocess.PIPE, stdout=stream, stderr=stream,
                                        close_fds=True,  # Чтобы закрыть сокет гуникорна
                                        cwd=settings.PROJECT_PATH)

                proc.stdin.write(task_data)

                proc.stdin.close()
                return proc

            proc = spawn()

            if auto_collect_zombie_process:
                thread = threading.Thread(target=lambda: proc.wait())
                thread.daemon = True
                thread.start()

            return proc
    else:
        call_command('runtask', task_data=task_data)


def get_instance_task_logs(task_name, instance_role):
    base_task_log_path = TaskLog.get_task_dir_path(task_name, instance_role)
    if not os.path.exists(base_task_log_path):
        return []

    log_paths = [os.path.join(base_task_log_path, p)
                 for p in os.listdir(base_task_log_path)
                 if p.endswith('.log')]

    return list(sorted(map(lambda lp: TaskLog.from_log_path(lp, instance_role), log_paths),
                       key=lambda tl: tl.created_at, reverse=True))


def get_task_log_blocks(task_name, instance_role=None, max_logs_to_show=None):
    instance_role = settings.INSTANCE_ROLE if instance_role is None else instance_role
    max_logs_to_show = MAX_LOGS_TO_SHOW if max_logs_to_show is None else max_logs_to_show
    instance_roles = [WorkInstance, ServiceInstance]
    instance_roles.sort(key=lambda i: 0 if i == instance_role else 1)

    logs_by_instance = [
        (_ir, get_instance_task_logs(task_name, _ir)[:max_logs_to_show])
        for _ir in instance_roles
    ]

    def get_log_title(ir):
        return {
            ServiceInstance: 'Сервисные логи',
            WorkInstance: 'Рабочие логи',
        }[ir]

    task_logs_blocks = []
    for instance_role, logs in logs_by_instance:
        task_logs_blocks.append({
            'title': get_log_title(instance_role),
            'logs': logs
        })

    return task_logs_blocks
