# -*- coding: utf-8 -*-

import json

from django.db.models import Count
from django.http import Http404

from common.models.geo import Station
from common.utils.iterrecipes import unique_everseen
from common.utils.httpresponses import jsonp_response
from travel.rasp.admin.importinfo.models import OriginalThreadData
from travel.rasp.admin.importinfo.models.two_stage_import import (
    TwoStageImportPackage, TwoStageImportThread, TSIThreadSetting, TwoStageImportThreadStation, TSIThreadStationFlag,
    TwoStageImportStation
)
from travel.rasp.admin.admin.blue.views import blueadmin_permission_required_or_403, transform_title, REVERSE_FUZZY_MAPPING
from travel.rasp.admin.admin.blue.convertors import package_to_json, thread_to_json, station_to_json, thread_settings_to_json


@blueadmin_permission_required_or_403
@jsonp_response
def packages(request):
    return [
        package_to_json(p)
        for p in TwoStageImportPackage.objects.select_related('settlement', 'settlement__region',
                                                              'settlement__region__country')
    ]


@blueadmin_permission_required_or_403
@jsonp_response
def threads(request):
    package_id = request.GET.get('package_id')
    threads = (
        TwoStageImportThread.objects.filter(package_id=package_id)
        .annotate(mapped=Count('threadstations__station__station_mapping'),
                  stations_count=Count('threadstations'))
        .values('id', 'title', 'mapped', 'stations_count', 'path_key')
    )

    path_keys = [t['path_key'] for t in threads]

    original_thread_data = OriginalThreadData.objects.filter(
        package_id=package_id, thread__path_key__in=path_keys
    ).select_related('thread__title').values('id', 'thread__path_key', 'thread__title')

    original_thread_data_dict = {}
    for d in original_thread_data:
        item = original_thread_data_dict.setdefault(d['thread__path_key'], [])
        item.append({'id': d['id'], 'title': d['thread__title']})

    def thread_setting_stub(path_key):
        return TSIThreadSetting(package_id=package_id, path_key=path_key)

    settings_dict = {}
    for thread_setting in TSIThreadSetting.objects.filter(package=package_id, path_key__in=path_keys):
        settings_dict[thread_setting.path_key] = thread_setting

    for thread in threads:
        thread['original_threads_data'] = original_thread_data_dict.get(thread['path_key'])
        thread['setting'] = thread_settings_to_json(
            settings_dict.get(thread['path_key']) or
            thread_setting_stub(thread['path_key'])
        )

    return [thread_to_json(p) for p in threads]


@blueadmin_permission_required_or_403
@jsonp_response
def threads_save_setting(request):
    setting_data = json.loads(request.body)

    if setting_data['id']:
        setting = TSIThreadSetting.objects.get(pk=setting_data['id'])
    else:
        tsi_thread = TwoStageImportThread.objects.get(package_id=setting_data['package_id'],
                                                      path_key=setting_data['path_key'])
        setting = tsi_thread.get_setting()

    setting.allow_to_import = setting_data['allow_to_import']
    setting.apply_base_stations = setting_data['apply_base_stations']
    setting.timezone_override = setting_data['timezone_override']

    setting.save()

    return thread_settings_to_json(setting)


@blueadmin_permission_required_or_403
@jsonp_response
def stations_list(request):
    thread_id = int(request.GET.get('thread_id'))
    tsi_rtstations = TwoStageImportThreadStation.objects \
                                                .filter(thread_id=thread_id) \
                                                .select_related('station__station_mapping__station')
    stations = list(unique_everseen([ts.station for ts in tsi_rtstations]))

    partner_station_keys = set([s.key for s in stations])

    tsi_thread = TwoStageImportThread.objects.get(pk=thread_id)
    fuzzy_flags = TSIThreadStationFlag.objects.filter(
        package=tsi_thread.package,
        path_key=tsi_thread.path_key,
        station_key__in=partner_station_keys,
    ).all()

    fuzzy_flags = dict([(flag.station_key, flag) for flag in fuzzy_flags])

    for index, tsi_station in enumerate(stations, 1):
        tsi_station.index = index
        tsi_station.fuzzy_flag = fuzzy_flags.get(tsi_station.key)
        tsi_station.thread_id = thread_id

    return [station_to_json(p) for p in stations]


@blueadmin_permission_required_or_403
@jsonp_response
def station(request, tsi_station_id):
    if request.method != 'PUT':
        raise Http404

    data = json.loads(request.body)

    tsi_station = TwoStageImportStation.objects.get(id=tsi_station_id)
    tsi_thread = TwoStageImportThread.objects.get(pk=int(data.get('thread_id')))

    if not tsi_thread.threadstations.filter(station=tsi_station).exists():
        raise Exception(u'Станция должна быть в списке остановок нитки')

    station_update_params = ['title', 'majority_id', 'station_type_id', 't_type_id',
                             'settlement_id', 'district_id', 'region_id', 'country_id']

    # Создание новой станции в нашей базе
    if not tsi_station.station_mapping or not tsi_station.station_mapping.station:
        station = Station()

        station.latitude = float(data['lat'])
        station.longitude = float(data['lon'])

        for param in station_update_params:
            value = data['www_station'].get(param)
            if param.endswith('_id') and value:
                value = int(value)

            if value:
                setattr(station, param, value)

        station.title = station.title or tsi_station.title

        if station.title == tsi_station.title:
            station.title = tsi_station.real_title or tsi_station.title

        station.title = transform_title(station.title)
        station.t_type_id = station.t_type_id or tsi_station.package.t_type_id
        station.station_type_id = station.station_type_id or 11

        station.time_zone = station.settlement_id and station.settlement.time_zone

        if not station.time_zone:
            station.time_zone = station.region_id and station.region.time_zone

        if not station.time_zone and station.country_id:
            station.time_zone = station.country.get_capital_tz()

        station.save()

        tsi_station.update_or_create_mapping(station)
        tsi_station = TwoStageImportStation.objects.get(id=tsi_station_id)

        return station_to_json(tsi_station)

    # Обновление станции в нашей базе
    station = tsi_station.station_mapping.station

    station.latitude = float(data['lat'])
    station.longitude = float(data['lon'])

    for param in station_update_params:
        value = data['www_station'].get(param)
        if param.endswith('_id') and value:
            value = int(value)

        if value:
            setattr(station, param, value)

    station.save()

    # Fuzzy
    def interpretate(val):
        return REVERSE_FUZZY_MAPPING[val]

    fuzzy_flag = tsi_station.get_fuzzy_flag(tsi_station.package, tsi_thread.path_key)
    for name in ['is_fuzzy',
                 'is_searchable_from',
                 'is_searchable_to',
                 'in_station_schedule']:
        if name in data:
            setattr(fuzzy_flag, name, interpretate(data[name]))

    fuzzy_flag.save()

    tsi_station.fuzzy_flag = fuzzy_flag
    tsi_station.save()
    tsi_station = TwoStageImportStation.objects.get(id=tsi_station_id)

    return station_to_json(tsi_station)


@blueadmin_permission_required_or_403
@jsonp_response
def station_map(request, tsi_station_id):
    tsi_station = TwoStageImportStation.objects.get(id=tsi_station_id)

    www_station_id = request.POST.get('www_station_id')
    station = Station.objects.get(pk=www_station_id)

    tsi_station.update_or_create_mapping(station)
    tsi_station = TwoStageImportStation.objects.get(id=tsi_station_id)

    return station_to_json(tsi_station)


@blueadmin_permission_required_or_403
@jsonp_response
def station_unmap(request, tsi_station_id):
    tsi_station = TwoStageImportStation.objects.get(id=tsi_station_id)
    tsi_station.unmap()
    tsi_station = TwoStageImportStation.objects.get(id=tsi_station_id)

    return station_to_json(tsi_station)
