# -*- coding: utf-8 -*-

from __future__ import absolute_import

import json
import logging

from django.conf import settings
from django.core.urlresolvers import reverse
from django.views.decorators.csrf import ensure_csrf_cookie
from django.shortcuts import redirect
from django.utils.html import escape
from django.db.models import Max, Min, Count
from django.utils import translation
from django.utils.translation import ugettext

from travel.rasp.admin.importinfo.models import OriginalThreadData
from travel.rasp.admin.importinfo.models.two_stage_import import (
    TwoStageImportPackage, TwoStageImportThreadStation, TwoStageImportThread,
    TSIThreadStationFlag, TwoStageImportStation, TSIThreadSetting
)
from common.utils.geo import center_span_zoom, clip
from travel.rasp.admin.lib.geocoder import geocode_to_rasp_geo_objects, safe_get_geocode_coords
from travel.rasp.admin.lib.jinja import render_to_response
from common.utils.httpresponses import jsonp_response
from common.models.geo import StationMajority, StationType, Station, Settlement, District, Region, Country
from common.models.transport import TransportType

from travel.rasp.admin.admin.utils import permission_required_or_403
from travel.rasp.admin.admin.views import geoadmin


log = logging.getLogger(__name__)


def blueadmin_permission_required_or_403(func):
    return permission_required_or_403(func, 'importinfo.change_stationmapping')


@blueadmin_permission_required_or_403
@ensure_csrf_cookie
def admin(request, package_id=None):
    context = {
        'MAIN_INSTANCE': 'false' if settings.SERVICE_INSTANCE else 'true'
    }

    packages = TwoStageImportPackage.objects.all()
    package = None

    if package_id:
        try:
            package = TwoStageImportPackage.objects.get(pk=package_id)
        except:
            pass

    if not package and len(packages):
        return redirect('admin', package_id=packages[0].id)

    metathreads = {}
    used_stations = set()
    for thread in package.threads.\
            annotate(stmin=Min('threadstations__id'),
                     stmax=Max('threadstations__id')).\
            values('id', 'title', 'path_key', 'stmin', 'stmax',).all():

        if not thread['stmin'] or not thread['stmax']:
            continue

        k = thread['path_key']

        if k not in metathreads:
            # идентифицируем метанитку первой её ниткой
            metathreads[k] = {'id': thread['id'], 'threads': list()}

        metathreads[k]['threads'].append(thread)
        used_stations.add(thread['stmin'])
        used_stations.add(thread['stmax'])

    used_stations = TwoStageImportThreadStation.objects.\
        filter(pk__in=used_stations).values_list('id', 'station__title')

    used_stations = dict(used_stations)

    for m in metathreads.values():
        m['title'] = m['threads'][0]['title']
        m['title'] = m['title'] or \
            u'%s - %s' % (used_stations[m['threads'][0]['stmin']],
                          used_stations[m['threads'][0]['stmax']])
        m['threads'] = sorted([{'id': t['id'], 'title': t['title']}
                               for t in m['threads']], key=lambda t: t['title'])

    mapped_values = (
        TwoStageImportThread.objects
        .filter(id__in=[m['id'] for m in metathreads.values()])
        .annotate(mapped=Count('threadstations__station__station_mapping'),
                  stations_count=Count('threadstations'))
        .values('id', 'mapped', 'stations_count')
    )
    mapped_values = dict([(v['id'], v) for v in mapped_values])

    for m in metathreads.values():
        m['unmapped'] = \
            mapped_values[m['id']]['stations_count'] - \
            mapped_values[m['id']]['mapped']

    metathreads = sorted(metathreads.values(), key=lambda m: "# {}".format(m['title']) if m['unmapped'] else m['title'])

    city, region, country = package.get_city_region_coutry()

    lon, lat = safe_get_geocode_coords(city)

    context.update({
        'STATIC_URL': settings.STATIC_URL,
        'package': package,
        'packages': packages,
        'packages_dict': {p.id: unicode(p) for p in packages},
        'route_packages': [package],
        't_types': dict([(t.id, {u'id': t.id, u'code': t.code, u'title': t.L_title()})
                         for t in TransportType.objects.all()]),
        'metathreads': metathreads,
        'metathreads_json': json.dumps({m['id']: m for m in metathreads}, ensure_ascii=False),
        'city': {
            'id': city.id,
            'title': city.L_title(),
            'lon': lon,
            'lat': lat
        } if city else {'id': '', 'title': ''},
        'district': {'id': '', 'title': ''},
        'region': {'id': region.id, 'title': region.L_title()} if region else {'id': '', 'title': ''},
        'country': {'id': country.id, 'title': country.L_title()} if country else {'id': '', 'title': ''},
        'station_majorities': [{'id': m.id, 'title': m.title}
                               for m in StationMajority.objects.all()],
        'station_types': [{'id': st.id, 'name': st.L_name()} for st in StationType.objects.all()]
    })

    country_code = ''
    if package.settlement and package.settlement.country:
        country_code = package.settlement.country.code
    if not country_code:
        if package.region and package.region.country:
            country_code = package.region.country.code
    country_code = country_code or ''

    context['map_lang'] = settings.ADMIN_COUNTRY_YMAPLANGUAGE_MAP.get(country_code, 'ru-RU')

    return render_to_response('admin/blue/admin.jhtml', context, request=request)


@blueadmin_permission_required_or_403
@jsonp_response
def stations(request):
    center, span, zoom = center_span_zoom(request)

    stations = request.GET.getlist('stations')

    if center is None or span is None or zoom is None:
        return None

    if zoom < 10:
        stations = Station.objects.filter(id__in=stations)
    else:
        stations = clip(Station.objects.filter(t_type__id=TransportType.BUS_ID), center, span,
                        zoom_radius=1.1, lon_field='longitude', lat_field='latitude')

    links = {}

    return [(
        s.id,
        s.title,
        s.longitude,
        s.latitude,
        s.settlement_id,
        0,  # s.station_group_id,
        list(links.get(s.id, [])),
        0,  # s.traffic_coefficient_id,
        s.majority_id
    ) for s in stations]


def www_station_to_json(station):
    common_params = ['id', 'majority_id', 't_type_id', 'station_type_id', 'country_id',
                     'region_id', 'district_id', 'settlement_id']
    json_station = {
        'title': station.L_title(),
    }

    for param in common_params:
        json_station[param] = getattr(station, param)

    json_station.update(flags_to_json(station))

    return json_station


REVERSE_FUZZY_MAPPING = {'yes': True, 'no': False, 'inherit': None}
FUZZY_MAPPING = {True: 'yes', False: 'no', None: 'inherit'}


def flags_to_json(self):
    return {
        name: FUZZY_MAPPING[getattr(self, name)]
        for name in [
            'is_fuzzy', 'is_searchable_from', 'is_searchable_to',
            'in_station_schedule'
        ]
    }


def get_bounds_from_tsi_thread_stations(t):
    pre = 'station__station_mapping__station__'
    bounds = t.threadstations.aggregate(min_lon=Min(pre + 'longitude'),
                                        min_lat=Min(pre + 'latitude'),
                                        max_lon=Max(pre + 'longitude'),
                                        max_lat=Max(pre + 'latitude'))

    tsi_bounds = t.threadstations.aggregate(min_lon=Min('station__longitude'),
                                            min_lat=Min('station__latitude'),
                                            max_lon=Max('station__longitude'),
                                            max_lat=Max('station__latitude'))

    if None in bounds.values():
        bounds = tsi_bounds
    elif None not in tsi_bounds.values():
        bounds['min_lon'] = tsi_bounds['min_lon'] \
            if tsi_bounds['min_lon'] < bounds['min_lon'] else \
            bounds['min_lon']

        bounds['min_lat'] = tsi_bounds['min_lat'] \
            if tsi_bounds['min_lat'] < bounds['min_lat'] else \
            bounds['min_lat']

        bounds['max_lon'] = tsi_bounds['max_lon'] \
            if tsi_bounds['max_lon'] > bounds['max_lon'] else \
            bounds['max_lon']

        bounds['max_lat'] = tsi_bounds['max_lat'] \
            if tsi_bounds['max_lat'] > bounds['max_lat'] else \
            bounds['max_lat']

    return bounds


@blueadmin_permission_required_or_403
@jsonp_response
def thread(request, thread_id):
    t = TwoStageImportThread.objects.get(pk=thread_id)

    bounds = get_bounds_from_tsi_thread_stations(t)

    if None in bounds.values():
        city, region, country = t.package.get_city_region_coutry()

        lon, lat = safe_get_geocode_coords(city)
        if lon and lat:
            bounds = {'min_lon': float(lon), 'min_lat': float(lat),
                      'max_lon': float(lon), 'max_lat': float(lat)}

    # На крайний случай, откроем карту на Москве
    if None in bounds.values():
        bounds = {'min_lon': 37.6622, 'min_lat': 55.757601,
                  'max_lon': 37.6622, 'max_lat': 55.757601}

    w_lon = abs(bounds['max_lon'] - bounds['min_lon'])
    w_lat = abs(bounds['max_lat'] - bounds['min_lat'])

    w_lon = max(w_lon, 0.05)
    w_lat = max(w_lat, 0.05)

    # if w_lon < radius:
    c_lon = ((bounds['min_lon'] + bounds['max_lon']) % 360) / 2
    bounds['min_lon'] = c_lon - w_lon
    bounds['max_lon'] = c_lon + w_lon

    # if w_lat < radius:
    c_lat = ((bounds['min_lat'] + bounds['max_lat']) % 360) / 2
    bounds['min_lat'] = (c_lat - w_lat + 180) % 360 - 180
    bounds['max_lat'] = (c_lat + w_lat + 180) % 360 - 180

    bounds = ((bounds['min_lon'], bounds['min_lat']),
              (bounds['max_lon'], bounds['max_lat']))

    stations = t.threadstations.select_related('station__station_mapping__station')
    stations = [ts.station for ts in stations]

    partner_station_keys = set([s.key for s in stations])

    fuzzy_flags = TSIThreadStationFlag.objects.filter(
        package=t.package,
        path_key=t.path_key,
        station_key__in=partner_station_keys,
    ).all()

    fuzzy_flags = dict([(flag.station_key, flag) for flag in fuzzy_flags])

    for j, s in enumerate(stations, 1):
        s.index = j
        s.fuzzy_flag = fuzzy_flags.get(s.key)
        s.allow_fuzzy = s.index != 1

    original_threads_data = {
        ot.id: ot.thread.title
        for ot in OriginalThreadData.objects.filter(thread__path_key=t.path_key).select_related('thread__title')
    }

    def station_json(s):
        data = {
            'number': s.index,
            'id': s.id,
            'code': s.code,
            'title': s.title,
            'real_title': s.real_title,
            'additional_info': escape(s.additional_info or ''),
            'fuzzy_flag_id': s.fuzzy_flag and s.fuzzy_flag.id or '',
            'allow_fuzzy': s.allow_fuzzy,
            'lon': s.longitude,
            'lat': s.latitude
        }

        if s.station_mapping:
            data.update({
                'lon': s.station_mapping.station.longitude,
                'lat': s.station_mapping.station.latitude,
                'station': www_station_to_json(s.station_mapping.station)
            })

        data.update(s.fuzzy_flag and flags_to_json(s.fuzzy_flag) or {})

        return data

    return {
        'id': t.id,
        'stations': map(station_json, stations),
        'bounds': bounds,
        'path_key': t.path_key,
        'originalTherads': original_threads_data
    }


def transform_title(title):
    if title.isupper():
        title = title.capitalize()

    return title


@blueadmin_permission_required_or_403
@jsonp_response
def station_action(request, tsi_station_id):
    if tsi_station_id:
        tsi_station = TwoStageImportStation.objects.get(pk=tsi_station_id)
    else:
        tsi_station = None

    if request.method == 'POST':
        action = request.GET.get('action')

        station_update_params = ['title', 'majority_id', 'station_type_id', 't_type_id',
                                 'settlement_id', 'district_id', 'region_id', 'country_id']

        if action == 'map':
            station_id = request.POST.get('www_station_id')
            station = Station.objects.get(pk=station_id)

            tsi_station.update_or_create_mapping(station)

            return {'status': 'ok'}

        elif action == 'new_map':
            station = Station()
            station.latitude = float(request.POST.get('lat'))
            station.longitude = float(request.POST.get('lon'))

            for param in station_update_params:
                value = request.POST.get(param)
                if param.endswith('_id') and value:
                    value = int(value)

                if value:
                    setattr(station, param, value)

            station.title = station.title or tsi_station.title

            if station.title == tsi_station.title:
                station.title = tsi_station.real_title or tsi_station.title

            station.title = transform_title(station.title)
            station.t_type_id = station.t_type_id or tsi_station.package.t_type_id
            station.station_type_id = station.station_type_id or 11

            station.time_zone = station.settlement_id and station.settlement.time_zone

            if not station.time_zone:
                station.time_zone = station.region_id and station.region.time_zone

            if not station.time_zone and station.country_id:
                station.time_zone = station.country.get_capital_tz()

            station.save()

            tsi_station.update_or_create_mapping(station)

            result = {
                'status': 'ok',
                'lon': station.longitude,
                'lat': station.latitude,
                'station': www_station_to_json(station)
            }

            return result

        elif action == 'unmap':
            tsi_station.unmap()
            return {'status': 'ok'}

        elif action == 'save_fuzzy_flag':
            fuzzy_flag = tsi_station.get_fuzzy_flag(tsi_station.package, request.POST['path_key'])

            def interpretate(val):
                return REVERSE_FUZZY_MAPPING[val]

            for name in ['is_fuzzy',
                         'is_searchable_from',
                         'is_searchable_to',
                         'in_station_schedule']:
                if name in request.POST:
                    setattr(fuzzy_flag, name, interpretate(request.POST[name]))

            fuzzy_flag.save()

            return {'status': 'ok'}
        elif action == 'update_station':
            station = Station.objects.get(id=int(request.POST['id']))

            for param in station_update_params:
                value = request.POST.get(param)
                if param.endswith('_id') and value:
                    value = int(value)

                if value:
                    setattr(station, param, value)

            station.save()

            result = {
                'status': 'ok',
                'station': www_station_to_json(station)
            }

            return result

    return {'status': 'ok'}


@blueadmin_permission_required_or_403
@jsonp_response
def station_move(request, station_id):
    try:
        station = Station.objects.get(pk=station_id)
    except Station.DoesNotExist:
        return {'status': "can't find station"}

    if request.method == 'POST':
        station.longitude = request.POST.get('lon', None)
        station.latitude = request.POST.get('lat', None)
        if station.longitude and station.latitude:
            try:
                station.save()
            except:
                return {'status': 'can\'t save'}
            return {'status': 'ok'}
    return {'status': 'wrong query'}


@blueadmin_permission_required_or_403
@jsonp_response
def info(request, what, id):
    result = {'status': 'ok'}
    if what == 'settlement':
        try:
            settlement = Settlement.objects.\
                select_related('region', 'region__country').get(pk=id)
        except Settlement.DoesNotExist:
            result['status'] = 'not found'
        else:
            result['city'] = {'id': settlement.id, 'title': settlement.L_title()}
            if settlement.region:
                result['region'] = {'id': settlement.region.id, 'title': settlement.region.L_title()}
                result['country'] = {'id': settlement.region.country.id, 'title': settlement.region.country.L_title()}

    elif what == 'district':
        try:
            district = District.objects.select_related('country').get(pk=id)
        except District.DoesNotExist:
            result['status'] = 'not found'
        else:
            result['district'] = {'id': district.id, 'title': district.title}
            if district.region:
                result['region'] = {'id': district.region.id, 'title': district.region.L_title()}
                result['country'] = {'id': district.region.country.id, 'title': district.region.country.L_title()}

    elif what == 'region':
        try:
            region = Region.objects.select_related('country').get(pk=id)
        except Region.DoesNotExist:
            result['status'] = 'not found'
        else:
            result['region'] = {'id': region.id, 'title': region.L_title()}
            result['country'] = {'id': region.country.id, 'title': region.country.L_title()}

    elif what == 'country':
        try:
            country = Country.objects.get(pk=id)
        except Country.DoesNotExist:
            result['status'] = 'not found'
        else:
            result['country'] = {'id': country.id, 'title': country.L_title()}
    else:
        result['status'] = 'not found'

    return result


@blueadmin_permission_required_or_403
@jsonp_response
def www_station_set_majority(request):
    try:
        station_id = request.POST.get('station_id')
        majority_id = request.POST.get('majority_id')
        station = Station.objects.get(pk=station_id)
        station.majority = StationMajority.objects.get(id=majority_id)
        station.save()
    except Exception as e:
        log.exception(e.message)
        return {'status': 'bad'}

    return {'status': 'ok'}


@blueadmin_permission_required_or_403
@ensure_csrf_cookie
@jsonp_response
def geocode(request):
    result = geocode_to_rasp_geo_objects(request.GET['geocode'])

    result_count = len(result.rasp_geoocode_objects)

    return {
        'status': 'ok' if result_count else 'not found',
        'geoobject': rasp_geocode_object_to_json(result.rasp_geoocode_objects[0]) if result_count else None,
        'geocoder_response': result.geocoder_response,
        'rasp_geoobjects': [rasp_geocode_object_to_json(go) for go in result.rasp_geoocode_objects]
    }


def rasp_geocode_object_to_json(rasp_geocode_object):
    json_data = {
        'lat': rasp_geocode_object.latitude,
        'lon': rasp_geocode_object.longitude,
        'maps_name': rasp_geocode_object.maps_name,
        'maps_kind': rasp_geocode_object.maps_kind
    }

    if isinstance(rasp_geocode_object.geo_object, Settlement):
        settlement = rasp_geocode_object.geo_object

        json_data['type'] = 'settlement'
        json_data['settlement'] = {
            'id': settlement.id,
            'title': settlement.L_title()
        }
        if settlement.region_id:
            json_data['region'] = {
                'id': settlement.region.id,
                'title': settlement.region.L_title()
            }
        if settlement.country_id:
            json_data['country'] = {
                'id': settlement.country.id,
                'title': settlement.country.L_title()
            }

    elif isinstance(rasp_geocode_object.geo_object, Region):
        region = rasp_geocode_object.geo_object

        json_data['type'] = 'region'
        json_data['region'] = {
            'id': region.id,
            'title': region.L_title()
        }
        if region.country_id:
            json_data['country'] = {
                'id': region.country.id,
                'title': region.country.L_title()
            }

    return json_data


@blueadmin_permission_required_or_403
@ensure_csrf_cookie
def new(request):
    lang = translation.get_language()

    timezone_override_choices = [
        [unicode(v), unicode(t)]
        for v, t in TSIThreadSetting.TIMEZONE_OVERRIDE_CHOICES
    ]

    options = {
        'title': ugettext(u'Синий автобусный геоадмин'),
        'adminType': 'blue',
        'staticUrl': settings.STATIC_URL + 'rasp/geoadmins/',
        'maintenanceStatusUrl': reverse('maintenance_status'),
        'mainInstance': False if settings.SERVICE_INSTANCE else True,
        'lang': lang,
        't_types': dict([(t.id, {u'id': t.id, u'code': t.code, u'title': t.L_title()})
                         for t in TransportType.objects.all()]),
        'station_majorities': [{'id': m.id, 'title': m.title}
                               for m in StationMajority.objects.all()],
        'station_types': [{'id': st.id, 'name': st.L_name()} for st in StationType.objects.all()],
        'timezone_override_choices': timezone_override_choices
    }

    return geoadmin.render(options, lang)
