# -*- coding: utf-8 -*-

import json
from itertools import groupby

from django.db import connection
from django.db.models import Q
from django.conf import settings
from django.utils.translation import get_language
from django.views.decorators.csrf import ensure_csrf_cookie

from travel.rasp.admin.admin.utils import permission_required_or_403 as permission_wrapper
from travel.rasp.admin.admin.views.geomap import station_json
from common.models.geo import Station
from common.models_utils import fetch_related
from common.utils.httpresponses import jsonp_response
from travel.rasp.admin.importinfo.models.two_stage_import import TwoStageImportPackage, TwoStageImportThread
from travel.rasp.admin.lib.geocoder import safe_get_geocode_coords
from travel.rasp.admin.lib.jinja import render_to_response
from travel.rasp.admin.www.models.geo import RoutePath


def permission_required_or_403(func):
    return permission_wrapper(func, 'importinfo.change_stationmapping')


@permission_required_or_403
@ensure_csrf_cookie
def admin(request):
    mode = request.GET.get('mode', 'threads')

    context = {
        'STATIC_URL': settings.STATIC_URL,
        'MAIN_INSTANCE': 'false' if settings.SERVICE_INSTANCE else 'true',
        'map_lang': get_language(),
        'mode': mode
    }

    return render_to_response('index.jhtml', context, request=request)


@permission_required_or_403
@jsonp_response
def threads(request, package_id):
    package = _get_package(package_id)

    meta_threads = {}

    thread_stations_raw = (
        Station.objects
        .filter(stationmapping__twostageimportstation__threadstations__thread__package=package)
        .order_by('stationmapping__twostageimportstation__threadstations__thread__id',
                  'stationmapping__twostageimportstation__threadstations__id')
        .values('stationmapping__twostageimportstation__threadstations__thread__id',
                'stationmapping__twostageimportstation__threadstations__thread__title', 'id', 'title')
    )

    thread_stations = [(r['stationmapping__twostageimportstation__threadstations__thread__id'], r['id'])
                       for r in thread_stations_raw]

    mapped_threads = RoutePath.is_route_paths_confirmed(thread_stations)

    thread_id_groups = groupby(thread_stations_raw,
                               lambda r: r['stationmapping__twostageimportstation__threadstations__thread__id'])
    for thread_id, values in thread_id_groups:
        values = list(values)

        confirmed = mapped_threads.get(thread_id, False)

        value = values[0]

        last = values[-1]

        title = value['stationmapping__twostageimportstation__threadstations__thread__title'] or \
            u'%s - %s' % (value['title'], last['title'])

        meta_threads[thread_id] = {
            'id': thread_id,
            'title': title if confirmed else '# %s' % title,
            'unmapped': confirmed
        }

    meta_threads_list = sorted(meta_threads.values(), key=lambda m: m['title'])

    return {
        'meta_threads': meta_threads_list
    }


@permission_required_or_403
@jsonp_response
def moved_stations(request):
    paths = RoutePath.objects.filter(Q(status_direct=RoutePath.STATUS_CHANGED) |
                                     Q(status_back=RoutePath.STATUS_CHANGED))

    stations = set(p.station_from for p in paths)

    stations = stations.union(p.station_to for p in paths)

    stations = sorted(stations, key=lambda s: s.L_title())

    return {
        'stations': map(station_json, stations)
    }


@permission_required_or_403
@jsonp_response
def thread_segments(request, thread_id):
    t = TwoStageImportThread.objects.get(pk=thread_id)

    return {
        'id': t.id,
        'segments': map(_route_path_json, t.get_route_paths()),
    }


@permission_required_or_403
@jsonp_response
def station_segments(request, station_id):
    """
    Возвращает геокодированные участки проходящие через станцию (начинающиеся или заканчивающиеся)
    """
    station = Station.objects.get(id=station_id)

    paths = RoutePath.objects.filter(Q(station_from=station) | Q(station_to=station)).\
        filter(station_from__id__isnull=False). \
        filter(station_to__id__isnull=False)

    fetch_related(paths, 'station_from', 'station_to', model=RoutePath)

    wrapped = [RoutePath._wrap(w, station == w.station_from) for w in paths]

    return {
        'segments': map(_route_path_json, wrapped),
    }


@permission_required_or_403
@jsonp_response
def station_all_segments(request, station_id):
    """
    Возвращает геокодированные и не геокодированные участки проходящие через станцию (начинающиеся или заканчивающиеся)
    """

    station_id = int(station_id)

    # SQL запросом получается на несколько порядков быстрее
    sql = '''
SELECT DISTINCT
  bstation_m_next.station_id,
  bstation_m_prev.station_id
FROM
  (
    SELECT
      MIN(ts_next.id) next_id,
      MAX(ts_prev.id) prev_id
    FROM
      importinfo_twostagebusimportthreadstation ts1
      JOIN importinfo_twostagebusimportstation tbstations
          ON tbstations.id = ts1.station_id
        JOIN importinfo_busstationtitlemapping bstation_m
            ON bstation_m.id = tbstations.station_mapping_id

      LEFT JOIN importinfo_twostagebusimportthreadstation ts_next
          ON
                ts1.thread_id = ts_next.thread_id
            AND ts1.id < ts_next.id

      LEFT JOIN importinfo_twostagebusimportthreadstation ts_prev
          ON
                ts1.thread_id = ts_prev.thread_id
            AND ts1.id > ts_prev.id

      WHERE
        bstation_m.station_id = %s

      GROUP BY
        ts1.id
  ) ts

  LEFT JOIN (
    importinfo_twostagebusimportthreadstation t_next
      JOIN importinfo_twostagebusimportstation tbstations_next
          ON tbstations_next.id = t_next.station_id
        JOIN importinfo_busstationtitlemapping bstation_m_next
            ON bstation_m_next.id = tbstations_next.station_mapping_id
    )
        ON t_next.id = ts.next_id

  LEFT JOIN (
    importinfo_twostagebusimportthreadstation t_prev
      JOIN importinfo_twostagebusimportstation tbstations_prev
          ON tbstations_prev.id = t_prev.station_id
        JOIN importinfo_busstationtitlemapping bstation_m_prev
            ON bstation_m_prev.id = tbstations_prev.station_mapping_id
    )
        ON t_prev.id = ts.prev_id
'''

    cursor = connection.cursor()

    cursor.execute(sql, [station_id])

    segments_ids = set()

    stations_ids = set()

    stations_ids.add(station_id)

    for r in cursor.fetchall():
        if r[0]:
            segments_ids.add((station_id, r[0]))
            stations_ids.add(r[0])

        if r[1]:
            segments_ids.add((r[1], station_id))
            stations_ids.add(r[1])

    if not segments_ids:
        return {
            'segments': []
        }

    stations = dict((s.id, s) for s in Station.objects.filter(id__in=stations_ids))

    segments = [(stations[station_from], stations[station_to]) for station_from, station_to in segments_ids]

    paths = sorted(RoutePath.bulk_load(segments), key=lambda p: (p.station_from.L_title(), p.station_to.L_title()))

    return {
        'segments': map(_route_path_json, paths),
    }


@permission_required_or_403
@jsonp_response
def save_route_path(request):
    segments = json.load(request)

    for segment in segments:
        station_from = Station.objects.get(id=segment['station_from'])

        station_to = Station.objects.get(id=segment['station_to'])

        for_two_directions = segment['for_two_directions']

        RoutePath.save_route_path(station_from, station_to, json.dumps(segment['path']), for_two_directions)

    return {'status': 'ok'}


@permission_required_or_403
@jsonp_response
def remove_route_path(request):
    segments = json.load(request)

    for segment in segments:
        station_from = Station.objects.get(id=segment['station_from'])

        station_to = Station.objects.get(id=segment['station_to'])

        RoutePath.remove_route_path(station_from, station_to)

    return {'status': 'ok'}


def _get_city_json(city):
    if not city:
        return {'id': '', 'title': ''}

    lon, lat = safe_get_geocode_coords(city)

    return {
        'id': city.id,
        'title': city.L_title(),
        'lon': lon,
        'lat': lat
    }


def _get_package(package_id):
    if package_id:
        try:
            return TwoStageImportPackage.objects.get(pk=package_id)

        except TwoStageImportPackage.DoesNotExist:
            pass

    return None


def _route_path_json(route_path):
    return {
        'id': "segment_%s_%s" % (route_path.station_from.id, route_path.station_to.id),
        'title': u"%s&nbsp;&mdash; %s" % (route_path.station_from.L_title(), route_path.station_to.L_title()),
        'station_from': station_json(route_path.station_from),
        'station_to': station_json(route_path.station_to),
        'for_two_directions': route_path.for_two_directions,
        'status': route_path.status,
        'path': route_path.data
    }
