# -*- coding: utf-8 -*-

import os.path
import logging
from collections import OrderedDict
from shutil import rmtree
from subprocess import call

from django.db.models import Q
from django.db import transaction, connection
from lxml import etree
from lxml.etree import ElementTree, Element, SubElement, tostring

from common.models.geo import (ExternalDirection, Direction, DirectionMarker, ExternalDirectionMarker,
                               Station, SuburbanZone, Settlement, Region, StationType, StationMajority)
from common.models.schedule import RThread, RTStation
from common.models.transport import TransportType
from travel.rasp.admin.lib.exceptions import SimpleUnicodeException
from travel.rasp.admin.lib.tmpfiles import temp_manager
from travel.rasp.admin.scripts.support_methods import esr_leading_zeros, spk_leading_zeros


log = logging.getLogger(__name__)


@transaction.atomic
def import_suburbanzones(zones_file, test=False):
    if test:
        log.info(u"Импорт в тестовом режиме, после загрузки откатываем изменения")

    try:
        zones = do_import_suburbanzones(zones_file)
    except Exception:
        raise

    if test:
        log.info(u"Импорт в тестовом режиме, откатываем изменения")
        transaction.set_rollback(True)

    return zones


@transaction.atomic
def do_import_suburbanzones(zones_file):
    zones = []
    tmp_directory = temp_manager.get_tmp_dir('zones')

    try:

        zipfilepath = os.path.join(tmp_directory, 'zones.zip')
        zipfile = open(zipfilepath, 'wb')
        zipfile.write(zones_file.read())
        zipfile.close()

        retcode = call(["unzip", "-oqd", tmp_directory, zipfilepath])
        if retcode != 0:
            raise SimpleUnicodeException(u"Ошибка распаковки архива")

        stations_filepath = os.path.join(tmp_directory, 'stations.xml')
        settlements_filepath = os.path.join(tmp_directory, 'settlements.xml')
        zones_filepath = os.path.join(tmp_directory, 'zones.xml')

        log.info(u" === ")
        if os.path.exists(stations_filepath):
            log.info(u"Импортируем станции")
            update_station_from_fetisov_xml(stations_filepath)

        log.info(u" === ")
        if os.path.exists(zones_filepath):
            log.info(u"Импортируем пригородные зоны")
            zones += import_zones(zones_filepath)

        log.info(u" === ")
        if os.path.exists(settlements_filepath):
            log.info(u"Импортируем города")
            import_cities(settlements_filepath)

        internaldirection_path = os.path.join(tmp_directory, 'directions')
        externaldirection_path = os.path.join(tmp_directory, 'external_directions')

        log.info(u" === ")
        if os.path.exists(internaldirection_path):
            log.info(u"Импорт внутренних направлений")
            import_directions_from_path(Direction, internaldirection_path)

        log.info(u" === ")
        if os.path.exists(externaldirection_path):
            log.info(u"Импорт внешний направлений")
            import_directions_from_path(ExternalDirection, externaldirection_path)

        log.info(u" === ")
        schemas_dir = os.path.join(tmp_directory, 'schemas')
        if os.path.exists(schemas_dir):
            log.info(u"Импорт схем внешних направлений")

            for file_name in os.listdir(schemas_dir):
                if file_name.endswith('.csv'):
                    file_path = os.path.join(schemas_dir, file_name)

                    code = file_name.replace('.csv', '')

                    log.info(code)

                    direction = ExternalDirection.objects.get(code=code)
                    direction.schema_file = open(file_path).read().decode('cp1251')
                    direction.save()

        # Обновим привязку городов и станций к пригородным зонам
        cursor = connection.cursor()

        cursor.execute("""
        UPDATE `www_station` s JOIN `www_directionmarker` dm ON dm.station_id = s.id
            JOIN `www_direction` d ON dm.direction_id = d.id
            JOIN `www_suburbanzone` sz ON d.suburban_zone_id = sz.id
        SET s.suburban_zone_id = sz.id
        WHERE s.suburban_zone_id IS NULL
        """)

    finally:
        rmtree(tmp_directory, ignore_errors=True)

    return zones


def import_directions_from_path(direction_model, direction_path):
    files = os.listdir(direction_path)
    for filename in files:
        if filename.endswith('.xml'):
            import_direction(direction_model, os.path.join(direction_path, filename))


def import_zones(path):
    tree = etree.parse(path)

    zones = []

    for zone_el in tree.findall('.//zone'):
        data = {}
        attrs = ('name', 'pzone', 'settlement_id')

        for attr in attrs:
            data[attr] = zone_el.get(attr, u"").strip()

        log.info(u"Импортируем зону %s", data['name'])
        try:
            zone = SuburbanZone.objects.get(code=data['pzone'])
        except SuburbanZone.DoesNotExist:
            zone = SuburbanZone(code=data['pzone'])
        zone.title = data['name']

        settlement = Settlement.objects.get(id=data['settlement_id'])

        zone.settlement = settlement

        if not zone.title_from:
            zone.title_from = u'от %s' % settlement.title_ru_genitive

        if not zone.title_to:
            zone.title_to = u'на %s' % settlement.title_ru_accusative

        zone.save()

        zones.append(zone)

    return zones


def import_cities(path):
    tree = etree.parse(path)

    for settlement_el in tree.findall('.//'):
        data = {}
        attrs = ('settlement_title', 'pzone', 'settlement_id')

        for attr in attrs:
            data[attr] = settlement_el.get(attr, u"").strip()

        if not data['pzone']:
            continue

        zone = SuburbanZone.objects.get(code=data['pzone'])

        settlement = Settlement.objects.get(id=data['settlement_id'])
        settlement_title = data['settlement_title'] and data['settlement_title'].strip()

        log.info(u"Привязываем город %s к зоне %s", settlement_title or settlement.title, data['pzone'])

        if settlement_title and settlement.title != settlement_title:
            log.warning(u"Название города в данных не совпадает с данными базы ('%s' != '%s')" %
                        (settlement_title, settlement.title))

        settlement.suburban_zone = zone

        settlement.save()


@transaction.atomic
def reimport_direction(direction, direction_file):
    tree = ElementTree(file=direction_file)

    needed_type = 'internal' if isinstance(direction, Direction) else 'external'

    if not tree.getroot().get('type') == needed_type:
        raise SimpleUnicodeException(u"Типы направлений в файле, и текущий, не совпадают")

    reload_markers(direction, tree)


@transaction.atomic
def import_direction(direction_model, direction_file):
    direction = direction_model()

    tree = ElementTree(file=direction_file)

    needed_type = 'internal' if isinstance(direction, Direction) else 'external'

    if not tree.getroot().get('type') == needed_type:
        raise SimpleUnicodeException(u"Типы направлений в файле, и текущий, не совпадают")

    direction_elem = tree.getroot()
    params = {}
    for child in direction_elem:
        if child.text:
            params[child.tag] = child.text

    if params.get('zone'):
        direction.suburban_zone = SuburbanZone.objects.get(code=params['zone'])

    for attr, value in params.items():
        setattr(direction, attr, value)

    for child in direction_elem:
        if child.tag == 'base_station':
            if child.get('esr'):
                try:
                    direction.base_station = Station.get_by_code('esr', child.get('esr'))
                except Station.DoesNotExist:
                    log.error(u"Не нашли базовую станцию по коду %s", child.get('esr'))
                    raise
            else:
                try:
                    direction.base_station = Station.objects.get(id=child.get('id'))
                except Station.DoesNotExist:
                    log.error(u"Не нашли базовую станцию по id %s", child.get('id'))
                    raise

    log.info(u"Импортируем %s направление %s", needed_type, direction.title)

    direction.save()

    reload_markers(direction, tree)

    return direction


def reload_markers(direction, tree):
    if isinstance(direction, Direction):
        marker_model = DirectionMarker
        remove_rtstaion_references(direction)
        direction.directionmarker_set.all().delete()
    else:
        marker_model = ExternalDirectionMarker
        direction.externaldirectionmarker_set.all().delete()

    marker_elements = tree.findall("./markers/marker")
    for order, marker_element in enumerate(marker_elements):

        marker = marker_model()
        marker.order = order
        if isinstance(direction, Direction):
            marker.direction = direction
        else:
            marker.external_direction = direction

        marker.station = get_station(marker_element)
        for attr, value in marker_element.attrib.items():
            if attr not in ('station_id', 'esr'):
                setattr(marker, attr, value)

        log.info(u"Добавляем маркер %s", marker.station.title)

        marker.save()


def get_station(marker_element):
    try:
        if marker_element.get('esr'):
            return Station.get_by_code('esr', marker_element.get('esr'))
        else:
            return Station.objects.get(id=marker_element.get('station_id'))
    except Station.DoesNotExist:
        raise SimpleUnicodeException(u"Не нашли станцию для маркера %s" %
                                     tostring(marker_element,
                                              encoding='utf-8', method="html", pretty_print=True).decode('utf-8'))


@transaction.atomic
def remove_rtstaion_references(direction):
    (RThread.objects
            .filter(Q(rtstation__arrival_direction=direction) | Q(rtstation__departure_direction=direction))
            .update(changed=True))
    (RTStation.objects
              .filter(Q(arrival_direction=direction) | Q(departure_direction=direction))
              .update(arrival_direction=None, departure_direction=None))


def get_direction_xml_tree(direction, use_esr=False):
    root = Element(
        'direction',
        {'type': 'external' if isinstance(direction, ExternalDirection) else 'internal'}
    )

    SubElement(root, 'zone').text = direction.suburban_zone and direction.suburban_zone.code

    for key, value in vars(direction).items():
        if value and key not in ('id', 'schema_file', 'suburban_zone_id',
                                 'base_station', 'base_station_id'):
            text = None
            if isinstance(value, (unicode, int, float, long)):
                text = unicode(value)
            elif isinstance(value, bool):
                text = unicode(value)
            if text is not None:
                SubElement(root, key).text = text

    # base station
    if hasattr(direction, 'base_station') and direction.base_station:
        attrib = {}
        if use_esr and direction.base_station.get_code('esr'):
            attrib['esr'] = direction.base_station.get_code('esr')
        else:
            attrib['id'] = unicode(direction.base_station.id)

        SubElement(root, 'base_station', attrib)

    marker_elements = SubElement(root, 'markers')
    markers_attr = 'externaldirectionmarker_set' if isinstance(direction, ExternalDirection) else \
        'directionmarker_set'
    for marker in getattr(direction, markers_attr).all():
        attrib = OrderedDict()
        if use_esr:
            esr = marker.station.get_code('esr')
            if esr:
                attrib['esr'] = unicode(esr)
            else:
                attrib['station_id'] = unicode(marker.station.id)

        else:
            attrib['station_id'] = unicode(marker.station.id)

        attrib['station_title'] = marker.station.title

        for key, value in vars(marker).items():
            if value and key not in ('id', 'station_id', 'order', 'direction_id',
                                     'external_direction_id'):
                if isinstance(value, (unicode, int, float, long)):
                    attrib[key] = unicode(value)
                elif isinstance(value, bool):
                    attrib[key] = unicode(value)

        SubElement(marker_elements, 'marker', attrib)

    return ElementTree(root)


def update_station_from_fetisov_xml(path):
    u""" Импортирует станции из файла xml в формате
    <stations>
        <station>
    title;express;esr;spk;station_type_id;settlement_id;region_id
    Если поле не заполнено, оставлять пустую строку.
    пример, файла.
    = = =
    title;express;esr;spk;station_type_id;settlement_id;region_id
    Трифоновка;99554411;223344;2211;1;;28
    Трифоновка большая;5254411;223344;;;;28
    = = =
    """
    # Формат оп таску RASP-2005: Механизм импорта станций от Саши Фетисова
    systems = ('express', 'esr', 'spk')

    tree = etree.parse(path)

    settlement_ids = set([sett.id for sett in Settlement.objects.all()])
    region_ids = set([r.id for r in Region.objects.all()])
    stype_ids = set([st.id for st in StationType.objects.all()])
    station_majority_ids = set([st.id for st in StationMajority.objects.all()])

    for station_el in tree.findall('.//station'):
        codes = {
            'esr': station_el.get('esr', u"").strip(),
            'spk': station_el.get('spk', u"").strip(),
            'express': station_el.get('express', u"").strip()
        }

        if codes['esr']:
            codes['esr'] = esr_leading_zeros(codes['esr'])

        if codes['spk']:
            codes['spk'] = spk_leading_zeros(codes['spk'])

        title = station_el.get('title', u"").strip()
        settlement_id = station_el.get('settlement_id', u"").strip()
        region_id = station_el.get('region_id', u"").strip()
        station_type_id = station_el.get('station_type_id', u"").strip()
        majority_id = station_el.get('majority_id', u"").strip()

        station = None
        for system in systems:
            code = codes[system]
            if code:
                try:
                    station = Station.get_by_code(system, codes[system])
                    break
                except Station.DoesNotExist:
                    pass

        if station:
            log.info(u'Обновляем станцию %s %s -> %s ',
                     station.id, station.title, title)

            # Помечаем станцию, как использующую только поднаправления.
            station.use_direction = u"subdir"

            if title and station.title != title:
                station.title = title
                station.title_ru_locative = None
                station.title_ru_accusative = None
                station.title_ru_genitive = None

        elif title:
            log.info(u'Создаем станцию %s', title)

            station = Station()
            station.title = title
            station.t_type = TransportType.objects.get(code='train')

            station.majority_id = 4

            if majority_id:
                try:
                    if int(majority_id) not in station_majority_ids:
                        log.error(u"В базе нет важности с id %s", majority_id)
                        continue
                except ValueError:
                    log.error(u"В базе нет важности с id %s", majority_id)
                    continue

                station.majority_id = int(majority_id)

            # Создаем станцию использующую только поднаправления
            station.use_direction = u"subdir"
            station.save()

            log.info(u'Создали станцию %s %s %s', station.id, station.title, station.majority.title)

        if station:
            for system in systems:
                if codes[system]:
                    station.set_code(system, codes[system])

            if settlement_id:
                try:
                    if int(settlement_id) not in settlement_ids:
                        log.error(u"В базе нет города с кодом %s", settlement_id)
                        continue
                except ValueError:
                    log.error(u"В базе нет города с кодом %s", settlement_id)
                    continue
                station.settlement_id = settlement_id

            if region_id:
                try:
                    if int(region_id) not in region_ids:
                        log.error(u"В базе нет региона с кодом %s", region_id)
                        continue
                except ValueError:
                    log.error(u"В базе нет региона с кодом %s", region_id)
                    continue
                station.region_id = region_id

            if station_type_id:
                try:
                    if int(station_type_id) not in stype_ids:
                        log.error(u"В базе нет типа станции с кодом %s", station_type_id)
                        continue
                except ValueError:
                    log.error(u"В базе нет типа станции с кодом %s", station_type_id)
                    continue
                station.station_type_id = station_type_id

            station.save()

    # Выставляем использует направления
    Station.objects.filter(use_direction__isnull=True).update(use_direction="subdir")
